/***********************************************************************/
/*                                                                     */
/*                      ADOBE CONFIDENTIAL                             */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                         */
/*                                                                     */
/*  Copyright 2018 Adobe                                               */
/*  All Rights Reserved.                                               */
/*                                                                     */
/* NOTICE: All information contained herein is, and remains            */
/* the property of Adobe and its suppliers, if any. The intellectual   */
/* and technical concepts contained herein are proprietary to Adobe    */
/* and its suppliers and are protected by all applicable intellectual  */
/* property laws, including trade secret and copyright laws.           */
/* Dissemination of this information or reproduction of this material  */
/* is strictly forbidden unless prior written permission is obtained   */
/* from Adobe.                                                         */
/*                                                                     */
/***********************************************************************/

//////////////////////////////////////////////////////////////////////////////
//
// Validator for a single incoming script event
//
// For each ScriptEventTrigger object a ScriptEventTriggerValidator object
// is created. On an incoming event each validator check if the incoming
// event match its definition and provides a validation result.
//
function ScriptEventTriggerValidator(/*ScriptEventTrigger*/ inTrigger)
{
    var trigger = Utils.getParamValue(inTrigger, Utils.REQUIRED);
    var status = new TriggerHandlerStatus();
    var resultHandler = null;

    //////////////////////////////////////////////////////////////////////////////
    //
    // Link validator into validator chain
    //
    this.link = function(/*[Function*/ inResultHandler)
    {
        resultHandler = inResultHandler;
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Validate incoming event
    //
    this.validate = function(/*EventObject*/ inEventObj, /*ITriggerHandlerStatus*/ inStatus)
    {
        Utils.throwInvalid(resultHandler);

		if (Utils.isValidProperty(inStatus))
		{
			status = inStatus;
		}
		else
		{
			status = new TriggerHandlerStatus();
		}

        if (status.state == TriggerHandlerStatus.STATUS_UNDEFINED)
        {
            var currentEventObj = inEventObj;

			logInfo('Validate script event, type: "' + currentEventObj.type + '", trigger script event: "' + trigger.triggerDefinition.event + '"');

            trigger.triggerDefinition.validate(currentEventObj, function(/*Boolean*/ inMatched)
            {
                onValidateResult(trigger, currentEventObj, inMatched);
            });
        }
        else
        {
            resultHandler(inEventObj, status);
        }
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Evaluate validation result
    //
    function onValidateResult(/*WorkflowTrigger*/ inTrigger, /*EventObject*/ inEventObj, /*Boolean*/ inMatch)
    {
        if (inMatch)
        {
			var trigger = inTrigger;
			var eventObj = inEventObj;

			inTrigger.executeMatchActions(function(/*Any*/ inResult)
			{
				switch (trigger.triggerDefinition.match)
				{
					case ScriptEventTrigger.MATCH_POSITIVE:
						status.state = TriggerHandlerStatus.STATUS_POSITIVE;
						break;

					case ScriptEventTrigger.MATCH_NEGATIVE:
						status.state = TriggerHandlerStatus.STATUS_NEGATIVE;
						break;

					case ScriptEventTrigger.MATCH_IGNORE:
					default:
						status.state = TriggerHandlerStatus.STATUS_IGNORE;
				}

				status.trigger = trigger;

				logInfo('Set trigger validation state: ' + status.state);

				resultHandler(eventObj, status);
			});
        }
		else
		{
			logInfo('Set trigger validation state: ' + status.state);
			resultHandler(inEventObj, status);
		}
    }

}

//////////////////////////////////////////////////////////////////////////////
//
// A ScriptEventTriggerHandler tracks the user actions based on script events
// (Based on ITriggerHandler)
//
function ScriptEventTriggerHandler()
{
    var triggers = null;
    var callback = null;		// function(ITriggerHandlerStatus)
    var processingBlocked = false;
    var pendingEvents = [];
    var registeredEventTypes = [];
    var validatorChain = null;

	var pause = false;
	var initialized = false;

    //////////////////////////////////////////////////////////////////////////////
    //
    // Initialize handler
    //
    this.initialize = function(/*Array of WorkflowTrigger*/ inTrigger, /*Function*/ inCallback)
    {
		logInfo('Initialize trigger handler');

		callback = Utils.getParamValue(inCallback, Utils.REQUIRED);		// function(TriggerHandlerStatus)

		triggers = [];
        var steptriggers = Utils.getParamValue(inTrigger, Utils.REQUIRED);
		triggers = triggers.concat(steptriggers);

        initialized = true;
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Start handler
    // Install event handler and validator chain
    //
    this.start = function()
    {
		logInfo('Start trigger handler');

        Utils.throwInvalid(initialized);

		pause = false;
        var registeredEvents = [];
        registeredEventTypes = [];

		var wildcardTrigger = null;

        Utils.forEach(triggers, function(trigger)
        {
            if (trigger.type == ScriptEventTrigger.TYPE_SCRIPTEVENT)
            {
				if (trigger.triggerDefinition.event == ScriptEventTriggerHandler.WILDCARD)
				{
					wildcardTrigger = trigger;
				}
				else if (registeredEvents.indexOf(trigger.triggerDefinition.event) < 0)
                {
                    registeredEvents.push(trigger.triggerDefinition.event);
                }
            }
        });

		// If there is a wildcard trigger, then create trigger instances
		// for all remaining event types with the same definition as the
		// wildcard trigger (but with individual event type)
		//
		if (Utils.isValidProperty(wildcardTrigger))
		{
			var allTypes = ScriptEventTypes.get().getAll();

			Utils.forEach(allTypes, function(/*String*/ inEventType)
			{
				if (registeredEvents.indexOf(inEventType) < 0)
				{
					var wctrigger = wildcardTrigger.clone();
					wctrigger.triggerDefinition.event = inEventType;
					triggers.push(wctrigger);
				}
			});
		}

        if (Utils.isValidProperty(wildcardTrigger))
        {
			// register for all available event types
			//
            registerDefaultEventHandler();
        }
        else
		{
			// register only the trigger event types
			//
			registerEventHandler(registeredEvents);
		}

        setupChain(); 
        // now, ready and waiting for events...
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Stop handler
    // Uninstall event handler and remove validator chain
    //
    this.stop = function()
    {
		logInfo('Stop trigger handler');

        Utils.throwInvalid(initialized);

        // block any incoming events
        blockProcessing(true);

        // remove event handler
        unregisterEventHandler();

        // reset members
        processingBlocked = false;
        pendingEvents = [];
        registeredEventTypes = [];
        validatorChain = null;
		pause = false;
    }

	//////////////////////////////////////////////////////////////////////////////
	//
	// Pause handler
	// Neither listen to nor queue any incoming events
	//
	this.pause = function(/*Boolean*/ inPause)
	{
		pause = Utils.getParamValue(inPause, Utils.OPTIONAL, pause);
	}

	//////////////////////////////////////////////////////////////////////////////
    //
    // Register event handler for inEventType
    //
    function registerEventHandler(/*Array of Strings*/ inEventTypes)
    {
        Utils.forEach(inEventTypes, function(/*String*/ inEventType)
        {
            // We need to capture when registering an event listener to the class factory
            // to receive all events, because we are likely not the event target
            ScriptedWorkflowSupport.addEventListener(inEventType, processEvent, true);

			logInfo('Register script event handler, type "' + inEventType + '"');
            registeredEventTypes.push(inEventType);
        });
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Register event handler for all available event types except inRegisteredEvents
    //
    function registerDefaultEventHandler()
    {
		var allEventTypes = ScriptEventTypes.get().getAll();
		registerEventHandler(allEventTypes);
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Unregister all registered event handler
    //
    function unregisterEventHandler()
    {
        Utils.forEach(registeredEventTypes, function(/*String*/ inEventType)
        {
            ScriptedWorkflowSupport.removeEventListener(inEventType, processEvent, true);
			logInfo('Unregister script event handler, type "' + inEventType + '"');
        });

        registeredEventTypes = [];
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Event dispatcher
    //
    function processEvent(/*EventObject*/ inEventObj)
    {
		if (!pause)
		{
			var eventObj = filterEvents(inEventObj);

			if (Utils.isValidProperty(eventObj))
			{
                // Waiting for async operation below to complete, so queue events
				if (processingBlocked)
				{
					if (pendingEvents.length < ScriptEventTriggerHandler.MAX_PENDING_EVENTS)
					{
						logInfo('Incomming script event queued for processing, type: "' + eventObj.type + '"');
						pendingEvents.push(eventObj);
					}
					else
					{
						logWarn('Discard incomming script event due to queue overflow, type: "' + eventObj.type + '"');
					}
				}
				else if (Utils.isValidProperty(validatorChain))
				{
					blockProcessing(true);
					logInfo('Incomming script event start validation, type: "' + eventObj.type + '"');

					try
					{
						WorkflowEnvironment.current.event = eventObj;
						validatorChain.validate(eventObj);
					}
					catch (exc)
					{
						// enable processing of further events
						blockProcessing(false);

						logExc(exc);
					}
				}
			}
		}
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Un/block event processing
    //
    function blockProcessing(/*Boolean*/ inBlock, /*Boolean*/ inClear)
    {
		if (Utils.getParamValue(inClear, Utils.OPTIONAL, false))
		{
			pendingEvents = [];
		}

        if (inBlock != processingBlocked)
        {
            processingBlocked = inBlock;

            if (!processingBlocked && pendingEvents.length > 0)
            {
                var evObj = pendingEvents.shift();
				logInfo('Process queued script event, type: "' + evObj.type + '"');
                processEvent(evObj);
            }
        }
    }

	//////////////////////////////////////////////////////////////////////////////
	//
	// Filter reserved events
	//
    function filterEvents(/*EventObject*/ inEventObj)
	{
		var ret = inEventObj;

		if (Utils.isValidProperty(inEventObj))
		{
			if (ScriptEventTypes.get().getAll().length == 0)
			{
				// no script event types available, just filter a few event types
				//
				switch (inEventObj.type)
				{
					case 'UIControl.ValueChanged':
					{
                        // TODO:
                        // This needs to be adjusted to run on all platforms
                        // Since we control, we can give the end node a unique and identical name on all
						if (inEventObj.ctrlPath == '-1/0/UI_TouchPopupContainer,0/1/OS_ViewContainer,0/1/UI_SubView,0/1/UI_SubView,0/1/UI_ControlView,2/1/UI_TextButton')
						{
							ret = null;
						}
					}
					break;
				}
			}
			else
			{
				// filter all event types which are not in the known event types list
				//
				if (!ScriptEventTypes.get().exists(inEventObj.type))
				{
					ret = null;
				}
			}
		}

		return ret;
	}

    //////////////////////////////////////////////////////////////////////////////
    //
    // Result from validation chain
    //
    function onValidationResult(/*EventObject*/ inEventObj, /*ITriggerHandlerStatus*/ inStatus)
    {
        var status = inStatus;

		logInfo('Process trigger validator chain result, state: ' + status.state);

        if (status.state ==  TriggerHandlerStatus.STATUS_POSITIVE)
        {
			logInfo('Positive trigger validator chain result');

			// clear queue
			blockProcessing(false, true);

            // Finished with positive trigger match
            // Execute positive trigger actions and return
            //
			if (Utils.isValidProperty(status.trigger))
			{
				status.trigger.executePositiveActions(function(/*Any*/ inResult)
				{
					status.result = inResult;
					callback(status);
				});
			}
			else
			{
				callback(status);
			}
        }
        else if (status.state ==  TriggerHandlerStatus.STATUS_NEGATIVE)
        {
			logInfo('Negative trigger validator chain result');

			// clear queue
			blockProcessing(false, true);

			// Finished with negative trigger match
            // Execute negative trigger actions and return
            //
			if (Utils.isValidProperty(status.trigger))
			{
				status.trigger.executeNegativeActions(function(/*Any*/ inResult)
				{
					status.result = inResult;
					callback(status);
				});
			}
			else
			{
				callback(status);
			}
        }
        else
        {
			status.state = TriggerHandlerStatus.STATUS_IGNORE;
			logInfo('No trigger match, continue listening for incoming script events');

			// NO trigger match
            // Enable processing of the next event
            //
            blockProcessing(false);
        }
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Setup validator chain
    //
    //  Connect all trigger, so that the output of the first is the input of the second
    //  As such, we have a simple chain of logical OR operation.
    //
    function setupChain()
    {
		logInfo('Setup trigger handler validator chain, trigger: ' + triggers.length);

        var validators = [];

        Utils.forEach(triggers, function(trigger)
        {
            if (trigger.type == ScriptEventTrigger.TYPE_SCRIPTEVENT &&
				trigger.triggerDefinition.event != ScriptEventTriggerHandler.WILDCARD)
            {
                var validator = new ScriptEventTriggerValidator(trigger);
                validators.push(validator);
            }
        });

        if (validators.length > 0)
        {
            validatorChain = validators.shift();
            var tail = validatorChain;

            Utils.forEach(validators, function (validator)
            {
                tail.link(validator.validate);
                tail = validator;
            });

            tail.link(onValidationResult);
        }
    }
}

ScriptEventTriggerHandler.MAX_PENDING_EVENTS = 20;
ScriptEventTriggerHandler.WILDCARD = '*';

// Register handler
//
TriggerHandlerFactory.addFactory(ScriptEventTrigger.TYPE_SCRIPTEVENT, function(){return new ScriptEventTriggerHandler();});
