/***********************************************************************/
/*                                                                     */
/*                      ADOBE CONFIDENTIAL                             */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                         */
/*                                                                     */
/*  Copyright 2018 Adobe                                               */
/*  All Rights Reserved.                                               */
/*                                                                     */
/* NOTICE: All information contained herein is, and remains            */
/* the property of Adobe and its suppliers, if any. The intellectual   */
/* and technical concepts contained herein are proprietary to Adobe    */
/* and its suppliers and are protected by all applicable intellectual  */
/* property laws, including trade secret and copyright laws.           */
/* Dissemination of this information or reproduction of this material  */
/* is strictly forbidden unless prior written permission is obtained   */
/* from Adobe.                                                         */
/*                                                                     */
/***********************************************************************/

//////////////////////////////////////////////////////////////////////////////
//
// The ContentProviderManager is a central point to retrieve content for a
// Workflow. The type used IContentProvider is defined by the type of
// the Workflow
//
// Typical element of a guided tour: 
//
//		"content" :
//		{
//			"title" :
//			{
//				"id" : "$$$/ProjectEdit/PlaybackStart1"
//				"mimetype" : "text/plain"
//			},
//			"text" :
//			{
//				"id" : "$$$/ProjectEdit/PlaybackStartDesc1"
//				"mimetype" : "text/plain"
//			}
//		}
//
//	"title" would be one content part referenced in getContent(inPart)
//
//	For example, when a coachmark is to be displayed, the actual content is retrieved
//	ContentProviderManager.getContent(inPart, inCallback)


function ContentProviderManager()
{
	var allProvider = {}; // [scope][type]
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Retrieve content for WorkflowContentPart
	//
	this.getContent = function(/*WorkflowContentPart*/ inPart, /*Function*/ inCallback)
	{
		if (!allProvider.hasOwnProperty(inPart.getWorkflowID()))
		{
			createProvider(inPart.getWorkflowType(), inPart.getWorkflowID());
		}

		var ret =	(allProvider.hasOwnProperty(inPart.getWorkflowID()) &&
					allProvider[inPart.getWorkflowID()].hasOwnProperty(inPart.getWorkflowType()));

		if (ret)
		{
			allProvider[inPart.getWorkflowID()][inPart.getWorkflowType()].getContent(inPart, inCallback);
		}
		else
		{
			logWarn('Content provider not available, content part: "' + inPart.contentID + '" , workflow id: "' + inPart.getWorkflowID() + '"');
			inCallback('');
		}

		return ret;
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	//
	function createProvider(/*String*/ inType, /*String*/ inScope)
	{
		var scope = Utils.getParamValue(inScope, Utils.OPTIONAL, 'global');
		var provider = null;

		try
		{
			provider = ContentProviderFactory.create(inType);
		}
		catch(exc)
		{
			logExc(exc);
		}

		if (Utils.isValidProperty(provider))
		{
			if (!Utils.isValidProperty(allProvider[scope]))
			{
				allProvider[scope] = {};
			}

			allProvider[scope][inType] = provider;
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
//
//
ContentProviderManager.instance = null;
ContentProviderManager.get = function()
{
	if (!Utils.isValidProperty(ContentProviderManager.instance))
	{
		ContentProviderManager.instance = new ContentProviderManager();
	}

	return ContentProviderManager.instance;
}

//////////////////////////////////////////////////////////////////////////////
//
// Retrieve content for WorkflowContentPart
//
ContentProviderManager.getContent = function(/*WorkflowContentPart*/ inPart, /*Function(result)*/ inCallback)
{
	var part = Utils.getParamValue(inPart, Utils.REQUIRED);
	return ContentProviderManager.get().getContent(part, inCallback);
}