/***********************************************************************/
/*                                                                     */
/*                      ADOBE CONFIDENTIAL                             */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                         */
/*                                                                     */
/*  Copyright 2018 Adobe                                               */
/*  All Rights Reserved.                                               */
/*                                                                     */
/* NOTICE: All information contained herein is, and remains            */
/* the property of Adobe and its suppliers, if any. The intellectual   */
/* and technical concepts contained herein are proprietary to Adobe    */
/* and its suppliers and are protected by all applicable intellectual  */
/* property laws, including trade secret and copyright laws.           */
/* Dissemination of this information or reproduction of this material  */
/* is strictly forbidden unless prior written permission is obtained   */
/* from Adobe.                                                         */
/*                                                                     */
/***********************************************************************/

//////////////////////////////////////////////////////////////////////////////
//
// Logger
// Logs either into the debugger console (if connected to such an debugger)
// and/or to the dva logging system
//
function Logger(/*Boolean*/ inConsole, /*Boolean*/ inFile)
{
	var logConsole	= Utils.getParamValue(inConsole, Utils.OPTIONAL, false);
	var logFile		= Utils.getParamValue(inFile, Utils.OPTIONAL, false);

	this.log = function(/*Number*/ inLevel, /*String*/ inPrefix, /*String*/ inMessage)
	{
		var enabled = DebugDatabase.get('dvascriptedworkflow.DisableScriptLogging');

		if (enabled != 'true')
		{
			var level  = Utils.getParamValue(inLevel, Utils.OPTIONAL, 5);
			var prefix = Utils.getParamValue(inPrefix, Utils.OPTIONAL, '');
			var msg    = Utils.getParamValue(inMessage, Utils.OPTIONAL, '');

			var message = (prefix.length > 0 ? (prefix + ': ') : '') + msg;

			// Android doesn't provide the console object
			if (logConsole && ScriptedWorkflowSupport.instance.appPlatform != "Android")
			{
				// log in debugger console
				console.log(message);
			}

			if (logFile)
			{
				// log into application log system
				ScriptedWorkflowSupport.instance.log(level, message);
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
//
// Logger instance
//
Logger.instance = null;
Logger.get = function()
{
	if (!Utils.isValidProperty(Logger.instance))
	{
		// TODO: generic params
		Logger.instance = new Logger(true, true);
	}

	return Logger.instance;
}

//////////////////////////////////////////////////////////////////////////////
//
// Log Level
//
Logger.LEVEL_INFO	= 1;
Logger.LEVEL_WARN	= 2;
Logger.LEVEL_ERROR	= 6;
Logger.LEVEL_FATAL	= 7;

function _where(/*Error*/ inError)
{
	// platform dependent constants
	var kFilepathBeginning = ScriptedWorkflowSupport.instance.appPlatform == 'Windows' ? ':\\' : '/';
	var kFilepathDelimitter = ScriptedWorkflowSupport.instance.appPlatform == 'Windows' ? '\\' : '/';
	var kLineDelimitter = ScriptedWorkflowSupport.instance.appPlatform == 'iOS' ? '@' : ' ';
	var kFirstLine = ScriptedWorkflowSupport.instance.appPlatform == 'iOS' ? 1 : 2;

	// check if inStr is a path string
	function isPath(/*String*/ inStr)
	{
		return (inStr.indexOf('(') == 0 || inStr.indexOf(kFilepathBeginning) >= 0);
	}

	// find first line containing a useful function name in call stack
	function firstFunctionLine(/*Array of String*/ inStack)
	{
		var ret = -1;
		var defIndex = inStack.length > kFirstLine ? kFirstLine : inStack.length - 1;	// kFirstLine is the index of the first line which might be of interesst
		var lineIndex = defIndex;

		// iterate over entries of call stack
		//
		while (ret < 0 && lineIndex < inStack.length)
		{
			var line      = inStack[lineIndex].trim();
			var lineParts = line.split(kLineDelimitter);

			if (lineParts.length)
			{
				var part = lineParts[0];

				// ignore if line begins with 'at'
				//
				if (part.indexOf('at') >= 0 && lineParts.length > 1)
				{
					part = lineParts[1];
				}

				// ignore if function is a Utils function
				// ignore if string is a path string
				//
				if (part.indexOf('Object.Utils') != 0 && !isPath(part))
				{
					ret = lineIndex;
				}
			}

			lineIndex++;
		}

		if (ret < 0)
		{
			ret = defIndex;
		}

		return ret;
	}

	var ret = '';
	var stack = inError.stack.split('\n');

	if (stack.length)
	{
		/* Example call stack (macos)
		 "Error
		 at logInfo (/Users/.../01_Logging.js:224:21)
		 at onDataLoaded (/Users/.../21_DataRequestManager.js:107:3)"
		 */

		// find entry in callstack to extract function- and file- name
		var lineIndex = firstFunctionLine(stack);

		var fctName   = '';
		var fileName  = '';
		var line      = stack[lineIndex].trim();
		var lineParts = line.split(kLineDelimitter);

		if (lineParts.length)
		{
			var index = 0;

			// ignore leading 'at' in line
			//
			if (lineParts[index].indexOf('at') >= 0)
			{
				index++;
			}

			if (index < lineParts.length)
			{
				var linePart = lineParts[index].trim();

				if (!isPath(linePart))
				{
					// if string is not a path then take as the function name
					fctName = linePart;
					index++;
				}

				if (index < lineParts.length)
				{
					linePart = lineParts[index].trim();

					if (isPath(linePart))
					{
						// if linePart contains a path string then take the last available part
						// and extract the filename
						//
						linePart = lineParts[lineParts.length - 1].trim();

						var si = linePart.lastIndexOf(kFilepathDelimitter);
						si     = si >= 0 ? si + 1 : 0;

						fileName = linePart.substring(si);

						if (fileName.lastIndexOf(')') == fileName.length - 1)
						{
							fileName = fileName.substring(0, fileName.length - 1);
						}
					}
				}
			}

			ret = fctName;
			ret += fctName.length ? ' ' : '';
			ret += fileName;
		}
	}

	return ret;
}

//////////////////////////////////////////////////////////////////////////////
//
// Log INFO
//
function logInfo(/*String*/ inMessage)
{
	var msg = inMessage;
	var where = _where(new Error());
	if (where.length)
	{
		msg = '[' + where + '] ' + msg;
	}

	Logger.get().log(Logger.LEVEL_INFO, 'INFO', msg);
}

//////////////////////////////////////////////////////////////////////////////
//
// Log WARNING
//
function logWarn(/*String*/ inMessage)
{
	var msg = inMessage;
	var where = _where(new Error());
	if (where.length)
	{
		msg = '[' + where + '] ' + msg;
	}

	Logger.get().log(Logger.LEVEL_WARN, 'WARNING', msg);
}

//////////////////////////////////////////////////////////////////////////////
//
// Log ERROR
//
function logError(/*String*/ inMessage)
{
	var msg = inMessage;
	var where = _where(new Error());
	if (where.length)
	{
		msg = '[' + where + '] ' + msg;
	}

	Logger.get().log(Logger.LEVEL_ERROR, 'ERROR', msg);
}

//////////////////////////////////////////////////////////////////////////////
//
// Log FATAL
//
function logFatal(/*String*/ inMessage)
{
	var msg = inMessage;
	var where = _where(new Error());
	if (where.length)
	{
		msg = '[' + where + '] ' + msg;
	}

	Logger.get().log(Logger.LEVEL_FATAL, 'FATAL', msg);
}

//////////////////////////////////////////////////////////////////////////////
//
// Log EXCEPTION
//
function logExc(/*Object*/ inException)
{
	var msg = '';

	if (Utils.isValidProperty(inException.name))
	{
		msg = inException.name + ': ';
	}

	if (Utils.isValidProperty(inException.message))
	{
		msg += inException.message;
	}

	if (msg.length == 0)
	{
		msg = inException.toString();
	}

	if (Utils.isValidProperty(inException.fileName))
	{
		msg += '\n' + inException.fileName;

		if (Utils.isValidProperty(inException.lineNumber))
		{
			msg += ', line ' + inException.lineNumber;
		}
	}

	if (Utils.isValidProperty(inException.stack))
	{
		msg += '\n' + inException.stack;
	}

	Logger.get().log(Logger.LEVEL_FATAL, 'EXCEPTION', msg);
}
