/*******************************************************************/
/*                                                                 */
/*                      ADOBE CONFIDENTIAL                         */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                     */
/*                                                                 */
/* Copyright 2004 Adobe Systems Incorporated                       */
/* All Rights Reserved.                                            */
/*                                                                 */
/* NOTICE:  All information contained herein is, and remains the   */
/* property of Adobe Systems Incorporated and its suppliers, if    */
/* any.  The intellectual and technical concepts contained         */
/* herein are proprietary to Adobe Systems Incorporated and its    */
/* suppliers and may be covered by U.S. and Foreign Patents,       */
/* patents in process, and are protected by trade secret or        */
/* copyright law.  Dissemination of this information or            */
/* reproduction of this material is strictly forbidden unless      */
/* prior written permission is obtained from Adobe Systems         */
/* Incorporated.                                                   */
/*                                                                 */
/*******************************************************************/

#include "ScreenQuadInclude.fx"
#include "VideoSourceTextureInclude.fx"
#include "ColorSpaceConversionsInclude.fx"

/*
**	FastColorCorrector
*/
float Magnitude <
	string UIWidget
	 = "slider";
> = 1.0f;

float MaxBalanceX <
	string UIWidget
	 = "slider";
> = 1.0f;

float MaxBalanceY <
	string UIWidget
	 = "slider";
> = 1.0f;

float SinHue <
	string UIWidget
	 = "slider";
> = 1.0f;

float CosHue <
	string UIWidget
	 = "slider";
> = 1.0f;

float Saturation <
	string UIWidget
	 = "slider";
> = 1.0f;

float InputRange <
	string UIWidget
	 = "slider";
> = 1.0f;

float OutputRange <
	string UIWidget
	 = "slider";
> = 1.0f;

float InputBlackLevel <
	string UIWidget
	 = "slider";
> = 1.0f;

float InputWhiteLevel <
	string UIWidget
	 = "slider";
> = 1.0f;

float InputGrayLevel <
	string UIWidget
	 = "slider";
> = 1.0f;

float OutputBlackLevel <
	string UIWidget
	 = "slider";
> = 1.0f;

float OutputWhiteLevel <
	string UIWidget
	 = "slider";
> = 1.0f;

/////////////////////////////////////////////////////////////////
// Functions
/////////////////////////////////////////////////////////////////

/*
** Calculate Composite View BGRA With Gamma
*/

float4 CalculateColorFastCompositeBGRAWithGamma(float4 inColorPixel)
{
	float4 oCol;

    float4 videoPixel = inColorPixel;
    float alpha = inColorPixel.a;
 
	// Convert to straight RGBA
	if (alpha > 0.0f)   
	{
	   videoPixel /= alpha;
	}

	//	straight RGB->YCbCr normalized conversion
	float4 YCbCrNormalized = RGBAToYCbCrA601Normalized(videoPixel);
	
	float y = YCbCrNormalized.r; 
	float u = YCbCrNormalized.g; 
	float v = YCbCrNormalized.b;

	//	Use Photoshop-like levels calculation.
	y = OutputBlackLevel + OutputRange * pow( max(0.0f,(y - InputBlackLevel)) / InputRange,InputGrayLevel);

	//	Rotate (u, v) by hue angle,
	//	by multiplying against sine and cosine of
	//	hue angle degrees converted to radians.
	float newU = u * CosHue - v * SinHue;
	float newV = v * CosHue + u * SinHue;

	//	interpolate towards (maxBalanceX, maxBalanceY) by balance magnitude
	newU = lerp(newU, MaxBalanceX, Magnitude);
	newV = lerp(newV, MaxBalanceY, Magnitude);

	//	Apply saturation.
	newU *= Saturation;
	newV *= Saturation;

	u = newU;
	v = newV;

	//	YCbCr->RGB conversion
	float4 RGBA;

	RGBA.r = y;
	RGBA.g = u;
	RGBA.b = v;
	RGBA.a = alpha;

	oCol = YCbCrA601NormalizedToPremultipliedRGBA(RGBA);

	// [NOTE] rprasad 08.16.05
	// Using the intrinsic function 'all' instead of 'if (InputRange == 0.0)'
	// to avoid a problem with Nvidia cards (Bug 1227189)
	oCol.rgb *= all(InputRange);

	return oCol;
}

/*
** Calculate Composite View BGRA Without Gamma
*/

float4 CalculateColorFastCompositeBGRAWithoutGamma(float4 inColorPixel)
{
	float4 oCol;

    float4 videoPixel = inColorPixel;
    float alpha = inColorPixel.a;
 
	// Convert to straight RGBA
	if (alpha > 0.0f)   
	{
	   videoPixel /= alpha;
	}

	//	straight RGB->YCbCr normalized conversion
	float4 YCbCrNormalized = RGBAToYCbCrA601Normalized(videoPixel);
	
	float y = YCbCrNormalized.r; 
	float u = YCbCrNormalized.g; 
	float v = YCbCrNormalized.b;

	//	Use Photoshop-like levels calculation.
	y = ((y - InputBlackLevel) / InputRange) * OutputRange + OutputBlackLevel;

	//	Rotate (u, v) by hue angle,
	//	by multiplying against sine and cosine of
	//	hue angle degrees converted to radians.
	float newU = u * CosHue - v * SinHue;
	float newV = v * CosHue + u * SinHue;

	//	interpolate towards (maxBalanceX, maxBalanceY) by balance magnitude
	newU = lerp(newU, MaxBalanceX, Magnitude);
	newV = lerp(newV, MaxBalanceY, Magnitude);

	//	Apply saturation.
	newU *= Saturation;
	newV *= Saturation;

	u = newU;
	v = newV;

	//	YCbCr->RGB conversion
	float4 RGBA;

	RGBA.r = y;
	RGBA.g = u;
	RGBA.b = v;
	RGBA.a = alpha;

	oCol = YCbCrA601NormalizedToPremultipliedRGBA(RGBA);

	return oCol;
}

/*
** Calculate Luma View BGRA With Gamma
*/
float4 CalculateColorFastLumaBGRAWithGamma(float4 inColorPixel)
{
	float4 oCol;

    float4 videoPixel = inColorPixel;
    float alpha = inColorPixel.a;
 
	// Convert to straight RGBA
	if (alpha > 0.0f)   
	{
	   videoPixel /= alpha;
	}

	//	straight RGB->YCbCr normalized conversion
	float4 YCbCrNormalized = RGBAToYCbCrA601Normalized(videoPixel);

	float y = YCbCrNormalized.r; 

	//	Use Photoshop-like levels calculation.
	y = OutputBlackLevel + OutputRange * pow( max(0.0f,(y - InputBlackLevel)) / InputRange,InputGrayLevel);

	//	YCbCr->RGB conversion
	float4 RGBA;

	RGBA.r = y;
	RGBA.g = 0.0;
	RGBA.b = 0.0;
	RGBA.a = alpha;
		
	oCol = YCbCrA601NormalizedToPremultipliedRGBA(RGBA);
	
	// [NOTE] rprasad 08.16.05
	// Using the intrinsic function 'all' instead of 'if (InputRange == 0.0)'
	// to avoid a problem with Nvidia cards (Bug 1227189)
	oCol.rgb *= all(InputRange);
		
	return oCol;
}

/*
** Calculate Luma View BGRA Without Gamma
*/
float4 CalculateColorFastLumaBGRAWithoutGamma(float4 inColorPixel)
{
	float4 oCol; // Output Pixel

    float4 videoPixel = inColorPixel;
    float alpha = inColorPixel.a;
 
	// Convert to straight RGBA
	if (alpha > 0.0f)   
	{
	   videoPixel /= alpha;
	}

	//	straight RGB->YCbCr normalized conversion
	float4 YCbCrNormalized = RGBAToYCbCrA601Normalized(videoPixel);

	float y = YCbCrNormalized.r; 
	float u = YCbCrNormalized.g; 
	float v = YCbCrNormalized.b;

	//	Use Photoshop-like levels calculation.
	y = ((y - InputBlackLevel) / InputRange) * OutputRange + OutputBlackLevel;

	//	YUV->RGB conversion
	float4 RGBA;

	RGBA.r = y;
	RGBA.g = 0.0;
	RGBA.b = 0.0;
	RGBA.a = alpha;

	oCol = YCbCrA601NormalizedToPremultipliedRGBA(RGBA);

	return oCol;
}

/*
** Calculate Mask View BGRA
*/
float4 CalculateColorFastMaskBGRA(float inColorPixelAlpha)
{
	float4 oCol = 1.0f; // Initialized to white
	// Output pixel is premultiplied with alpha
	oCol *= inColorPixelAlpha;
	return oCol;
}

/////////////////////////////////////////////////////////////////
// Pixel Shaders
/////////////////////////////////////////////////////////////////
/*
/** Pixel Shader - Composite View Mode with Gamma
*/
float4 ColorFastCompositeBGRAWithGamma_PS(
	QuadVertexOutput inVertexInfo) : COLOR
{
	float4 texturePixel = tex2D(ImageSampler, inVertexInfo.UV);
	return CalculateColorFastCompositeBGRAWithGamma(texturePixel);	
}

/*
/** Pixel Shader - Composite View Mode without Gamma
*/
float4 ColorFastCompositeBGRAWithoutGamma_PS(
	QuadVertexOutput inVertexInfo) : COLOR
{
	float4 texturePixel = tex2D(ImageSampler, inVertexInfo.UV);
	return CalculateColorFastCompositeBGRAWithoutGamma(texturePixel);	
}

/*
/** Pixel Shader - Luma View Mode with Gamma
*/
float4 ColorFastLumaBGRAWithGamma_PS(
	QuadVertexOutput inVertexInfo) : COLOR
{
	float4 texturePixel = tex2D(ImageSampler, inVertexInfo.UV);
	return CalculateColorFastLumaBGRAWithGamma(texturePixel);	
}

/*
/** Pixel Shader - Luma View Mode without Gamma 
*/
float4 ColorFastLumaBGRAWithoutGamma_PS(
	QuadVertexOutput inVertexInfo) : COLOR
{
	float4 texturePixel = tex2D(ImageSampler, inVertexInfo.UV);
	return CalculateColorFastLumaBGRAWithoutGamma(texturePixel);
}

/*
/** Pixel Shader - Mask View Mode with/without Gamma 
*/
float4 ColorFastMaskBGRA_PS(
	QuadVertexOutput inVertexInfo) : COLOR
{
	float4 texturePixel = tex2D(ImageSampler, inVertexInfo.UV);
	return CalculateColorFastMaskBGRA(texturePixel.a);
}

/*
**	Straight RGBA copy
*/
float4 ColorFastOriginalBGRA_PS(
	QuadVertexOutput inVertexInfo) : COLOR
{
	float4 texturePixel = tex2D(ImageSampler, inVertexInfo.UV);
	return texturePixel;	
}

/////////////////////////////////////////////////////////////////
//	Techniques
/////////////////////////////////////////////////////////////////

/*
**	Composite view mode with gamma
*/
technique ColorFastCompositeBGRAWithGamma
{
	pass P0
	{
		// shaders
		VertexShader = compile vs_1_1 ScreenQuadVS();
		PixelShader  = compile ps_2_0 ColorFastCompositeBGRAWithGamma_PS();
	}  
	pass P1
	{
		// shaders
		PixelShader  = compile ps_2_0 ColorFastOriginalBGRA_PS();
	}  
    
}

/*
** Composite view mode without gamma
*/
technique ColorFastCompositeBGRAWithoutGamma
{
	pass P0
	{
		// shaders
		VertexShader = compile vs_1_1 ScreenQuadVS();
		PixelShader  = compile ps_2_0 ColorFastCompositeBGRAWithoutGamma_PS();
	}
	pass P1
	{
		// shaders
		PixelShader  = compile ps_2_0 ColorFastOriginalBGRA_PS();
	}        
}

/*
** Luma view mode with gamma 
*/
technique ColorFastLumaBGRAWithGamma
{
	pass P0
	{
		// shaders
		VertexShader = compile vs_1_1 ScreenQuadVS();
		PixelShader  = compile ps_2_0 ColorFastLumaBGRAWithGamma_PS();
	}  
	pass P1
	{
		// shaders
		PixelShader  = compile ps_2_0 ColorFastOriginalBGRA_PS();
	}  

}

/*
** Luma view mode without gamma 
*/
technique ColorFastLumaBGRAWithoutGamma
{
	pass P0
	{
		// shaders
		VertexShader = compile vs_1_1 ScreenQuadVS();
		PixelShader  = compile ps_2_0 ColorFastLumaBGRAWithoutGamma_PS();
	}  
	pass P1
	{
		// shaders
		PixelShader  = compile ps_2_0 ColorFastOriginalBGRA_PS();
	}  
}

/*
** Mask view mode with/without gamma
*/
technique ColorFastMaskBGRA
{
	pass P0
	{
		// shaders
		VertexShader = compile vs_1_1 ScreenQuadVS();
		PixelShader  = compile ps_2_0 ColorFastMaskBGRA_PS();
	}  
	pass P1
	{
		// shaders
		PixelShader  = compile ps_2_0 ColorFastOriginalBGRA_PS();
	}  
}