/*******************************************************************/
/*                                                                 */
/*                      ADOBE CONFIDENTIAL                         */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                     */
/*                                                                 */
/* Copyright 2002 Adobe Systems Incorporated                       */
/* All Rights Reserved.                                            */
/*                                                                 */
/* NOTICE:  All information contained herein is, and remains the   */
/* property of Adobe Systems Incorporated and its suppliers, if    */
/* any.  The intellectual and technical concepts contained         */
/* herein are proprietary to Adobe Systems Incorporated and its    */
/* suppliers and may be covered by U.S. and Foreign Patents,       */
/* patents in process, and are protected by trade secret or        */
/* copyright law.  Dissemination of this information or            */
/* reproduction of this material is strictly forbidden unless      */
/* prior written permission is obtained from Adobe Systems         */
/* Incorporated.                                                   */
/*                                                                 */
/*******************************************************************/

/*
**-----------------------------------------------------------------------------
** Effect File Variables - HLSLRefract.fx
**-----------------------------------------------------------------------------
*/

uniform float4x4	gWorldViewProj : WorldViewProjection; // This matrix will be loaded by the application
uniform float4x4	gWorldViewInverse;
uniform float4x4	gWorld;
uniform float4		gEyePosition;
uniform float		gBumpAmount;
uniform float		gRefractiveIndex;
uniform float		gRippleParameter;
uniform float		gDepth;
texture				gVideoTexture;
texture				gBumpTexture;
const float			kModelDim = 4.0f;

 


/*
**-----------------------------------------
**		Sampler States
**-----------------------------------------
*/
//incoming video texture
sampler SamplerVideo = sampler_state
{
    Texture   = (gVideoTexture);
    MipFilter = LINEAR;
    MinFilter = LINEAR;
    MagFilter = LINEAR;
    BorderColor = 0x00000000;
    AddressU = BORDER;
    AddressV = BORDER;
    
};

sampler SamplerBump = sampler_state
{
    Texture   = (gBumpTexture);
    MipFilter = LINEAR;
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};

/*
**-----------------------------------------------------------------------------
** Vertex Definitions
**-----------------------------------------------------------------------------
** APP_OUTPUT is the structure in which we receive the vertices from the application
*/
struct APP_OUTPUT
{
    float3 mPosition	: POSITION;
	float3 mNormal		: NORMAL;
	float2 mTexCoord	: TEXCOORD0;
	
};

/* 
** Pixel Shader structure declaration
*/

struct REF_VS_OUTPUT
{
    float4 mPosition			:POSITION;
	float2 mTexcoord			:TEXCOORD0;
	float3 mPositionT			:TEXCOORD1;
	float3 mIncidentVectorT		:TEXCOORD2;
	float3 mBacktexS			:TEXCOORD3;
	float3 mBacktexT			:TEXCOORD4;
};

struct REF_PS_OUTPUT
{
	float4 mColor : COLOR;
};

struct TEMP_OUT
{
	float4 mPosition : POSITION;
	float3 mNormal	 : NORMAL0;
	float3 mTangent;
	float3 mBinormal;
};

/*
**------------------------------------------------
**		Ripple effect
**------------------------------------------------
*/
TEMP_OUT MakeRipple( float3 position, uniform float param )
{
	TEMP_OUT returnVertex;
	float sinVal, dist = 0;
	
	dist = distance ( position, float3( 0.0, 0.0, 0.1) );

	sinVal = sin((dist+param)*25);
	
    returnVertex.mPosition = float4(position.x,
								position.y, 
								position.z,
								1.0f );
	
	returnVertex.mTangent = normalize( float3( 1.0, 0, param*0.10*sinVal*position.x/dist) );
	returnVertex.mBinormal= normalize( float3( 0, 1.0, param*0.10*sinVal*position.y/dist) );
	returnVertex.mNormal = cross( returnVertex.mTangent, returnVertex.mBinormal );

	return returnVertex;
}


/*
**------------------------------------------------
**		my_refract function 
**------------------------------------------------
*/

float3 my_refract( float3 I, float3 N, float eta )
{
	return I + eta*(-N-I);
}


/*
**------------------------------------------------
**		Glass vertex shader
**------------------------------------------------
*/
REF_VS_OUTPUT refract_vs( APP_OUTPUT IN )
{
    REF_VS_OUTPUT OUT;

	TEMP_OUT tempVtx = MakeRipple( IN.mPosition, gRippleParameter );
	
	// transform vertex position into homogenous clip-space
    OUT.mPosition = mul(gWorldViewProj, (tempVtx.mPosition) );
	
	// copy texture coordinates
	OUT.mTexcoord.xy = IN.mTexCoord.xy;
    
	float3x3 stspacemat = float3x3(tempVtx.mTangent,tempVtx.mBinormal,tempVtx.mNormal);
	
	//compute positon and normal in world space
    float3 positionW = mul(gWorld,tempVtx.mPosition);
        
    //compute the incident vector(in world space)- and transform to texture space
    float3 incidentVectorW = positionW - gEyePosition;
    incidentVectorW = normalize(incidentVectorW);
    OUT.mIncidentVectorT=mul(stspacemat,mul(gWorldViewInverse,incidentVectorW));
    OUT.mIncidentVectorT = normalize( OUT.mIncidentVectorT );
    
    //carrying fwd the positionW in texturespace
    OUT.mPositionT =mul(stspacemat,tempVtx.mPosition);
    
    OUT.mBacktexS = mul(stspacemat,float3(1.0f,0.0,0.0));
    OUT.mBacktexT = mul(stspacemat,float3(0.0,1.0f,0.0));
    
    return OUT;
}


REF_VS_OUTPUT refract_vs_1_3( APP_OUTPUT IN )
{
    REF_VS_OUTPUT OUT;

	TEMP_OUT tempVtx = MakeRipple( IN.mPosition, gRippleParameter );
	
	// transform vertex position into homogenous clip-space
    OUT.mPosition = mul(gWorldViewProj, (tempVtx.mPosition) );
	    
	float3x3 stspacemat = float3x3(tempVtx.mTangent,tempVtx.mBinormal,tempVtx.mNormal);
	
	//compute positon and normal in world space
    float3 positionW = mul(gWorld,tempVtx.mPosition);
        
    //compute the incident vector(in world space)- and transform to texture space
    float3 incidentVectorW = positionW - gEyePosition;
    incidentVectorW = normalize(incidentVectorW);
    OUT.mIncidentVectorT=mul(stspacemat,mul(gWorldViewInverse,incidentVectorW));
    OUT.mIncidentVectorT = normalize( OUT.mIncidentVectorT );
    
    //carrying fwd the positionW in texturespace
    OUT.mPositionT =mul(stspacemat,tempVtx.mPosition);
    
    OUT.mBacktexS = mul(stspacemat,float3(1.0f,0.0,0.0));
    OUT.mBacktexT = mul(stspacemat,float3(0.0,1.0f,0.0));
    
    float3 refractedVectorT =my_refract(OUT.mIncidentVectorT, float3(0.0, 0.0, 1.0), gRefractiveIndex );

	float3 pointOnBackTextureT;
	pointOnBackTextureT = gDepth*refractedVectorT + OUT.mPositionT;

	OUT.mTexcoord.x = dot(OUT.mBacktexS, pointOnBackTextureT);
	OUT.mTexcoord.y = dot(OUT.mBacktexT, pointOnBackTextureT);
    
    return OUT;
}


/*
**------------------------------------------------
**		Glass pixel shader 2.0
**------------------------------------------------
*/
REF_PS_OUTPUT refract_ps_2_0( REF_VS_OUTPUT IN )
{
REF_PS_OUTPUT OUT;
	
	//compute local normals - in texture (s&t) space
	float3 localNormalT;
	localNormalT = tex2D( SamplerBump, IN.mTexcoord ).xyz;
	localNormalT = 2*(localNormalT - 0.5);
	
	localNormalT=lerp(float3(0.0f,0.0f,1.0f),localNormalT,gBumpAmount);
	
	//find the refracted vector in TextureSpace
	float3 refractedVectorT =my_refract(IN.mIncidentVectorT, localNormalT, gRefractiveIndex );
	
	//locate intersection with back surface
	float3 pointOnBackTextureT;
	pointOnBackTextureT = gDepth*refractedVectorT + IN.mPositionT;
	
	float2 lookupPoint;
	lookupPoint.x = dot(IN.mBacktexS, pointOnBackTextureT);
	lookupPoint.y = dot(IN.mBacktexT, pointOnBackTextureT);
	
	OUT.mColor = tex2D( SamplerVideo,lookupPoint);
	
	return OUT;
}


/*
**------------------------------------------------
**		Glass pixel shader 1.4
**------------------------------------------------
*/
REF_PS_OUTPUT refract_ps_1_4( REF_VS_OUTPUT IN )
{
    REF_PS_OUTPUT OUT;
	
	//compute local normals - in texture (s&t) space
	float3 localNormalT;
	localNormalT = tex2D( SamplerBump, IN.mTexcoord ).xyz;
	localNormalT = 2*(localNormalT - 0.5)*gBumpAmount; //normal is in s&t space
	
	//find the refracted vector in TextureSpace
	float3 refractedVectorT =my_refract(IN.mIncidentVectorT, localNormalT, gRefractiveIndex );
	
	//locate intersection with back surface
	float3 pointOnBackTextureT;
	pointOnBackTextureT = gDepth*refractedVectorT + IN.mPositionT;
	
	float2 lookupPoint;
	lookupPoint.x = dot(IN.mBacktexS, pointOnBackTextureT);
	lookupPoint.y = dot(IN.mBacktexT, pointOnBackTextureT);
	
	OUT.mColor = tex2D( SamplerVideo, lookupPoint); 
	
    return OUT;
}

REF_PS_OUTPUT refract_ps_1_3( REF_VS_OUTPUT IN )
{
    REF_PS_OUTPUT OUT;
	
	OUT.mColor = tex2D( SamplerVideo, IN.mTexcoord );
	
    return OUT;
}


/*
**----------------------------------------------------------
**		Technique with one pass [TODO] add pass for ps_1_4
**----------------------------------------------------------
*/


technique refract_2_0
{
	pass Pass0 //for the rectangular glass piece
    {
		Sampler[0] = (SamplerVideo); 
		Sampler[1] = (SamplerBump);
		
		VertexShader = compile vs_2_0 refract_vs();
		PixelShader  = compile ps_2_0 refract_ps_2_0();
		
    }
    
}



technique refract_1_4
{
	pass Pass0 //for the rectangular glass piece
    {
		Sampler[0] = (SamplerVideo); 
		Sampler[1] = (SamplerBump);
		
		VertexShader = compile vs_1_1 refract_vs();
		PixelShader  = compile ps_1_4 refract_ps_1_4();
		
    }
    
}

technique refract_1_3
{
	pass Pass0 //for the rectangular glass piece
    {
		Sampler[0] = (SamplerVideo); 
	
		VertexShader = compile vs_1_1 refract_vs_1_3();
		PixelShader  = compile ps_1_3 refract_ps_1_3();
		
    }
    
}
