
//==============================================================================
// @mark	Clustering stepper
//==============================================================================

var google = {};
google.maps = {};
google.maps.event = {};
google.maps.event.addListener = function() {} 
google.maps.event.addDomListener = function(e, event, handler) { e.addEventListener(event, handler); } 

// maximum amount of time to spend processing markers before pausing the stepper
var STEP_MAX_TIME = 100;

// process this many markers before checking the time again
var STEP_TIME_CHECK_MODULO = 20;

// rate at which the stepper timer fires
var STEPPER_INTERVAL = 10;

//no of images in hover prefetched 
var prefetchValue = 5;

// Threshold for treating two coordinates as being exactly equal and therefore in a
// group rather than a cluster.

var arcsecond = 1 / 60 / 60;
var kIdenticalLocationEpsilon = 0.01 * arcsecond;
var widthOfLocationText = 182;
var editWidth = 50;
var eDragStates = {
	NODRAG: 0,
	DRAGSTART: 1,
	DRAGMOVE: 2
};

// annggupta: A Basic Event dispatcer
// https://stackoverflow.com/questions/20835768/addeventlistener-on-custom-object
function Dispatcher() {
	this.dispatchHandlers_ = {};
}

Dispatcher.prototype.on = function(eventName, handler) {
	
	if(typeof this.dispatchHandlers_[eventName] === "undefined") {
		this.dispatchHandlers_[eventName] = [];
	}

	this.dispatchHandlers_[eventName].push(handler);
};

Dispatcher.prototype.dispatch = function(eventName, p1, p2) {
	var handler, i, len, ref;
	ref = this.dispatchHandlers_[eventName];
	if (typeof ref === "undefined") {
	    return;
	}
	for (i = 0, len = ref.length; i < len; i++) {
	  handler = ref[i];
	  setTimeout(handler(p1,p2), 0);
	}
};

function lngLatBoundsContainPoint(bounds, point) 
{
	var newBound = new mapboxgl.LngLatBounds(bounds._sw, bounds._ne);
	newBound.extend(point);
	
	return (newBound._sw.lng == bounds._sw.lng) && (newBound._sw.lat == bounds._sw.lat) && (newBound._ne.lng == bounds._ne.lng) && (newBound._ne.lat == bounds._ne.lat);
}

//-------------------------------------------------------------------------------

var _clusterers = [];

//var aspectRatio_ = 180/110;
//-------------------------------------------------------------------------------

MarkerClusterer.stepper = function() {
	var clusterChanged = false;
	for ( var c = 0, clusterer; clusterer = _clusterers[c]; c++ ) {

		var pos = clusterer.stepperPosition;
		var bounds = clusterer.stepperBounds;
		var start = new Date().getTime();
		var total = clusterer.markers_.length;

		/*
		if ( clusterer.progressCallback_ ) {
					clusterer.progressCallback_( 0, total );
				}*/
		
		
	//	clusterer.setReady_(false);
		
		for (var i = pos; i < clusterer.markers_.length; i++) {

			if ( i % STEP_TIME_CHECK_MODULO == 0 ) {
				var now = new Date().getTime();
				if ( now - start > STEP_MAX_TIME ) {
					break;
				}
			}
		
			var marker = clusterer.markers_[ i ];
			var addedToCluster = false;

			if ( !marker.isAdded && clusterer.isMarkerInBounds_( marker, bounds ) ) {
				clusterChanged = true;
				var markerPos = marker.getLngLat();
				for ( var j = 0, cluster; cluster = clusterer.clusters_[ j ]; j++ ) {

					if ( !addedToCluster
						&& cluster.center_
						&& /*cluster.bounds_.contains( markerPos )*/ lngLatBoundsContainPoint(cluster.bounds_, markerPos) ) {

						addedToCluster = true;
						cluster.addMarker( marker, true, true );

						// performance optimization: assume added markers in a sequence likely
						// belong to the same cluster (like the case of tagging multiple photos
						// in a single operation)
						if (j > 0) {
							clusterer.clusters_[j] = clusterer.clusters_[0];
							clusterer.clusters_[0] = cluster;
						}				

						break;
					}
				}

				if ( !addedToCluster ) {
					// Create a new cluster.
					//alert("creating new cluster");
					var cluster = new Cluster( clusterer );
					cluster.addMarker( marker, true, true );
					clusterer.clusters_.push( cluster );
					clusterChanged = true;

				}
			}

			clusterer.stepperPosition = i;
		}

		var amount = clusterer.stepperPosition + 1;

		/*
		if ( clusterer.progressCallback_ ) {
					clusterer.progressCallback_( amount, total, "progress for: " + clusterer.stepperInterval );
				}*/
		

//		clusterer.setReady_(true);
//		clusterer.redraw();
		clusterer.updateClustersAsNeeded();

		if ( total == 0 || amount >= total ) { // all done

		//	clusterer.progressCallback_( amount, total, "FINAL progress for: " + clusterer.stepperInterval );

//			clusterer.setReady_(true);
//			clusterer.redraw();
			clusterer.updateClustersAsNeeded();

			if ( clusterer.stepperInterval ) {
				clearInterval( clusterer.stepperInterval );		
				clusterer.stepperInterval = null;
			}
		}
	}
	if(clusterChanged) {
		//alert('inside MarkerClusterer.stepper > clusterChanged');
		 markerClusterer.trigger('clusterChanged');
	}
}

//==============================================================================

// ==ClosureCompiler==
// @compilation_level ADVANCED_OPTIMIZATIONS
// @externs_url http://closure-compiler.googlecode.com/svn/trunk/contrib/externs/google_maps_api_v3.js
// ==/ClosureCompiler==

/**
 * @name MarkerClusterer for Google Maps v3
 * @version version 1.0
 * @author Luke Mahe
 * @fileoverview
 * The library creates and manages per-zoom-level clusters for large amounts of
 * markers.
 * <br/>
 * This is a v3 implementation of the
 * <a href="http://gmaps-utility-library-dev.googlecode.com/svn/tags/markerclusterer/"
 * >v2 MarkerClusterer</a>.
 */

/**
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


/**
 * Model for Marker
 *
 *
 */
function MarkerInfo() {
	this.lngLat_ = null;
	this.visible_ = false;
	this.draggable_ = false;
}

/**
 * Set LngLat of the marker.
 * @ignore
 */
MarkerInfo.prototype.setLngLat = function(lngLat) {

  this.lngLat_ = lngLat;
};

/**
 * Get LngLat of the marker.
 * @ignore
 */
MarkerInfo.prototype.getLngLat = function() {

  return this.lngLat_;
};

/**
 * Set visibilty of the marker.
 * @ignore
 */
MarkerInfo.prototype.setVisible = function(visible) {

  this.visible_ = visible;
};

/**
 * Get visibilty of the marker.
 * @ignore
 */
MarkerInfo.prototype.getVisible = function() {

  return this.visible_;
};

/**
 * Set draggable of the marker.
 * @ignore
 */
MarkerInfo.prototype.setDraggable = function(draggable) {

  this.draggable_ = draggable;
};

/**
 * Get draggable of the marker.
 * @ignore
 */
MarkerInfo.prototype.getDraggable = function() {

  return this.draggable_;
};


/**
 * A Marker Clusterer that clusters markers.
 *
 * @param {mapboxgl.Map} map The MapBox GL map to attach to.
 * @param {Object} opt_options support the following options:
 *     'gridSize': (number) The grid size of a cluster in pixels.
 *     'maxZoom': (number) The maximum zoom level that a marker can be part of a
 *                cluster.
 *     'zoomOnClick': (boolean) Whether the default behaviour of clicking on a
 *                    cluster is to zoom into it.
 *     'styles': (object) An object that has style properties:
 *       'url': (string) The image url.
 *       'height': (number) The image height.
 *       'width': (number) The image width.
 *       'anchor': (Array) The anchor position of the label text.
 *       'textColor': (string) The text color.
 *       'textSize': (number) The text size.
 * @constructor
 * @extends google.maps.OverlayView
 */
function MarkerClusterer(map, opt_options) {
  // MarkerClusterer implements google.maps.OverlayView interface. We use the
  // extend function to extend MarkerClusterer with google.maps.OverlayView
  // because it might not always be available when the code is defined so we
  // look for it at the last possible moment. If it doesn't exist now then
  // there is no point going ahead :)
  //this.extend(MarkerClusterer, google.maps.OverlayView);
  this.map_ = map;
  this.markers_ = [];
  this.clusters_ = [];
  this.styles_ = [];
  this.ready_ = false;
  this.dispatcher_ = new Dispatcher();

  var options = opt_options || {};

  this.gridSize_ = options['gridSize'] || 60;
  this.maxZoom_ = options['maxZoom'] || null;
  this.styles_ = options['styles'] || [];
  this.zoomOnClick_ = options['zoomOnClick'] != false;
  this.enableHoverIcons_ = true;
  

  Cluster.setStrings(options['hoverIconStrings']);
  
 // this.progressCallback_ = options['progressCallback'];

  this.setMap(map);

  this.prevZoom_ = this.map_.getZoom();
    
  this.draggedCluster_ = null;
  
  // Add the map event listeners
  var that = this;
  this.map_.on('zoomend', function() {
	//alert('zoom changed');
	if(this.draggedCluster_)
		this.draggedCluster_.cancelDrag();
	
  	//var maxZoom = that.map_.mapTypes[that.map_.getMapTypeId()].maxZoom;
	var maxZoom = that.map_.getMaxZoom(); //TODO(anggupta)::Need to check max zoom level based on map type
  	var zoom = that.map_.getZoom();
  	if (zoom < 0 || zoom > maxZoom) {
  	  return;
  	}
  	
    if (that.prevZoom_ != zoom) {
      that.prevZoom_ = that.map_.getZoom();     
      that.resetViewport();
    }
	
	if ( that.map_ != null ) {
	 	var mapBounds = that.map_.getBounds();
	  	if ( mapBounds != null ) {
			mapBounds = new mapboxgl.LngLatBounds( mapBounds.getSouthWest(), mapBounds.getNorthEast() );
			var bounds = that.getExtendedBounds( mapBounds );
			that.stepperBounds = bounds;
	  	}
	}

  	that.redraw();
  });

  this.map_.on('bounds_changed', function() {
	  if ( that.map_ != null ) {
		  var mapBounds = that.map_.getBounds();
		  if ( mapBounds != null ) {
			mapBounds = new mapboxgl.LngLatBounds( mapBounds.getSouthWest(), mapBounds.getNorthEast() );
			var bounds = that.getExtendedBounds( mapBounds );
			that.stepperBounds = bounds;
		}
	}

    that.redraw();
  });
}

/**
 * Extends a objects prototype by anothers.
 *
 * @param {Object} obj1 The object to be extended.
 * @param {Object} obj2 The object to extend with.
 * @return {Object} The new extended object.
 * @ignore
 */
MarkerClusterer.prototype.extend = function(obj1, obj2) {
  return (function(object) {
    for (property in object.prototype) {
      this.prototype[property] = object.prototype[property];
    }
    return this;
  }).apply(obj1, [obj2]);
};


/**
 * Implementaion of the interface method.
 * @ignore
 */
MarkerClusterer.prototype.onAdd = function() {

  this.setReady_(true);
};


/**
 * Implementation of the interface.
 * @ignore
 */
MarkerClusterer.prototype.idle = function() {};


/**
 * Implementation of the interface.
 * @ignore
 */
MarkerClusterer.prototype.draw = function() {};

/**
 *  Sets the styles.
 *
 *  @param {Object} styles The style to set.
 */
MarkerClusterer.prototype.setStyles = function(styles) {
  this.styles_ = styles;
};


/**
 *  Gets the styles.
 *
 *  @return {Object} The styles object.
 */
MarkerClusterer.prototype.getStyles = function() {
  return this.styles_;
};


/**
 * Whether zoom on click is set.
 *
 * @return {boolean} True if zoomOnClick_ is set.
 */
MarkerClusterer.prototype.isZoomOnClick = function() {
  return this.zoomOnClick_;
};



MarkerClusterer.prototype.getEnableHoverIcons = function() {
return this.enableHoverIcons_;
}

/**
 *  Returns the array of markers in the clusterer.
 *
 *  @return {Array.<google.maps.Marker>} The markers.
 */
MarkerClusterer.prototype.getMarkers = function() {
  return this.markers_;
};


/**
 *  Returns the array of markers in the clusterer.
 *
 *  @return {Array.<google.maps.Marker>} The number of markers.
 */
MarkerClusterer.prototype.getTotalMarkers = function() {
  return this.markers_;
};


/**
 *  Sets the max zoom for the clusterer.
 *
 *  @param {number} maxZoom The max zoom level.
 */
MarkerClusterer.prototype.setMaxZoom = function(maxZoom) {
  this.maxZoom_ = maxZoom;
};


/**
 *  Gets the max zoom for the clusterer.
 *
 *  @return {number} The max zoom level.
 */
MarkerClusterer.prototype.getMaxZoom = function() {
  return this.maxZoom_ || this.map_.mapTypes[this.map_.getMapTypeId()].maxZoom;
};


/**
 *  The function for calculating the cluster icon image.
 *
 *  @param {Array.<google.maps.Marker>} markers The markers in the clusterer.
 *  @param {number} numStyles The number of styles available.
 *  @param {boolean} identicalLocation True iff all markers in the cluster share the same GPS location.
 *  @return {Object} A object properties: 'text' (string) and 'index' (number).
 *  @private
 */
MarkerClusterer.prototype.calculator_ = function(markers, numStyles, identicalLocation) {
  var digits = 0;
  var count = markers.length;
  var dv = count;
  while (dv !== 0) {
    dv = parseInt(dv / 10, 10);
    digits++;
  }
    
  if(digits <= 1)
    count = "0" + count;

  digits = Math.min(digits, numStyles/2);

  var index = (digits > 0) ? (digits - 1) : 0;

  index *= 2;
  if (identicalLocation)
	index++;

  return {
    text: count,
    index: index
  };
};


/**
 * Set the calculator function.
 *
 * @param {function(Array, number)} calculator The function to set as the
 *     calculator. The function should return a object properties:
 *     'text' (string) and 'index' (number).
 *
 */
MarkerClusterer.prototype.setCalculator = function(calculator) {
  this.calculator_ = calculator;
};


/**
 * Get the calculator function.
 *
 * @return {function(Array, number)} the calculator function.
 */
MarkerClusterer.prototype.getCalculator = function() {
  return this.calculator_;
};


/**
 * Pushes a marker to the clusterer.
 *
 * @param {google.maps.Marker} marker The marker to add.
 * @private
 */
MarkerClusterer.prototype.pushMarkerTo_ = function(marker) {
  marker.setVisible(false);
  //anggupta-temp  marker.setMap(null);
  marker.isAdded = false;
  this.markers_.push(marker);
};


/**
 * Adds a marker to the clusterer and redraws if needed.
 *
 * @param {google.maps.Marker} marker The marker to add.
 * @param {boolean} opt_nodraw Whether to redraw the clusters.
 */
MarkerClusterer.prototype.addMarker = function(marker, opt_nodraw) {
  this.pushMarkerTo_(marker);
  if (!opt_nodraw) {
//    this.redraw();
  } else {
  // spam
//	  this.addMarkerToClusters_(marker, opt_nodraw);
  }
};


/**
 * Remove a marker from the cluster.
 *
 * @param {google.maps.Marker} marker The marker to remove.
 * @param {boolean} opt_nodraw Whether to redraw the clusters.
 * @return {boolean} True if the marker was removed.
 */
MarkerClusterer.prototype.removeMarker = function(marker, opt_nodraw) {
  var index = -1;
  if (this.markers_.indexOf) {
    index = this.markers_.indexOf(marker);
  } else {
    for (var i = 0, m; m = this.markers_[i]; i++) {
      if (m == marker) {
        index = i;
        break;
      }
    }
  }

  if (index == -1) {
    // Marker is not in our list of markers.
    return false;
  }

  this.markers_.splice(index, 1);
  marker.setVisible(false);
  //anggupta-temp marker.setMap(null);
  //marker.setMap(null);

 // if (!opt_nodraw) {
//	this.resetViewport();
//	this.redraw();
 // } else {
	this.removeMarkerFromClusters_(marker, opt_nodraw);
 // }
	
  return true;
};


MarkerClusterer.prototype.updateEnableHoverIcons = function(enableHoverIcons)
{
	this.enableHoverIcons_ = enableHoverIcons;
}



/**
 * Updates the location name visible on hover view of cluster.
 */
MarkerClusterer.prototype.updateClusterText = function(centerPt, locationName, bHaveCommonTags, tagIdList) {
	if (!this.ready_) {
		return;
	}
	
	if(locationName == "")
		return;
		var lat1 = centerPt.lat;
		var lng1 = centerPt.lng;
	
	for (var j = 0, cluster; cluster = this.clusters_[j]; j++) {

		var lat = cluster.getCenter().lat;
		var lng = cluster.getCenter().lng;

		if (Math.abs( lat1 - lat ) <= kIdenticalLocationEpsilon && Math.abs( lng1 - lng ) <= kIdenticalLocationEpsilon) {
				cluster.updateLocationText(locationName, bHaveCommonTags, tagIdList);
		}
	}
};


/**
 * Update clusters when thumbnail loaded
 */
MarkerClusterer.prototype.updateClusterFrontImage = function(mediaId) {
	
	for (var j = 0, cluster; cluster = this.clusters_[j]; j++) {
		if (cluster.markers_[0].id == mediaId) {
				cluster.updateIcon();
		}
	}
	
}
 

 
/**
 * Update clusters that need update.
 */
MarkerClusterer.prototype.updateClustersAsNeeded = function(forceUpdateAll) {
	if (!this.ready_) {
		return;
	}
	for (var j = 0, cluster; cluster = this.clusters_[j]; j++) {
		if (forceUpdateAll || cluster.needUpdate) {
			cluster.updateIcon();
		}
	}
};

/**
 * Sets the clusterer's ready state.
 *
 * @param {boolean} ready The state.
 * @private
 */
MarkerClusterer.prototype.setReady_ = function(ready) {
  if (this.ready_ != ready) {
    this.ready_ = ready;
    if ( ready )
	this.createClusters_();

  }
};



/**
 * Returns the number of clusters in the clusterer.
 *
 * @return {number} The number of clusters.
 */
MarkerClusterer.prototype.getTotalClusters = function() {
  return this.clusters_.length;
};


/**
 * Returns the google map that the clusterer is associated with.
 *
 * @return {google.maps.Map} The map.
 */
MarkerClusterer.prototype.getMap = function() {
  return this.map_;
};


/**
 * Sets the google map that the clusterer is associated with.
 *
 * @param {google.maps.Map} map The map.
 */
MarkerClusterer.prototype.setMap = function(map) {
  this.map_ = map;
};


/**
 * Returns the size of the grid.
 *
 * @return {number} The grid size.
 */
MarkerClusterer.prototype.getGridSize = function() {
  return this.gridSize_;
};


/**
 * Returns the size of the grid.
 *
 * @param {number} size The grid size.
 */
MarkerClusterer.prototype.setGridSize = function(size) {
  this.gridSize_ = size;
};

/**
 * Extends a bounds object by the grid size.
 *
 * @param {boolean} if true, longitudes will be ignored when testing if markers are visible on the map.
 */
MarkerClusterer.prototype.setMapIsWrapped = function( isWrapped ) {
  this.mapIsWrapped = isWrapped;
}

/**
 * Extends a bounds object by the grid size.
 *
 * @param {google.maps.LatLngBounds} bounds The bounds to extend.
 * @return {google.maps.LatLngBounds} The extended bounds.
 */
MarkerClusterer.prototype.getExtendedBounds = function(bounds) {

  if ( bounds == null )
	return null;
 
  //var projection = this.getProjection();

  // Turn the bounds into latlng.
  var tr = new mapboxgl.LngLat(bounds.getNorthEast().lng, bounds.getNorthEast().lat);
  var bl = new mapboxgl.LngLat(bounds.getSouthWest().lng, bounds.getSouthWest().lat);

  // Convert the points to pixels and the extend out by the grid size.
  var trPix = this.map_.project(tr);
  trPix.x += this.gridSize_;
  trPix.y -= this.gridSize_;

  var blPix = this.map_.project(bl);
  blPix.x -= this.gridSize_;
  blPix.y += this.gridSize_;

  // Convert the pixel points back to LatLng
  var ne = this.map_.unproject(trPix);
  var sw = this.map_.unproject(blPix);

  // Extend the bounds to contain the new bounds.
  bounds.extend(ne);
  bounds.extend(sw);

  return bounds;
};


/**
 * Determins if a marker is contained in a bounds.
 *
 * @param {google.maps.Marker} marker The marker to check.
 * @param {google.maps.LatLngBounds} bounds The bounds to check against.
 * @return {boolean} True if the marker is in the bounds.
 * @private
 */
MarkerClusterer.prototype.isMarkerInBounds_ = function(marker, bounds) {
	return true;// vaarora: tepm change..to add the marker even if it is not in bounds
  if ( bounds == null )
	return false;
  
  if ( this.mapIsWrapped ) {
	// all longitudes are visible, only test latitudes
	var lat = marker.getLngLat().lat;
	return bounds.getNorthEast().lat >= lat && bounds.getSouthWest().lat <= lat;
  }

  return lngLatBoundsContainPoint( bounds, marker.getLngLat() );
};


/**
 * Clears all clusters and markers from the clusterer.
 */
MarkerClusterer.prototype.clearMarkers = function() {
  this.resetViewport();

  // Set the markers a empty array.
  this.markers_ = [];
};


/**
 * Clears all existing clusters and recreates them.
 */
MarkerClusterer.prototype.resetViewport = function() {
  // Remove all the clusters
  for (var i = 0, cluster; cluster = this.clusters_[i]; i++) {
    cluster.remove();
  }

  // Reset the markers to not be added and to be invisible.
  for (var i = 0, marker; marker = this.markers_[i]; i++) {
    marker.isAdded = false;
    //anggupta-temp marker.setMap(null);
    marker.setVisible(false);
  }

  this.clusters_ = [];
};


/**
 * Redraws the clusters.
 */
MarkerClusterer.prototype.redraw = function() {
  this.createClusters_();
};



MarkerClusterer.prototype.clusterMoved = function(cluster, center) {
    cluster.changePosition(center);
}
/**
 * Creates the clusters.
 *
 * @private
 */
MarkerClusterer.prototype.createClusters_ = function() {

  if (!this.ready_) {
    return;
  }

  if ( this.stepperInterval ) {
	clearInterval( this.stepperInterval );
	this.stepperInterval = null;
  }
  
  var registered = false;
  for( var i = 0, clusterer; clusterer = _clusterers[i]; i++) {
	if ( clusterer == this ) {
		registered = true;
		break;
	}
  }
  
  if ( !registered ) {

	_clusterers.push( this );
  }
  
  this.stepperPosition = 0;

  // Get our current map view bounds.
  // Create a new bounds object so we don't affect the map.
  
  if ( this.map_ != null ) {
	  var mapBounds = this.map_.getBounds();
	  if ( mapBounds != null ) {
		mapBounds = new mapboxgl.LngLatBounds( mapBounds.getSouthWest(), mapBounds.getNorthEast() );
		var bounds = this.getExtendedBounds( mapBounds );

		this.stepperBounds = bounds;
		this.stepperInterval = setInterval( "MarkerClusterer.stepper()", STEPPER_INTERVAL );
	  }  
  }
};

/**
 * Add marker to the existing clusters.
 * @private
 * @param {google.maps.Marker} marker The marker to add.
 * @param {boolean} opt_nodraw Whether to redraw the clusters.
 */
MarkerClusterer.prototype.addMarkerToClusters_ = function(marker, opt_nodraw) {

//AgDebugPrint( "addMarkerToClusters_" );

	//if (!this.ready_ || this.map_ == null ) {
	if (this.map_ == null ) {
//AgDebugPrint( "map is null" );
		return false;
	}
	
	var mapBounds = this.map_.getBounds();
	
	if ( mapBounds == null ) {
//AgDebugPrint( "map bounds null" );
		return false;
	}
	
	// Get our current map view bounds.
	// Create a new bounds object so we don't affect the map.
	mapBounds = new mapboxgl.LngLatBounds( mapBounds.getSouthWest(), mapBounds.getNorthEast() );
	var bounds = this.getExtendedBounds( mapBounds );
	
	var added = false;

//AgDebugPrint( "marker already added? " + marker.isAdded );
	
	if (!marker.isAdded && this.isMarkerInBounds_(marker, bounds)) {

		//alert( "checking clusters..." );

		for (var j = 0, cluster; cluster = this.clusters_[j]; j++) {
			if (cluster.getCenter() &&
				cluster.isMarkerInClusterBounds(marker)) {
				added = true;
				cluster.addMarker(marker, opt_nodraw);

				//AgDebugPrint( "+ added to cluster, now contains: " + cluster.markers_.length );
				
				// performance optimization: assume added markers in a sequence likely
				// belong to the same cluster (like the case of tagging multiple photos
				// in a single operation)
				if (j > 0) {
					this.clusters_[j] = this.clusters_[0];
					this.clusters_[0] = cluster;
				}				
				break;
			} 
		}
		
		if (!added) {

			//alert( "* creating new cluster" );

			// Create a new cluster.
			var cluster = new Cluster(this);
			cluster.addMarker(marker, opt_nodraw);
			
			// performance optimization: see above
			if (this.clusters_.length > 0) {
				this.clusters_.push(this.clusters_[0]);
				this.clusters_[0] = cluster;
			} else {
				this.clusters_.push(cluster);
			}
		}
	}
	
	return added;
};


/**
 * Check if a marker can be drag-n-dropped into an existing cluster.
 * @param {google.maps.LatLng} point The position of the marker.
 * @param {boolean} opt_nodraw Whether to highlight the target cluster.
 * @return {Cluster} The target cluster, null otherwise.
*/
MarkerClusterer.prototype.canAcceptMarkerDropImp = function(point, opt_nodraw, currentTargetCluster) {
    if (!this.ready_) {
        return null;
    }

    // Get our current map view bounds.
    // Create a new bounds object so we don't affect the map.
    var mapBounds = new mapboxgl.LngLatBounds(this.map_.getBounds().getSouthWest(),
												 this.map_.getBounds().getNorthEast());
    var bounds = this.getExtendedBounds(mapBounds);
	//alert(bounds);
    var targetCluster = null;
    if (lngLatBoundsContainPoint(bounds, point)) {
		//alert("Map contains point");
        for (var j = 0, cluster; cluster = this.clusters_[j]; j++) {
            if (!cluster.isClusterDragged() &&
				cluster.clusterIcon_.visible_ &&
				cluster.getCenter() &&
				cluster.containsPoint(point)) {
                	targetCluster = cluster;
                break;
            }
        }
    }

  /*   Not needed now as we are treating group cluster (single pin multiple media) similar to a multiple pin cluster
	 if (targetCluster && !targetCluster.isGroupCluster) {
		//alert("Not group cluster");
        if (currentTargetCluster) {
            currentTargetCluster.clusterIcon_.showSmallIcon_();
        }
        return null;
    }*/

    if (!opt_nodraw) {
		//alert("drawing hover icon");
		if (currentTargetCluster != targetCluster) {
            if (currentTargetCluster) {
                currentTargetCluster.clusterIcon_.showSmallIcon_();
            }

            if (targetCluster) {
                targetCluster.clusterIcon_.showDropIcon_();
            }
        }
    }

    return targetCluster;
};

/**
 * Check if a marker can be drag-n-dropped into an existing cluster.
 * @param {google.maps.Marker} marker The marker to check.
 * @param {boolean} opt_nodraw Whether to highlight the target cluster.
 * @return {Cluster} The target cluster, null otherwise.
 */
MarkerClusterer.prototype.canAcceptMarkerDrop = function(marker, opt_nodraw, currentTargetCluster) {
	return this.canAcceptMarkerDropImp( marker.getLngLat(), opt_nodraw, currentTargetCluster )
};

/**
 * Remove marker from the existing clusters.
 * @private
 * @param {google.maps.Marker} marker The marker to remove.
 * @param {boolean} opt_nodraw Whether to redraw the clusters.
*/
MarkerClusterer.prototype.removeMarkerFromClusters_ = function(marker, opt_nodraw) {

//AgDebugPrint( "removeMarkerFromClusters_" );

//	if (!this.ready_) {
//AgDebugPrint( "NOT READY" );
//		return false;
//	}
	
	for (var j = 0, cluster; cluster = this.clusters_[j]; j++) {
		marker.isAdded = false;
		if (cluster.removeMarker(marker, opt_nodraw)) {
			if (cluster.markers_.length == 0) {
//AgDebugPrint( "delete empty cluster" );
				// remove the emtpy cluster
				this.clusters_.splice(j, 1);
				cluster.remove(); // remove the cluster
			}
			return true; // removed
		}
	}
	
	return false;
};

/**
 * Add an event listener on this object.
 * @private
 * @param {eventName} name of the event.
 * @param {fucntion} handler to invoke.
*/
MarkerClusterer.prototype.addListener = function(eventName, handler) {
	this.dispatcher_.on(eventName, handler);
};

/**
 * Trigger the event on this object.
 * @private
 * @param {eventName} name of the event.
*/
MarkerClusterer.prototype.trigger = function(eventName, p1, p2) {
	this.dispatcher_.dispatch(eventName, p1, p2);
};


/**
 * Return if any cluster dragged
 * @return {Cluster} The dragged cluster, null otherwise.
 */
MarkerClusterer.prototype.getDraggedClusterIfAny = function() {
	return this.draggedCluster_;
};

MarkerClusterer.prototype.getClusters = function() {
	return this.clusters_; 
};


	
	var getResourceData  = function() {
		var imageName = arguments[0];
		var imgVar = arguments[1];
		var queryParams = {};
		queryParams['QStringArgument'] = imageName;
		var callbackFunc = (response,imgVarArg = imgVar) => {
			window[imgVarArg] = "data:image/png;base64," + response;
		};
		CEMHelper.sendEventToApp(CEMHelper.sEventName.kGetResourceData, queryParams, callbackFunc);
	}
	
	
/**
 * A cluster that contains markers.
 *
 * @param {MarkerClusterer} markerClusterer The markerclusterer that this
 *     cluster is associated with.
 * @constructor
 * @ignore
 */
function Cluster(markerClusterer) {
  this.markerClusterer_ = markerClusterer;
  this.map_ = markerClusterer.getMap();
  this.gridSize_ = markerClusterer.getGridSize();
  this.center_ = null;
  this.markers_ = [];
  this.sumX_ = 0;
  this.sumY_ = 0;
  this.needUpdate = true;
  this.bounds_ = null;
  this.iconBounds_ = null;
  this.isGroupCluster = false;
  this.isSelected = false;
  this.clusterIcon_ = new ClusterIcon(this, markerClusterer.getStyles(),
      markerClusterer.getGridSize());
  this.dragState_ = eDragStates.NODRAG;
  this.lastDragTime_ = 0;
  this.tagIdList_ = 0;
  this.bHasCommonTags_ = false;
  
  var that = this;
	this.map_.on( 'mousemove', function( event ) {
			if(that.dragState_ == eDragStates.DRAGSTART || that.dragState_ == eDragStates.DRAGMOVE)
			{	
				that.clusterDragMove(event.lngLat);
			}
	} );
	
	this.map_.on( this.map_, 'mouseout', function( event ) {
			if(that.dragState_ == eDragStates.DRAGSTART || that.dragState_ == eDragStates.DRAGMOVE)
			{	
				that.cancelDrag();
			}
	} );
}


Cluster.prototype.updateLocationText = function(locationName, bHaveCommonTags, tagIdList)
{
	this.tagIdList_ = tagIdList;
	this.bHasCommonTags_ = bHaveCommonTags;
    this.hasLocationNameForCluster = true;
	this.clusterIcon_.updateLocationName(locationName);
}

/**
 * Determins if a marker is already added to the cluster.
 *
 * @param {google.maps.Marker} marker The marker to check.
 * @return {boolean} True if the marker is already added.
 */
Cluster.prototype.isMarkerAlreadyAdded = function(marker) {
  if (this.markers_.indexOf) {
    return this.markers_.indexOf(marker) != -1;
  } else {
    for (var i = 0, m; m = this.markers_[i]; i++) {
      if (m == marker) {
        return true;
      }
    }
  }
  return false;
};

/**
 * Get cluster icon bounds (to account for all area taken by cluster)
 *
 * @return {google.maps.LatLngBounds} The bounds.
 */
Cluster.prototype.getClusterBounds = function() {
  
  var clusterCenter = this.center_;
  var gridSize =  this.gridSize_;
  
  // Convert the points to pixels and the extend out by the grid size.
  var trPix = this.map_.project(clusterCenter);
  trPix.x += gridSize / 2;
  trPix.y -= gridSize;
  
  var blPix = this.map_.project(clusterCenter);
  blPix.x -= gridSize / 2;

  // Convert the pixel points back to LatLng

  var ne = this.map_.unproject(trPix);
  var sw = this.map_.unproject(blPix);
  
  var bounds = new mapboxgl.LngLatBounds();

  // Extend the bounds to contain the new bounds.
  bounds.extend(ne);
  bounds.extend(sw);
  
  return bounds;
};


/**
 * Add a marker to the cluster.
 *
 * @param {google.maps.Marker} marker The marker to add.
 * @param {boolean} opt_nodraw Whether to redraw the clusters.
 * @return {boolean} True if the marker was added.
 */
Cluster.prototype.addMarker = function(marker, opt_nodraw, opt_nocheckmembership) {

//AgDebugPrint( "addMarker" );

  if (opt_nocheckmembership != true && this.isMarkerAlreadyAdded(marker)) {
	marker.isAdded = true;
//AgDebugPrint( "already added" );
    return false;
  }

  //make native marker (google.maps.Marker) invisible
  
  marker.setVisible(false);
  //anggupta-temp marker.setMap(null);
 
  marker.isAdded = true;
  marker.isOver = false;

  this.markers_.push(marker);

  // Use the average location of all markers as the cluster's center

  var pos = marker.getLngLat();
  this.sumX_ = this.sumX_ + pos.lat;
  this.sumY_ = this.sumY_ + pos.lng;
  var count = this.markers_.length;

  this.center_ = new mapboxgl.LngLat( this.sumY_ / count, this.sumX_ / count);

  this.calculateBounds_();
  this.needUpdate = true;

  var that = this;  

  /*
  marker.clusterAnimationListener = google.maps.event.addListener(marker, 'animation_changed', function() {  
	if ( marker.getAnimation && marker.getAnimation() == google.maps.Animation.BOUNCE ) {
		that.clusterIcon_.attention_ = true;
	} else {
		that.clusterIcon_.attention_ = null;
	}
	if (!opt_nodraw && that.markers_ )
		that.updateIcon();
  } );
  */

  if (!opt_nodraw) {
	this.updateIcon();
  }
  return true;
};

/**
 * Remove the marker from the cluster.
 *
 * @param {google.maps.Marker} marker The marker to add.
 * @param {boolean} opt_nodraw Whether to redraw the clusters.
 * @return {boolean} True if the marker was added.
 */
Cluster.prototype.removeMarker = function(marker, opt_nodraw) {

//AgDebugPrint( "removeMarker, count:" + this.markers_.length );

	if ( this.markers_.length < 1 ) {
//AgDebugPrint( "minimal cluster, nothing to remove" );
		return false;
	}
	
	var index = -1;
	if (this.markers_.indexOf) {
		index = this.markers_.indexOf(marker);
	} else {
		for (var i = 0, m; m = this.markers_[i]; i++) {
			if (m == marker) {
				index = i;
				break;
			}
		}
	}
	
	if (index == -1) {
		// Marker is not in our list of markers.
//AgDebugPrint("marker not in this cluster's list" );
		return false;
	}

	 //AgDebugPrint( "old center: " + new google.maps.LatLng( this.sumX_ / this.markers_.length, this.sumY_ / this.markers_.length ) );

	// remove the marker from the cluster
	this.markers_.splice(index, 1);
	marker.setVisible(false);
	//marker.setMap(null);

//AgDebugPrint("marker removed from our list, remaining: " + this.markers_.length );

	if ( marker.clusterAnimationListener ) {
		google.maps.event.removeListener( marker.clusterAnimationListener );
		marker.clusterAnimationListener = null;
	}

	// update cluster's center	
	var pos = marker.getLngLat();
	this.sumX_ = this.sumX_ - pos.lat;
	this.sumY_ = this.sumY_ - pos.lng;
	var count = this.markers_.length;
	
	if (count > 0) {
//AgDebugPrint( "old center: " + this.center_ );
	  this.center_ = new mapboxgl.LngLat( this.sumY_ / count, this.sumX_ / count );
//AgDebugPrint( "new center: " + this.center_ );
	  this.calculateBounds_();
	} else {
		this.center_ = null;
	}
	
	this.needUpdate = true;
	
	if (!opt_nodraw) {

//AgDebugPrint("updating cluster icon" );

		this.updateIcon();
	}

	return true;
};

Cluster.prototype.markerMoved = function( marker ) {
	this.removeMarkerFromClusters_( marker, true );
	
}


Cluster.prototype.changePosition = function(center) {
    this.center_ = center;
    this.clusterIcon_.setCenter(center);
    this.clusterIcon_.show();

}


/**
 * Returns the marker clusterer that the cluster is associated with.
 *
 * @return {MarkerClusterer} The associated marker clusterer.
 */
Cluster.prototype.getMarkerClusterer = function() {
  return this.markerClusterer_;
};


/**
 * Returns the bounds of the cluster.
 *
 * @return {google.maps.LatLngBounds} the cluster bounds.
 */
Cluster.prototype.getBounds = function() {
  this.calculateBounds_();
  return this.bounds_;
};


/**
 * Removes the cluster
 */
Cluster.prototype.remove = function() {

  for (var i = 1, marker; marker = this.markers_[i]; i++) {
	if ( marker.clusterAnimationListener ) {
		google.maps.event.removeListener( marker.clusterAnimationListener );
		marker.clusterAnimationListener = null;
	}
  }

  this.clusterIcon_.remove();
  delete this.markers_;
};


/**
 * Returns the center of the cluster.
 *
 * @return {google.maps.LatLng} The cluster center.
 */
Cluster.prototype.getCenter = function() {
  return this.center_;
};


/**
 * Calculated the bounds of the cluster with the grid.
 *
 * @private
 */
Cluster.prototype.calculateBounds_ = function() {
  var bounds = new mapboxgl.LngLatBounds(this.center_, this.center_);
  this.bounds_ = this.markerClusterer_.getExtendedBounds(bounds);
  this.iconBounds_ = this.getClusterBounds();
};


/**
 * Determines if a marker lies in the clusters bounds.
 *
 * @param {google.maps.Marker} marker The marker to check.
 * @return {boolean} True if the marker lies in the bounds.
 */
Cluster.prototype.isMarkerInClusterBounds = function(marker) {
  if ( this.center_ == null )
	return false;
  if ( this.bounds_ == null )
	return false;
  return lngLatBoundsContainPoint(this.bounds_, marker.getLngLat());
};


/**
 * Determines if a point lies in the clusters bounds.
 *
 * @param {google.maps.LatLng} marker The marker to check.
 * @return {boolean} True if the marker lies in the bounds.
 */
Cluster.prototype.isInClusterBounds = function(point) {
	if ( this.bounds_ == null )
		return false;
	return lngLatBoundsContainPoint(this.bounds_, point);
};

/**
 * Determines if a point lies in the cluster's icon bounds.
 *
 * @param {google.maps.LatLng} point The point to check.
 * @return {boolean} True if the point lies in the bounds.
 */
Cluster.prototype.containsPoint = function(point) {
	if ( this.iconBounds_ == null )
		return false;
	return lngLatBoundsContainPoint(this.iconBounds_, point);
};

/**
 * Returns the map that the cluster is associated with.
 *
 * @return {google.maps.Map} The map.
 */
Cluster.prototype.getMap = function() {
  return this.map_;
};

/**
 * Updates the cluster icon
 */
Cluster.prototype.updateIcon = function() {
  this.needUpdate = false;
  this.isGroupCluster = false;

  var clusterLocked = false;
  var clusterSelected = false;

  for (var i = 0, marker; marker = this.markers_[i]; i++) {
	  if (!clusterLocked && !marker.getDraggable())
		  clusterLocked = true;
	  if (!clusterSelected && marker.selected)
		  clusterSelected = true;
  }
  
  this.isSelected = clusterSelected;
  this.hasLocationNameForCluster = false;
  this.clusterIcon_.setLocked(clusterLocked);
  this.clusterIcon_.setSelected(clusterSelected);
  this.clusterIcon_.updateHoverViewForDialogGrid();
  var identicalLocation = false;

  if(this.markers_.length == 1) {
	  this.isGroupCluster = true;   //treat single marker as group will do no harm as we treat them identically at UI end as well
  }
  if (this.markers_.length >= 2) {
	  var pos0 = this.markers_[0].getLngLat();
	  var lat = pos0.lat;
	  var lng = pos0.lng;

	  identicalLocation = true;

	  for (var i = 1, marker; marker = this.markers_[i]; i++) {
		  var pos = marker.getLngLat();
		  if (Math.abs( pos.lat - lat ) > kIdenticalLocationEpsilon ||
			  Math.abs( pos.lng - lng ) > kIdenticalLocationEpsilon) {
			  identicalLocation = false;
			  break;
		  }
	  }

	  this.isGroupCluster = identicalLocation;
  }

  if (!identicalLocation) {
	  var zoom = this.map_.getZoom();
	  var mz = this.markerClusterer_.getMaxZoom();

	  if (zoom > mz) {
		// The zoom is greater than our max zoom so show all the markers in cluster.
		for (var i = 0, marker; marker = this.markers_[i]; i++) {
		  //anggupta-temp  marker.setMap(null);                                //masingha: this should be set to null. need to verify
		  //marker.setVisible(true);
		}
		return;
	  }
  } 
  /* Enable this piece of code for debugging pin/cluster anchor positions
  else {
	  var zoom = this.map_.getZoom();
	  var mz = this.markerClusterer_.getMaxZoom();
	  
	  if (zoom > mz) {
		  // The zoom is greater than our max zoom so show all the markers in cluster.
		  for (var i = 0, marker; marker = this.markers_[i]; i++) {
			  if ((i%2)==1) marker.setIcon(null);
			  marker.setMap(this.map_);
			  marker.setVisible(true);
		  }
	  }
  }
  */

  /*if (this.markers_.length < 2) {
    // We have 0 or 1 markers so hide the icon.
    this.clusterIcon_.hide();
	
	if (this.markers_.length > 0) {
		// exactly 1 marker in the cluster, make sure it is visible
		this.markers_[0].setMap(this.map_);
		this.markers_[0].setVisible(true);
	}
	  
    return;
  }*/
	
  var numStyles = this.markerClusterer_.getStyles().length;
  var sums = this.markerClusterer_.getCalculator()(this.markers_, numStyles, identicalLocation);
  this.clusterIcon_.setCenter(this.center_);
  this.clusterIcon_.setSums(sums);
  this.clusterIcon_.show();
};

/**
 * put the hover view of cluster in visible map bounds
 *
 */
Cluster.prototype.putClusterInBounds = function() {

		// Get our current map view bounds.
		var bounds = new mapboxgl.LngLatBounds(this.getMarkerClusterer().getMap().getBounds().getSouthWest(),
												 this.getMarkerClusterer().getMap().getBounds().getNorthEast());
		
		
		if ( bounds == null )
			return null;
 
		// Turn the bounds into latlng.
		var tr = new mapboxgl.LngLat(bounds.getNorthEast().lng, bounds.getNorthEast().lat);
		var bl = new mapboxgl.LngLat(bounds.getSouthWest().lng, bounds.getSouthWest().lat);

		// Convert the points to pixels
		var trPix = this.map_.project(tr);
		var blPix = this.map_.project(bl);

		var clPix = this.map_.project(this.center_);
   

		var bottomAxis = blPix.y;
		var leftAxis = blPix.x;
		var topAxis = trPix.y;  
		var rightAxis  = trPix.x;

		var clusterX = clPix.x;
		var clusterY = clPix.y;
  
		var clusterWidth = 215;
		var clusterHeight = 206;
		
		if(!this.getMarkerClusterer().getEnableHoverIcons())
		clusterHeight = 186;
		
		if(leftAxis >= clusterX - clusterWidth/2)
		{
			var diff = leftAxis - clusterX + clusterWidth/2;
			leftAxis = leftAxis - diff ;
			rightAxis = rightAxis - diff ;
		}
		else if(rightAxis <= clusterX + clusterWidth/2)
		{
			var diff =  clusterX + clusterWidth/2 - rightAxis ;
			leftAxis = leftAxis + diff;
			rightAxis = rightAxis + diff;
		}
	
		if(topAxis  >= clusterY - clusterHeight)
		{
			var diff = topAxis - clusterY + clusterHeight ;
			topAxis = topAxis - diff;
			bottomAxis = bottomAxis - diff;
		}
		else if(bottomAxis <= clusterY )
		{
			var diff = clusterY - bottomAxis ;
			topAxis = topAxis + diff;
			bottomAxis = bottomAxis + diff;
		}
 
	//check for top left view : <Hybrid>
	if(   (clusterY  - clusterHeight -topAxis ) < 20 && (rightAxis - clusterX - clusterWidth/2) < 120 )
	{
	var diff = 120 - (rightAxis - clusterX - clusterWidth/2); 
	rightAxis = rightAxis + diff;
	leftAxis = leftAxis + diff;
	
	diff = 20 - (clusterY - clusterHeight - topAxis  ); 
	
	topAxis = topAxis - diff;
	bottomAxis = bottomAxis - diff;
	}
	var newX = ( rightAxis + leftAxis )/2;
	var newY = (topAxis + bottomAxis )/2;

  
	var point = new mapboxgl.Point( newX, newY );

    // Convert the pixel points back to LatLng
	var cntr = this.map_.unproject(point);

	this.getMarkerClusterer().getMap().setCenter(cntr);

}


/**
 * return whether cluster is being dragged 
 *
 */
Cluster.prototype.isClusterDragged = function() {
	return (this.dragState_ == eDragStates.DRAGMOVE);  
};

/**
 * return current drag state of cluster
 *
 */
Cluster.prototype.getDragState = function() {
	return this.dragState_;  
};

/**
 * 
 *
 */
Cluster.prototype.cancelDrag = function() {
	//alert("cancel drag");
	this.dragState_ = eDragStates.NODRAG;
	this.lastDragTime_ = 0;
	this.markerClusterer_.draggedCluster_ = null;

};

/**
 * 
 *
 */
Cluster.prototype.clusterDragStart = function(event) {

	//No need to check for groupCluster as we are treating all pins similar
	//if(this.isGroupCluster)
	{
		//alert("Cluster: drag start");
		this.lastDragTime_ = new Date().getTime();
		this.dragState_ = eDragStates.DRAGSTART;
		this.markerClusterer_.draggedCluster_ = this;
	}	
};

/**
 * 
 * * @param {google.maps.LatLng} latlng The position in latlng.
 */
Cluster.prototype.clusterDragMove = function(lngLat) {
	//alert("Cluster: drag move");
	this.dragState_ = eDragStates.DRAGMOVE;
	this.changePosition(lngLat);
	
	markerClusterer.trigger( 'clusterdragmove', this);
};

Cluster.prototype.getFirstTagId = function(){
	
	var strList = [];
	strList = this.tagIdList_.split(",");
	
	if(strList.length)
		return strList[0];
}

Cluster.prototype.getTagIdList = function() {
	return this.tagIdList_;
}

Cluster.prototype.hasCommonTags = function(){
	return this.bHasCommonTags_;
}

/**
 * 
 *
 */
Cluster.prototype.clusterDragEnd = function(event) {
	var now = new Date().getTime();
	if(!this.lastDragTime_ || !((now - this.lastDragTime_) > 300))  return;
	
	//alert("Cluster: drag end");
	this.dragState_ = eDragStates.NODRAG;   //drag workflow completed
	this.lastDragTime_ = 0;
	this.markerClusterer_.draggedCluster_ = null;   //we should not be needing this after drag end

	markerClusterer.trigger( 'clusterdragend', this);
	
};

Cluster.addLocationString_ = "";
Cluster.editString_ = "";
Cluster.seeAllString_ = "";

Cluster.setStrings = function( hoverIconStrings )
{

	Cluster.addLocationString_ = hoverIconStrings['addLocationFieldText'];
	Cluster.editString_ = hoverIconStrings['editFieldText'];
	Cluster.seeAllString_ = hoverIconStrings['seeAllFieldText'];

}

	var baseUrl				  = "http://adobe.com/lightroom_blank/";
	var selectedImageBackground;
	var normalImageBackground;
	var seeAll_N   			  ;
	var seeAll_H   			  ;
	var trash_N				  ;
	var trash_H				  ;
	var editLocationPencil_N  ;
	var editLocationPencil_H  ;
	var navigateLeftIcon 	  ;
	var navigateRightIcon 	  ;
	var getLocationNameIcon_N ;
	var getLocationNameIcon_H ;
	var editButtonBackgroind_H;
	var backgroundHoverGrid;
	var backgroundHoverDiaog;
	var greenTickImage;
	var searchYellowMarker;
	var redCrossImage;
	var callOutCenter;
	var callOutLeft;
	var callOutRight;
	var noImageAvailableHoverUrl ; 
	var noImageAvailableNormalUrl ;
	
getResourceData("Cluster_Pin_Normal.png"   ,"normalImageBackground");
getResourceData("Cluster_Pin_Selected.png" ,"selectedImageBackground");
getResourceData("seeAll_N.png"			   ,"seeAll_N");
getResourceData("seeAll_H.png"			   ,"seeAll_H");
getResourceData("editLocationPencil_N.png" ,"editLocationPencil_N");
getResourceData("editLocationPencil_H.png" ,"editLocationPencil_H");
getResourceData("navigateLeft.png"		   ,"navigateLeftIcon");
getResourceData("navigateRight.png"		   ,"navigateRightIcon");
getResourceData("Getlocation_N.png"		   ,"getLocationNameIcon_N");
getResourceData("Getlocation_H.png"		   ,"getLocationNameIcon_H");
getResourceData("Edit_button.png"		   ,"editButtonBackgroind_H");
getResourceData("Hover_Pin_Grid.png"	   ,"backgroundHoverGrid");
getResourceData("Hover_Pin_Dialog.png"	   ,"backgroundHoverDiaog");
getResourceData("E_TeaserBody.png"	       ,"greenTickImage");
getResourceData("O_Search_Pin.png"		   ,"searchYellowMarker");
getResourceData("E_RemoveIcon.png"	       ,"redCrossImage");
getResourceData("CallOut_Center.png"	   ,"callOutCenter");
getResourceData("CallOut_Left.png"	       ,"callOutLeft");
getResourceData("CallOut_Right.png"	       ,"callOutRight");
getResourceData("generic_thumb_hover.png"  ,"noImageAvailableHoverUrl");
getResourceData("generic_thumb_normal.png" ,"noImageAvailableNormalUrl");


/**
 * A cluster icon
 *
 * @param {Cluster} cluster The cluster to be associated with.
 * @param {Object} styles An object that has style properties:
 *     'url': (string) The image url.
 *     'height': (number) The image height.
 *     'width': (number) The image width.
 *     'anchor': (Array) The anchor position of the label text.
 *     'textColor': (string) The text color.
 *     'textSize': (number) The text size.
 * @param {number} opt_padding Optional padding to apply to the cluster icon.
 * @constructor
 * @extends google.maps.OverlayView
 * @ignore
 */
function ClusterIcon(cluster, styles, opt_padding) {
  //cluster.getMarkerClusterer().extend(ClusterIcon, google.maps.OverlayView);

	this.styles_ = styles;
	this.padding_ = opt_padding || 0;
	this.cluster_ = cluster;
	this.center_ = null;
	this.map_ = cluster.getMap();
	this.marker_ = null;
	this.div_ = document.createElement('DIV');
	this.sums_ = null;
	this.visible_ = false;
	this.current_state_ = "small";
	this.states_ = ["small",  "hover"];
	this.views_ = {};
	this.hoverTimeOut_ = null;
	this.hasValidLocationName_ = false;   //implies if it's default text or valid location
	
 //hover view divs
	this.hoverHTML_			 = null;
	this.addLocationField_	 = null;
	this.addLocationText_	 = null;
	this.getLocationNameIcon_= null;
	this.addLocationToolTip_ = "";
	this.editLocationIcon_ 	 = null;
	this.thumbnailHoverDiv_  = null;
	this.thumbnailHover_	 = null;
	this.navigateLeft_		 = null;
	this.navigateRight_		 = null;
	this.editField_			 = null;
	this.editText_			 = null;
	this.editToolTip_ 		 = null;
	this.seeAllField_		 = null;
	this.seeAllText_		 = null;
	this.seeAllIcon_		 = null;
	this.trashField_         = null;
	this.trashIcon_          = null;
	this.currentHoverIndex   = 0;
	this.textDiv_ 		     = null;
	this.initialize();
    
	//this.addTo(this.map_);
	this.onAdd();
	this.draw();
}


ClusterIcon.prototype.createHoverView = function(){

var that  = this;

	
this.hoverHTML_= document.createElement('DIV');
this.hoverHTML_.setAttribute("class","hovered backgroundHoverGrid");
this.hoverHTML_.style.backgroundImage = "url(" + backgroundHoverGrid + ")"; 


this.addLocationField_ = document.createElement('DIV');
this.addLocationField_.setAttribute("class", "addLocationFieldHover");

this.addLocationText_ =  document.createElement('DIV');
this.addLocationText_.setAttribute("class", "addLocationTextHover");
this.addLocationText_.innerHTML = addLocationText;
this.addLocationText_.style.paddingTop = "8px";
this.addLocationText_.style.paddingLeft = "32px";
this.addLocationText_.style.cursor = "pointer";

this.getLocationNameIcon_ = document.createElement('img');
this.getLocationNameIcon_.setAttribute("class",'getLocationNameIconHover');
this.getLocationNameIcon_.src = getLocationNameIcon_N;
//this.getLocationNameIcon_.style.visibility = "hidden";

//bug4163924
//here the calls for localization strings are async due to which we have to assign the fresh
//updated value to the toolTip
//addLocationText is global variable which contains localized one string
this.addLocationToolTip_ = addLocationText;
this.addLocationField_.appendChild(this.getLocationNameIcon_);
this.addLocationField_.appendChild(this.addLocationText_);

google.maps.event.addDomListener(this.addLocationField_, 'click' ,function(e) { 
	  e.stopPropagation();
	  if(!that.hasValidLocationName_ ) {
		  that.getLocationOfCluster();
	  }
});

google.maps.event.addDomListener(this.addLocationField_, 'dlbclick' ,function(e) { 
	  e.stopPropagation();
});


google.maps.event.addDomListener(this.addLocationText_, 'mouseover' ,function() { 


	if(!that.hasValidLocationName_ )
	{
//		that.addLocationText_.style.fontWeight = "600";		
		that.addLocationText_.style.color = "white";
		that.getLocationNameIcon_.src = getLocationNameIcon_H;
		
	}
		showHoverTip(that.addLocationToolTip_);	
		
});

google.maps.event.addDomListener(this.addLocationText_, 'mouseout' ,function() { 


		that.addLocationText_.style.fontWeight = "normal";	
		that.getLocationNameIcon_.src = getLocationNameIcon_N;		
		that.addLocationText_.style.color = "black";
		hideHoverTip();

});
this.editField_  = document.createElement('DIV');
this.editField_.setAttribute("class", "editFieldHover showForGridView");

this.editLocationIcon_ = document.createElement('img');
this.editLocationIcon_.setAttribute("class",'editPencilHover');
this.editLocationIcon_.src = editLocationPencil_N;

this.editText_ = document.createElement('DIV');
this.editText_.setAttribute("class", "editTextHover");
this.editText_.innerHTML = editText; 
  
this.editToolTip_ = editLocationToolTip;

this.editField_.appendChild(this.editLocationIcon_);
this.editField_.appendChild(this.editText_);

google.maps.event.addDomListener(this.editField_, 'click' ,function(e) { 
	if (that.visible_ ) {
            that.showEditClusterView();
        }
		e.stopPropagation();
});

google.maps.event.addDomListener(this.editField_, 'dblclick' ,function(e) { 

		e.stopPropagation();
});



google.maps.event.addDomListener(this.editField_, 'mouseover' ,function(e) { 


		var targetEle = e.target; //which triggered the event 
		
		var relatedTargetEle = e.relatedTarget;   //the element being exited
		while (relatedTargetEle && relatedTargetEle.nodeName != 'BODY') {
			if (relatedTargetEle == targetEle || relatedTargetEle == that.editField_) return;     //either mouse is entering a child element or a sibling of icon main div.
			relatedTargetEle = relatedTargetEle.parentNode;
		}
		
that.editText_.style.color = "white";
that.editLocationIcon_.src = editLocationPencil_H;
showHoverTip(editLocationToolTip);
});

google.maps.event.addDomListener(this.editField_, 'mouseout' ,function(e) { 

	var targetEle = e.target; //which triggered the event 
		
		var relatedTargetEle = e.relatedTarget;   //the element to which mouse is going to
		while (relatedTargetEle && relatedTargetEle.nodeName != 'BODY') {
			if (relatedTargetEle == targetEle || relatedTargetEle == that.editField_) return;     //either mouse is entering a child element or a sibling of icon main div.
			relatedTargetEle = relatedTargetEle.parentNode;
		}
	


that.editText_.style.color = "black";
that.editLocationIcon_.src = editLocationPencil_N;
hideHoverTip();
});



this.thumbnailHoverDiv_ = document.createElement('DIV');
this.thumbnailHoverDiv_.setAttribute("class", "thumbnailHoverDiv thumbnailHoverDivGrid");

this.thumbnailHover_ = document.createElement('img');
this.thumbnailHover_.setAttribute("class", "thumbnailHover");

this.thumbnailHoverDiv_.appendChild(this.thumbnailHover_);


this.navigateLeft_ = document.createElement('img');
this.navigateLeft_.src = navigateLeftIcon;
this.navigateLeft_.setAttribute("class", "navigateLeftHover navigateHoverGrid");


this.thumbnailHover_.onerror = function(){
  // image not found or change src like this as default image:

   this.src = noImageAvailableHoverUrl;
};


//if(this.sums_ == 1)
{
	this.navigateLeft_.style.visibility = 'hidden';
}

this.navigateRight_ = document.createElement('img');
this.navigateRight_.src = navigateRightIcon;
this.navigateRight_.setAttribute("class", "navigateRightHover navigateHoverGrid");



google.maps.event.addDomListener(this.navigateLeft_, 'click' ,function(e) { 
that.currentHoverIndex--;
if(that.currentHoverIndex == 0)
	that.navigateLeft_.style.visibility = 'hidden';

that.navigateRight_.style.visibility = 'visible';
var imgSrc = getHoverThumbnail(that.cluster_.markers_[that.currentHoverIndex].id) ;
that.views_["hover"].getElementsByClassName("thumbnailHover")[0].src =   imgSrc;
that.thumbnailHover.src =  getHoverThumbnail(that.cluster_.markers_[that.currentHoverIndex].id);
//that.maintainAspectRatio();
  e.stopPropagation();

});

google.maps.event.addDomListener(this.navigateLeft_, 'dblclick' ,function(e) { 
 e.stopPropagation();

});


google.maps.event.addDomListener(this.navigateRight_, 'click' ,function(e) { 
that.currentHoverIndex++;
if(that.currentHoverIndex ==parseInt(that.sums_.text , 10) - 1)
	that.navigateRight_.style.visibility = 'hidden';

	that.navigateLeft_.style.visibility = 'visible';
	var imgSrc = getHoverThumbnail(that.cluster_.markers_[that.currentHoverIndex].id);

		var limit =  Math.min(that.currentHoverIndex + prefetchValue, parseInt(that.sums_.text , 10) - 1);
		for(var i=that.currentHoverIndex + 1; i<= limit; i++)
		{
				getHoverThumbnail(that.cluster_.markers_[i].id);
		}


that.views_["hover"].getElementsByClassName("thumbnailHover")[0].src =   imgSrc;
  e.stopPropagation();

});

google.maps.event.addDomListener(this.navigateRight_, 'dblclick' ,function(e) { 
 e.stopPropagation();

});

this.trashField_  = document.createElement('DIV');
this.trashField_.setAttribute("class", "trashFieldHover");

this.trashIcon_ = document.createElement('img');
this.trashIcon_.src = trash_N;
this.trashIcon_.setAttribute("class", "trashIconHover");

this.trashField_.appendChild(this.trashIcon_);


google.maps.event.addDomListener(this.trashField_, 'click' ,function(e) { 
		if (that.visible_) {
            that.trashIconClick();
        }
		e.stopPropagation();
});
google.maps.event.addDomListener(this.trashField_, 'dlbclick' ,function(e) { 
		e.stopPropagation();
});

google.maps.event.addDomListener(this.trashField_, 'mouseover' ,function() { 
		that.trashIcon_.src = trash_H;
});

google.maps.event.addDomListener(this.trashField_, 'mouseout' ,function() { 
		that.trashIcon_.src = trash_N;
});

this.seeAllField_  = document.createElement('DIV');
this.seeAllField_.setAttribute("class", "seeAllFieldHover showForGridView");

this.seeAllText_ = document.createElement('DIV');
this.seeAllText_.setAttribute("class", "seeAllTextHover");
this.seeAllText_.innerHTML = seeAllText;


this.seeAllIcon_ = document.createElement('img');
this.seeAllIcon_.src = seeAll_N;
this.seeAllIcon_.setAttribute("class", "seeAllIconHover");

this.seeAllField_.appendChild(this.seeAllText_);
this.seeAllField_.appendChild(this.seeAllIcon_);

google.maps.event.addDomListener(this.seeAllField_, 'click' ,function(e) { 

		if (that.visible_) {
            that.seeAllClick();
        }
		e.stopPropagation();
});

google.maps.event.addDomListener(this.seeAllField_, 'dlbclick' ,function(e) { 
e.stopPropagation();
});

google.maps.event.addDomListener(this.seeAllField_, 'mouseover' ,function() { 
		that.seeAllIcon_.src = seeAll_H;
		that.seeAllText_.style.color = "white";
});
google.maps.event.addDomListener(this.seeAllField_, 'mouseout' ,function() { 
		that.seeAllIcon_.src = seeAll_N;
		that.seeAllText_.style.color = "black";
});

this.hoverHTML_.appendChild(this.addLocationField_);
this.hoverHTML_.appendChild(this.editField_);
this.hoverHTML_.appendChild(this.thumbnailHoverDiv_);
this.hoverHTML_.appendChild(this.navigateLeft_);
this.hoverHTML_.appendChild(this.navigateRight_);
this.hoverHTML_.appendChild(this.trashField_);
this.hoverHTML_.appendChild(this.seeAllField_);
return this.hoverHTML_;
}
/**
 * Initializes cluster icon UI for different icon state.
 */
ClusterIcon.prototype.initialize = function() {

    //small view
    var normalMainDiv = document.createElement('DIV');
    
	var thumbDiv = document.createElement('DIV');
    thumbDiv.className = "smallThumbNormal smallThumb";

	var normalThumbDiv = document.createElement('DIV');
    normalThumbDiv.className = "normalThumbDiv";
	
	
	var normalImageDiv = document.createElement('img');
	normalImageDiv.className = "normalImageDiv";

    
    var textSumDiv = document.createElement('DIV');
    textSumDiv.setAttribute("class", "smallTextSums smallTextSumsNormal");
    textSumDiv.setAttribute("id", "textSum");
	
	normalThumbDiv.appendChild(normalImageDiv);
	thumbDiv.appendChild(normalThumbDiv);
	
    normalMainDiv.appendChild(thumbDiv);
    normalMainDiv.appendChild(textSumDiv);



	normalImageDiv.onerror = function(){
	// image not found or change src like this as default image:
	this.src = noImageAvailableNormalUrl;
	};

	

    
    this.views_["small"] = normalMainDiv;
    this.div_.appendChild(this.views_["small"]);
    
    
	

	
	this.textDiv_ = document.createElement('DIV'); 
	this.textDiv_.style.position = "absolute"; 
	this.textDiv_.style.whiteSpace = "nowrap";
	this.textDiv_.style.color = "black";
	this.textDiv_.style.fontSize = "13px";
	this.textDiv_.style.fontFamily = "Tahoma, Geneva, sans-serif";			//TODO: need to localize it for japane
	this.textDiv_.style.verticalAlign = "center";
	this.textDiv_.style.display = '';

	    //Hover view
    this.views_["hover"] =  this.createHoverView();
    this.div_.appendChild(this.views_["hover"]);

	
	this.updateHoverViewForDialogGrid();
	 
};
ClusterIcon.prototype.updateLocationName = function(locationName)
{
	if(locationName != "") {

		var widthOfLine = widthOfLocationText;
		//if it can come in single line
		var len = this.getVisualLength(locationName);
	
		
		this.addLocationToolTip_ = locationName;
	
		if(len < (widthOfLine-4))
		{
			this.addLocationText_.style.paddingTop = "8px";			
			this.addLocationText_.innerHTML = locationName;
		}
		else
		{
			this.addLocationText_.style.paddingTop = "0px";
			this.addLocationText_.innerHTML =   this.getEllipsisString(locationName,widthOfLine);			
		}
		this.addLocationText_.style.cursor = "default";
		this.hasValidLocationName_ = true;
		this.getLocationNameIcon_.style.visibility = "hidden";
		this.addLocationText_.style.paddingLeft = "5px";


	}
}




ClusterIcon.prototype.getEllipsisString = function(locationName, widthOfLine)
{

	var arr = locationName.split(" ");
	var lineLength = 0;
	var firstLineTillIndex;	
	var spacingLength = 4;
	for(firstLineTillIndex=0; firstLineTillIndex < arr.length; firstLineTillIndex++)
	{
		var len = this.getVisualLength(arr[firstLineTillIndex]);
		//honor spacing
		lineLength = lineLength + len + spacingLength;  
		if(lineLength > widthOfLine)
			break;
	}
	if(firstLineTillIndex == arr.length && lineLength <= widthOfLine)
	{
		//this should never be true cause its handling has been done separately
		return locationName;
	}
	else
	{

			//case when first line will be shorter to fill the text
			
			//make sure the first line should be filled with text always
			if(firstLineTillIndex  == 0)
			{ 
				//change the text to ... in the end of first word.
				var firstStr = arr[0];
				var firstStrLength = firstStr.length;
				
				var truncatedString = firstStr[0];
				var index = 1;
				while(this.getVisualLength(truncatedString) < widthOfLine && index < firstStrLength)
				{
					truncatedString = truncatedString + firstStr[index];
					index++;
				}
				
				//go till index-3 for rest fill it with ...
				
				arr[0] = truncatedString.substring(0,truncatedString.length - 3);
				arr[0] = arr[0] +  "....";
				firstLineTillIndex = 1;
			}
			var firstLine = arr[0];
			for(var i = 1; i < firstLineTillIndex;i++)
			{
				firstLine = firstLine + " " + arr[i];
			}
			var secondLine = "";
			if(typeof(arr[firstLineTillIndex]) != 'undefined')
			 secondLine = arr[firstLineTillIndex];
			for(var i = firstLineTillIndex + 1 ; i < arr.length;i++)
			{
				secondLine = secondLine + " " + arr[i];
			}
			
			var finalAns = "";
			
			if(this.getVisualLength(secondLine) < widthOfLine )
			{
				finalAns = firstLine + " " + secondLine;
			}
			else
			{
			
				var str = arr[secondLineTillIndex];
				var dots = "...";
				finalAns = str + dots;
				
				var secondLineTillIndex = firstLineTillIndex;	
				
				while(this.getVisualLength(finalAns) < widthOfLine && secondLineTillIndex < arr.length)
				{					
					secondLineTillIndex++;
					str = str + " "  + arr[secondLineTillIndex];
					finalAns = str + dots;
				}

				
				if(secondLineTillIndex == arr.length && this.getVisualLength(finalAns)  <= widthOfLine)
				{
					//this case should never happen

					finalAns = firstLine +  " " + secondLine;
		
					
				}
				else
				{
					//take index-1 and append ...
					var secondLine = arr[firstLineTillIndex];
					for(var i = firstLineTillIndex + 1; i < secondLineTillIndex;i++)
					{
						secondLine = secondLine + " " + arr[i];
					}				
					
					secondLine = secondLine + dots;
					
					finalAns = firstLine +   " " + secondLine;
				}
				
		}
		
		return finalAns;
				

	}
}

ClusterIcon.prototype.updateHoverViewForDialogGrid = function()
{
	if(this.cluster_ && this.cluster_.getMarkerClusterer() && this.cluster_.getMarkerClusterer().getEnableHoverIcons())
	{

			// dheekuma: this loop will be getting shortened as we are replacing this class name with showForGridView
			var s = this.div_.getElementsByClassName("hideForDialogView");
			while(s.length)
			{
				var classN = s[0].className;
				this.div_.getElementsByClassName("hideForDialogView")[0].className =   classN.replace( /(?:^|\s)hideForDialogView(?!\S)/g , ' showForGridView' );
			}

			s = this.div_.getElementsByClassName("hovered")[0].className;
			if(s.indexOf("backgroundHoverDiaog") > -1)
			{
				this.div_.getElementsByClassName("hovered")[0].className =   this.div_.getElementsByClassName("hovered")[0].className.replace( /(?:^|\s)backgroundHoverDiaog(?!\S)/g , '' );
				this.div_.getElementsByClassName("hovered")[0].className += " backgroundHoverGrid"; 
				this.div_.getElementsByClassName("hovered")[0].style.backgroundImage = "url(" + backgroundHoverGrid + ")"; 

	
			}
			
		
			var r = this.div_.getElementsByClassName("thumbnailHoverDiv")[0].className;
			if(r.indexOf("thumbnailHoverDivDialog") > -1)
			{
				this.div_.getElementsByClassName("thumbnailHoverDiv")[0].className =   this.div_.getElementsByClassName("thumbnailHoverDiv")[0].className.replace( /(?:^|\s)thumbnailHoverDivDialog(?!\S)/g , '' );
				this.div_.getElementsByClassName("thumbnailHoverDiv")[0].className += " thumbnailHoverDivGrid"; 
			}
		
		
			r = this.div_.getElementsByClassName("navigateRightHover")[0].className;
			if(r.indexOf("navigateHoverDialog") > -1)
			{
				this.div_.getElementsByClassName("navigateRightHover")[0].className =   this.div_.getElementsByClassName("navigateRightHover")[0].className.replace( /(?:^|\s)navigateHoverDialog(?!\S)/g , '' );
				this.div_.getElementsByClassName("navigateRightHover")[0].className += " navigateHoverGrid"; 
			}
		
			r = this.div_.getElementsByClassName("navigateLeftHover")[0].className;
			if(r.indexOf("navigateHoverDialog") > -1)
			{
				this.div_.getElementsByClassName("navigateLeftHover")[0].className =   this.div_.getElementsByClassName("navigateLeftHover")[0].className.replace( /(?:^|\s)navigateHoverDialog(?!\S)/g , '' );
				this.div_.getElementsByClassName("navigateLeftHover")[0].className += " navigateHoverGrid"; 
			}

	}
	else 
	{
			var s = this.div_.getElementsByClassName("showForGridView");
			while(s.length)
			{
				var classN = s[0].className;
				this.div_.getElementsByClassName("showForGridView")[0].className =   classN.replace( /(?:^|\s)showForGridView(?!\S)/g , ' hideForDialogView' );
			}
			
			s = this.div_.getElementsByClassName("hovered")[0].className;
			if(s.indexOf("backgroundHoverGrid") > -1)
			{
				this.div_.getElementsByClassName("hovered")[0].className =   this.div_.getElementsByClassName("hovered")[0].className.replace( /(?:^|\s)backgroundHoverGrid(?!\S)/g , '' );
				this.div_.getElementsByClassName("hovered")[0].className += " backgroundHoverDiaog"; 
				this.div_.getElementsByClassName("hovered")[0].style.backgroundImage = "url(" + backgroundHoverDiaog + ")"; 

			}
			
			var r = this.div_.getElementsByClassName("thumbnailHoverDiv")[0].className;

			if(r.indexOf("thumbnailHoverDivGrid") > -1)
			{
				this.div_.getElementsByClassName("thumbnailHoverDiv")[0].className =   this.div_.getElementsByClassName("thumbnailHoverDiv")[0].className.replace( /(?:^|\s)thumbnailHoverDivGrid(?!\S)/g , '' );
				this.div_.getElementsByClassName("thumbnailHoverDiv")[0].className += " thumbnailHoverDivDialog"; 
	
			}
			
			r = this.div_.getElementsByClassName("navigateRightHover")[0].className;
			if(r.indexOf("navigateHoverGrid") > -1)
			{
				this.div_.getElementsByClassName("navigateRightHover")[0].className =   this.div_.getElementsByClassName("navigateRightHover")[0].className.replace( /(?:^|\s)navigateHoverGrid(?!\S)/g , '' );
				this.div_.getElementsByClassName("navigateRightHover")[0].className += " navigateHoverDialog"; 
			}
		
			r = this.div_.getElementsByClassName("navigateLeftHover")[0].className;
			if(r.indexOf("navigateHoverGrid") > -1)
			{
				this.div_.getElementsByClassName("navigateLeftHover")[0].className =   this.div_.getElementsByClassName("navigateLeftHover")[0].className.replace( /(?:^|\s)navigateHoverGrid(?!\S)/g , '' );
				this.div_.getElementsByClassName("navigateLeftHover")[0].className += " navigateHoverDialog"; 
			}

			
			
	}

}


/**
 * Show the icon and sets it's position. Icon ui will change as per it's new state.
 */
ClusterIcon.prototype.showIcon = function(state) {
	var leftLongitude = map.getBounds()._sw.lng;
	var rightLongitude = map.getBounds()._ne.lng;

	if(this.center_.lng < leftLongitude)
		this.center_.lng += 360;

	if(this.center_.lng > rightLongitude)
		this.center_.lng -= 360;

    var newstate = (typeof state !== 'undefined') ? state : this.current_state_;
    
    //Show current state div, while hiding rest of states
    for (i = 0; i < this.states_.length; i++) {
        if(this.states_[i] == newstate) {
            this.views_[newstate].style.display = '';

			if(newstate == "small")
			{
				//Change here to pick default cluster icon
				
				this.views_[newstate].getElementsByClassName("normalImageDiv")[0].src =  getNormalThumbnail(this.cluster_.markers_[0].id);
				this.views_[newstate].getElementsByClassName("smallTextSums")[0].innerHTML =  this.sums_.text.toString();
					
				this.setIconSelectionUIState_(this.selected_);
				
				if(this.cluster_.isClusterDragged()) {
					this.views_[newstate].getElementsByClassName("smallThumb")[0].style.zIndex = 10;
				} else {
					this.views_[newstate].getElementsByClassName("smallThumb")[0].style.zIndex = -1;
				}
				
				if(this.currentHoverIndex == 0 && typeof(hoverThumbMap[this.cluster_.markers_[0].id]) == 'undefined')
				{
					getHoverThumbnail(this.cluster_.markers_[this.currentHoverIndex].id) ;
				}
			
			}
			else if(newstate == "hover")
			{
				
				if(parseInt(this.sums_.text, 10) == 1)
				{
					this.navigateRight_.style.visibility = 'hidden'; 
				}
				var imgSrc =  getHoverThumbnail(this.cluster_.markers_[this.currentHoverIndex].id) ;
				var limit =  Math.min(this.currentHoverIndex + prefetchValue, parseInt(this.sums_.text , 10) - 1);
				for(var i=this.currentHoverIndex  + 1; i<= limit; i++)
				{
					getHoverThumbnail(this.cluster_.markers_[i].id);
				}

				this.views_[newstate].getElementsByClassName("thumbnailHover")[0].src =   imgSrc;
			

				this.views_[newstate].getElementsByClassName("seeAllTextHover")[0].innerHTML =  this.sums_.text.toString();

			
				//this.maintainAspectRatio();
			}
		}
        
		else
            this.views_[this.states_[i]].style.display = 'none';
    }

    //Change icon position on map and show
    var pos = this.getPosFromLatLng_(this.center_);
    
    var style = [];
	if(newstate == "hover")
	{		
		style.push('top:' + pos.y + 'px; left:' +
               pos.x + 'px; position:absolute;z-index: 60;');
	}
    else
	{
		style.push('top:' + pos.y + 'px; left:' +
               pos.x + 'px; position:absolute;z-index: 1;');
    }
    this.div_.style.cssText = style.join('');
    this.div_.style.display = '';
    
    this.current_state_ = newstate;

    // fix for bug EO-4196270 that the dragged cluster is going behind the other cluster sometimes as all of them have z-index = 1 
    if(draggedPin == this.cluster_)
    	this.div_.style.zIndex = "100";
};

/**
 * Triggers the editClusterView event.
 */
ClusterIcon.prototype.showEditClusterView = function() {

	var markerClusterer = this.cluster_.getMarkerClusterer();
	markerClusterer.trigger( 'editClusterClicked', [this.cluster_]);
	
};

/**
 * Triggers the requestReverseGeocodeLocation event.
 */
ClusterIcon.prototype.getLocationOfCluster = function() {

	var markerClusterer = this.cluster_.getMarkerClusterer();
	markerClusterer.trigger( 'requestReverseGeocodeLocation', [this.cluster_]);
	
};

/**
 * Deletes the markers GPS tag */
ClusterIcon.prototype.trashIconClick = function() {
	var markerClusterer = this.cluster_.getMarkerClusterer();
	markerClusterer.trigger( 'trashIconClicked', [this.cluster_]);
}



/**
 * Shows the See All view in grid and zooms if the option is set.
 */
ClusterIcon.prototype.seeAllClick = function() {
	
	var markerClusterer = this.cluster_.getMarkerClusterer();
	markerClusterer.trigger( 'seeAllClicked', [this.cluster_]);
  
  if (markerClusterer.isZoomOnClick()) {
    // Center the map on this cluster.
    this.map_.panTo(this.cluster_.getCenter());

    // Zoom into the cluster.
    this.map_.fitBounds(this.cluster_.getBounds());
  }
	
}


/**
 * Triggers the clusterclick event and zooms if the option is set.
 */
ClusterIcon.prototype.triggerClusterClick = function() {
  var markerClusterer = this.cluster_.getMarkerClusterer();

  // Trigger the clusterclick event.
  markerClusterer.trigger( 'clusterclick', [this.cluster_]);

  
  if (markerClusterer.isZoomOnClick()) {
    // Center the map on this cluster.
    this.map_.panTo(this.cluster_.getCenter());

    // Zoom into the cluster.
    this.map_.fitBounds(this.cluster_.getBounds());
  }
};



ClusterIcon.prototype.maintainAspectRatio = function(){
	var imgTemp = new Image();
	imgTemp.src = this.views_["hover"].getElementsByClassName("thumbnailHover")[0].src;
	var thatt = this;
	imgTemp.onload = function()
	{
		var newHeight = 0;
		var newWidth  = 0;
		var imageHt = this.naturalHeight;
		var imageWd = this.naturalWidth;
		
		if(((imageHt)/(imageWd))*180 > 110)
		{
			newWidth = imageWd * 110 / imageHt;
			newHeight = 110;
		}
		else
		{
			newHeight = imageHt * 180 / imageWd;
			newWidth = 180;
			
		}

//		alert(imageHt + " and " + imageWd);
		
			thatt.views_["hover"].getElementsByClassName("thumbnailHover")[0].setAttribute("width", newWidth);
			thatt.views_["hover"].getElementsByClassName("thumbnailHover")[0].setAttribute("height", newHeight);
		
	}
	
}

/**
 * Triggers the clusterdoubleclick event.
 */
ClusterIcon.prototype.triggerClusterDoubleClick = function() {
  var markerClusterer = this.cluster_.getMarkerClusterer();

  markerClusterer.trigger( 'clusterdoubleclick', [this.cluster_]);
};

/**
 * Triggers the clusterrightclick event
 */
ClusterIcon.prototype.triggerClusterRightClick = function() {
  var markerClusterer = this.cluster_.getMarkerClusterer();

  // Trigger the clusterclick event.
  markerClusterer.trigger( 'clusterrightclick', [this.cluster_]);
};

/**
 * Triggers the clustermouseover event
 */
ClusterIcon.prototype.triggerClusterMouseover = function() {
	var markerClusterer = this.cluster_.getMarkerClusterer();
	
	// Trigger the clustermouseover event.
	markerClusterer.trigger( 'clustermouseover', [this.cluster_]);
};


/**
* Triggers the clustermousedown event
*/
ClusterIcon.prototype.triggerClusterMouseDown = function(event) {
	if(!this.locked_)
		this.cluster_.clusterDragStart(event);
	var markerClusterer = this.cluster_.getMarkerClusterer();
	 // Trigger the clustermousedown event.
    markerClusterer.trigger( 'clustermousedown', event, [this.cluster_]);
};


/**
* Triggers the clustermouseup event
*/
ClusterIcon.prototype.triggerClusterMouseUp = function() {
	if(this.cluster_.isClusterDragged())
		this.cluster_.clusterDragEnd();    
	else
		this.cluster_.cancelDrag();    //clear any drag operation
	
    var markerClusterer = this.cluster_.getMarkerClusterer();

    // Trigger the clustermouseup event.
    markerClusterer.trigger( 'clustermouseup', [this.cluster_]);
};

/**
 * Triggers the clustermouseout event
 */
ClusterIcon.prototype.triggerClusterMouseMove = function() {
	var markerClusterer = this.cluster_.getMarkerClusterer();
	// Trigger the clustermousemove event.
	markerClusterer.trigger( 'clustermousemove', [this.cluster_]);
};

/**
 * Triggers the clustermouseout event
 */
ClusterIcon.prototype.triggerClusterMouseout = function() {
	var markerClusterer = this.cluster_.getMarkerClusterer();
	// Trigger the clustermouseout event.
	markerClusterer.trigger( 'clustermouseout', [this.cluster_]);
};

/**
 * Adding the cluster icon to the dom.
 * @ignore
 */
ClusterIcon.prototype.onAdd = function() {
	
	this.map_.getCanvasContainer().appendChild(this.div_);
	this.map_.getCanvasContainer().appendChild(this.textDiv_);

	this.draw = this.draw.bind(this);
    this.map_.on('move', this.draw);
    this.map_.on('moveend', this.draw);
	
	
    var that = this;
    google.maps.event.addDomListener(this.div_, 'click', function(e) {
	
	that.triggerClusterClick();
	
    });

    google.maps.event.addDomListener(this.div_, 'dblclick', function(e) {

        // Prevent map zoom when double-clicking on a cluster:
        e.cancelBubble = true;
        if (e.stopPropagation) {
            e.stopPropagation();
        }

        if (that.visible_) {
            that.triggerClusterDoubleClick();
        }
    });

    google.maps.event.addDomListener(this.div_, 'contextmenu', function() {
        if (that.visible_) {
            that.triggerClusterRightClick();
        }
    });

    google.maps.event.addDomListener(this.div_, 'mouseover', function(e) {
		if(that.cluster_.isClusterDragged()) return;    //nothing to do when cluster dragged
		
		//if any cluster dragged then we don't need to show over view on other clusters
		var markerClusterer = that.cluster_.getMarkerClusterer();
		var draggedCluster = markerClusterer.getDraggedClusterIfAny();
		if(draggedCluster != null && typeof(draggedCluster) != 'undefined' )  return; 

		var targetEle = e.target;
		
		var relatedTargetEle = e.relatedTarget;   //the element to which mouse is coming from
		while (relatedTargetEle && relatedTargetEle.nodeName != 'BODY') {
			if (relatedTargetEle == targetEle || relatedTargetEle == that.div_) return;     //either mouse is coming from a child element or a sibling of icon main div.
			relatedTargetEle = relatedTargetEle.parentNode;	
		}

		// Request Location name, if we don't already have.
		if(!that.cluster_.hasLocationNameForCluster)
		{
			var markerClusterer = that.cluster_.getMarkerClusterer();
			markerClusterer.trigger('getLocationNameForClusterViaMediaList', [that.cluster_]);
		}
		
		that.hoverTimeOut_ = setTimeout(function(){ 
			if (that.visible_) {
				that.showOverIcon_();
				that.cluster_.putClusterInBounds();

				that.triggerClusterMouseover();
						
				that.div_.style.zIndex = 55;

			} 
		}, 500);
        
    });

	
    google.maps.event.addDomListener(this.div_, 'mouseout', function(e) {
		var targetEle = e.target;
		
		var relatedTargetEle = e.relatedTarget;   //the element to which mouse is going to
		while (relatedTargetEle && relatedTargetEle.nodeName != 'BODY') {
			if (relatedTargetEle == targetEle || relatedTargetEle == that.div_) return;     //either mouse is entering a child element or a sibling of icon main div.
			relatedTargetEle = relatedTargetEle.parentNode;
		}
		
		//alert("clusterIcon: mouse out");
		if(that.hoverTimeOut_) {
			clearTimeout(that.hoverTimeOut_);
			that.hoverTimeOut_ = null;
		}
        if (that.visible_) {
            that.showSmallIcon_();

            that.triggerClusterMouseout();
			that.div_.style.zIndex = 1;
	
        }
    });


    google.maps.event.addDomListener(this.div_, 'mousedown', function(event) {
		if(that.hoverTimeOut_) {
			clearTimeout(that.hoverTimeOut_);
			that.hoverTimeOut_ = null;
		}
        if (that.visible_) {
            if (typeof event.stopPropagation != "undefined") {
                event.stopPropagation();
            } else {
                event.cancelBubble = true;
            }
			if( event.button == 0)  // denotes the left click
				that.triggerClusterMouseDown(event);	
        }
    });

	google.maps.event.addDomListener(this.div_, 'mousemove', function(evt) {
        if (that.visible_ ) {
			if(that.cluster_.getDragState() == eDragStates.DRAGSTART)  //we want to avoid map mouse move events when mouse is moving over cluster only and that cluster can be dragged.
				evt.stopPropagation();			
            that.triggerClusterMouseMove();
        }
    });

	
    google.maps.event.addDomListener(this.div_, 'mouseup', function() {
        if (that.visible_) {

            that.triggerClusterMouseUp();
        }
    });


	
	//if it can come in single line
	var len = this.getVisualLength(Cluster.editString_);

	if(len>editWidth)
		this.editText_.style.visibility = "hidden";

};


/**
 * Returns the position to place the div dending on the latlng.
 *
 * @param {google.maps.LatLng} latlng The position in latlng.
 * @return {google.maps.Point} The position in pixels.
 * @private
 */
ClusterIcon.prototype.getPosFromLatLng_ = function(latlng) {
  var pos = this.map_.project(latlng);
  pos.x -= parseInt(this.width_ / 2, 10);
  pos.y -= parseInt(this.height_ , 10);
  if ( this.offset_y )
	pos.y += this.offset_y;
  return pos;
};

/**
 * Draw the icon.
 * @ignore
 */
ClusterIcon.prototype.draw = function() {
  if (this.visible_) {
    this.showIcon();
  }
};


/**
 * Hide the icon.
 */
ClusterIcon.prototype.hide = function() {
  if (this.div_) {
    this.div_.style.display = 'none';
  }
	  
  this.visible_ = false;
};


/**
 * Position and show the icon.
 */
ClusterIcon.prototype.show = function() {
  if (this.div_) {
    //var pos = this.getPosFromLatLng_(this.center_);
    //this.div_.style.cssText = this.createCss(pos);
    //this.div_.style.display = '';
      
    this.showIcon();
  }
  this.visible_ = true;
};


/**
 * Remove the icon from the map
 */
ClusterIcon.prototype.remove = function() {
  this.visible_ = false;
  //this.setMap(null);  
  this.onRemove();
};


/**
 * Implementation of the onRemove interface.
 * @ignore
 */
ClusterIcon.prototype.onRemove = function() {
  if (this.div_ && this.div_.parentNode) {
    this.hide();
    this.div_.parentNode.removeChild(this.div_);
    this.div_ = null;
  }

  this.map_.off('move', this.draw);
  this.map_.off('moveend', this.draw);
};


/**
 * Set the sums of the icon.
 *
 * @param {Object} sums The sums containing:
 *   'text': (string) The text to display in the icon.
 *   'index': (number) The style index of the icon (0-based offset).
 */
ClusterIcon.prototype.setSums = function(sums) {
  this.sums_ = sums;
  this.useStyle();
};

ClusterIcon.prototype.getVisualLength = function(text) {
			this.textDiv_.style.display = '';
			this.textDiv_.innerHTML = text;
			//var panes = this.getPanes();  
			//panes.overlayImage.appendChild(this.textDiv_);			
			var len = (this.textDiv_.clientWidth);
			this.textDiv_.style.display = 'none';
			return len;			

}



/**
 * Set the lock/unlock the icon.
 *
 * @param {boolean} lock The lock flag
 */
ClusterIcon.prototype.setLocked = function(locked) {
	this.locked_ = locked;
};

/**
 * Set the selection state.
 *
 * @param {boolean} selected The selection flag
 */
ClusterIcon.prototype.setSelected = function(selected) {
	this.selected_ = selected;
	
};

/**
 * Show the small icon.
 * @ignore
 */
ClusterIcon.prototype.showSmallIcon_ = function() {
	if (this.visible_) {
		this.showIcon("small");
	}
};

/**
 * Show the rollover icon.
 * @ignore
 */
ClusterIcon.prototype.showOverIcon_ = function() {
	if (this.visible_) {
		this.showIcon("hover");
	}
};

/**
 * Show the drop icon.
 * @ignore
 */
ClusterIcon.prototype.showDropIcon_ = function() {
	if (this.visible_) {
		if(this.current_state_ == "small")  //else something is wrong
		{
				
			this.setIconSelectionUIState_(true);
			
		}	
	}
};

/**
 * Switches between selected and normal state of icon. This is for UI only , selected UI doesn't necessarily mean selected cluster
 * @ignore
 */
ClusterIcon.prototype.setIconSelectionUIState_ = function(bSelected) {
	if(bSelected)
	{
		this.views_["small"].getElementsByClassName("smallThumb")[0].className =   this.views_["small"].getElementsByClassName("smallThumb")[0].className.replace( /(?:^|\s)smallThumbNormal(?!\S)/g , '' );
		this.views_["small"].getElementsByClassName("smallThumb")[0].className += " smallThumbSelected"; 
		this.views_["small"].getElementsByClassName("smallThumb")[0].style.backgroundImage = "url(" + selectedImageBackground + ")"; 
		
		this.views_["small"].getElementsByClassName("smallTextSums")[0].className =   this.views_["small"].getElementsByClassName("smallTextSums")[0].className.replace( /(?:^|\s)smallTextSumsNormal(?!\S)/g , '' );
		this.views_["small"].getElementsByClassName("smallTextSums")[0].className += " smallTextSumsSelected"; 
		
	}
	else
	{
		this.views_["small"].getElementsByClassName("smallThumb")[0].className =   this.views_["small"].getElementsByClassName("smallThumb")[0].className.replace( /(?:^|\s)smallThumbSelected(?!\S)/g , '' );
		this.views_["small"].getElementsByClassName("smallThumb")[0].className += " smallThumbNormal"; 
		this.views_["small"].getElementsByClassName("smallThumb")[0].style.backgroundImage = "url(" + normalImageBackground + ")"; 
		
		this.views_["small"].getElementsByClassName("smallTextSums")[0].className =   this.views_["small"].getElementsByClassName("smallTextSums")[0].className.replace( /(?:^|\s)smallTextSumsSelected(?!\S)/g , '' );
		this.views_["small"].getElementsByClassName("smallTextSums")[0].className += " smallTextSumsNormal"; 
		
	}
};

/**
 * Sets the icon to the the styles.
 */
ClusterIcon.prototype.useStyle = function() {
  var index = Math.max(0, this.sums_.index);
  index = Math.min(this.styles_.length - 1, index);
  var style = this.styles_[index];
  this.url_ = style.url;
  this.url_normal_ = style.url_normal;
  this.url_over_ = style.url_over;
  this.url_selected_ = style.url_selected;
  this.height_ = style.height;
  this.width_ = style.width;
  this.textColor_ = style.opt_textColor;
  this.textColorHighlight_ = style.opt_textColorHighlight;
  this.anchor_ = style.opt_anchor;
  this.textSize_ = style.opt_textSize;
  this.offset_y = style.opt_offset_y;
};


/**
 * Sets the center of the icon.
 *
 * @param {google.maps.LatLng} center The latlng to set as the center.
 */
ClusterIcon.prototype.setCenter = function(center) {
  this.center_ = center;
};


/**
 * Create the css text based on the position of the icon.
 *
 * @param {google.maps.Point} pos The position.
 * @return {string} The css style text.
 */
ClusterIcon.prototype.createCss = function(pos, url) {
  var style = [];

  /*if ((typeof(url) == 'undefined' || url == null)
		&& (this.selected_ == true || this.attention_ == true)) {
	url = this.url_selected_;
  }

  if (typeof(url) == 'undefined' || url == null)
   style.push('background:url(' + this.url_ + ');');
  else
   style.push('background:url(' + url + ');');

  if (typeof this.anchor_ === 'object') {
    if (typeof this.anchor_[0] === 'number' && this.anchor_[0] > 0 &&
        this.anchor_[0] < this.height_) {
      style.push('height:' + (this.height_ - this.anchor_[0]) +
          'px; padding-top:' + this.anchor_[0] + 'px;');
    } else {
      style.push('height:' + this.height_ + 'px; line-height:' + this.height_ +
          'px;');
    }
    if (typeof this.anchor_[1] === 'number' && this.anchor_[1] > 0 &&
        this.anchor_[1] < this.width_) {
      style.push('width:' + (this.width_ - this.anchor_[1]) +
          'px; padding-left:' + this.anchor_[1] + 'px;');
    } else {
      style.push('width:' + this.width_ + 'px; text-align:center;');
    }
  } else {
    style.push('height:' + this.height_ + 'px; line-height:' +
        this.height_ + 'px; width:' + this.width_ + 'px; text-align:center;');
  }*/

  var txtColor = this.textColor_ ? this.textColor_ : 'black';
  var txtSize = this.textSize_ ? this.textSize_ : 11;

  if (typeof(this.locked_) == 'undefined' || !this.locked_) {
    style.push('cursor:pointer;');
  }

  if (this.selected_ == true || this.attention_ == true) {
	txtColor = this.textColorHighlight_ ? this.textColorHighlight_ : 'white';
  }
	
  style.push('top:' + pos.y + 'px; left:' +
			   pos.x + 'px; color:' + txtColor + '; position:absolute; font-size:' +
			   txtSize + 'px; font-family:Tahoma,Arial,sans-serif; font-weight:bold');
	
  return style.join('');
};


// Export Symbols for Closure
// If you are not going to compile with closure then you can remove the
// code below.
window['MarkerClusterer'] = MarkerClusterer;
MarkerClusterer.prototype['addMarker'] = MarkerClusterer.prototype.addMarker;
MarkerClusterer.prototype['clearMarkers'] =
    MarkerClusterer.prototype.clearMarkers;
MarkerClusterer.prototype['canAcceptMarkerDrop'] =
MarkerClusterer.prototype.canAcceptMarkerDrop;
MarkerClusterer.prototype['canAcceptMarkerDropImp'] =
MarkerClusterer.prototype.canAcceptMarkerDropImp;
MarkerClusterer.prototype['getCalculator'] =
    MarkerClusterer.prototype.getCalculator;
MarkerClusterer.prototype['getGridSize'] =
    MarkerClusterer.prototype.getGridSize;
MarkerClusterer.prototype['getMap'] = MarkerClusterer.prototype.getMap;
MarkerClusterer.prototype['getMarkers'] = MarkerClusterer.prototype.getMarkers;
MarkerClusterer.prototype['getMaxZoom'] = MarkerClusterer.prototype.getMaxZoom;
MarkerClusterer.prototype['getStyles'] = MarkerClusterer.prototype.getStyles;
MarkerClusterer.prototype['getTotalClusters'] =
    MarkerClusterer.prototype.getTotalClusters;
MarkerClusterer.prototype['getTotalMarkers'] =
    MarkerClusterer.prototype.getTotalMarkers;
MarkerClusterer.prototype['redraw'] = MarkerClusterer.prototype.redraw;
MarkerClusterer.prototype['removeMarker'] =
    MarkerClusterer.prototype.removeMarker;
MarkerClusterer.prototype['updateClustersAsNeeded'] =
MarkerClusterer.prototype.updateClustersAsNeeded;
MarkerClusterer.prototype['resetViewport'] =
    MarkerClusterer.prototype.resetViewport;
MarkerClusterer.prototype['setCalculator'] =
    MarkerClusterer.prototype.setCalculator;
MarkerClusterer.prototype['setGridSize'] =
    MarkerClusterer.prototype.setGridSize;
MarkerClusterer.prototype['onAdd'] = MarkerClusterer.prototype.onAdd;
MarkerClusterer.prototype['draw'] = MarkerClusterer.prototype.draw;
MarkerClusterer.prototype['idle'] = MarkerClusterer.prototype.idle;
MarkerClusterer.prototype['clusterMoved'] = MarkerClusterer.prototype.clusterMoved;
MarkerClusterer.prototype['addListener'] = MarkerClusterer.prototype.addListener;
MarkerClusterer.prototype['trigger'] = MarkerClusterer.prototype.trigger;
MarkerClusterer.prototype['getDraggedClusterIfAny'] =
MarkerClusterer.prototype.getDraggedClusterIfAny;

ClusterIcon.prototype['onAdd'] = ClusterIcon.prototype.onAdd;
ClusterIcon.prototype['draw'] = ClusterIcon.prototype.draw;
ClusterIcon.prototype['onRemove'] = ClusterIcon.prototype.onRemove;

window['MarkerInfo'] = MarkerInfo;
MarkerInfo.prototype['setLngLat'] = MarkerInfo.prototype.setLngLat;
MarkerInfo.prototype['getLngLat'] = MarkerInfo.prototype.getLngLat;
MarkerInfo.prototype['setVisible'] = MarkerInfo.prototype.setVisible;
MarkerInfo.prototype['getVisible'] = MarkerInfo.prototype.getVisible;
MarkerInfo.prototype['setDraggable'] = MarkerInfo.prototype.setDraggable;
MarkerInfo.prototype['getDraggable'] = MarkerInfo.prototype.getDraggable;
