/**
* 	Premiere Learning Panel: JSX Module
* 	for Premiere Pro v12.1+
*
*	@author: Florian Lange <f.lange@labor.digital>
*	LABOR – Agentur für moderne Kommunikation GmbH, Mainz, Germany
* 	https://labor.digital
*
*/

//"that" is the global "this" for potential object inspection
var that = $.that;

var APIModule = ( function(){

	var mruMaxDocuments = 10,
		mruDocPathKey = "BE.Prefs.MRU.Document.",
		mruDocSizeKey = "BE.Prefs.MRU.Document.Size.",
		mruDocDateKey = "BE.Prefs.MRU.Document.Date.";

	var totalSelectedProjectItems = 0;

	var methods = {

		//STATUS: OK
		isJsonString: function( str ){
			try {
				JSON.parse( str );
			} catch (e) {
				return false;
			}
			return true;
		},

		//STATUS: TESTING
		getAllProps: function( obj ) {
			var props = [];

			do {
				props = props.concat( Object.getOwnPropertyNames( obj ) );
			} while ( obj = Object.getPrototypeOf( obj ) );

			return props;
		},

		//STATUS: TESTING
		goToMarker: function( markerIdentifier, sequenceName ){

			var currentMarker = false;
			var projectSequence = methods.getSequenceByName( sequenceName );

			if (projectSequence) {

				var markers = projectSequence.markers;

				if (markers) {

					var numMarkers	= markers.numMarkers;
					var index = ( typeof markerIdentifier === 'number' ) ? markerIdentifier : numMarkers;

					if (numMarkers > 0) {

						var firstMarker = markers.getFirstMarker();
						var previousMarker = 1;

						if (firstMarker){
							for(var i = 0; i < index; i++){

								if (i === 0){
									currentMarker = firstMarker;
								} else {
									currentMarker = markers.getNextMarker( previousMarker );
								}

								if (currentMarker){
									projectSequence.setPlayerPosition( currentMarker.start.ticks );
									previousMarker = currentMarker;
								}
							}
						}
					}
				}
			}

			return currentMarker;
		},

		/*
			JS can listen to this via:
				glo.environmentVars.csInterface.addEventListener("My Custom Event", function( e ) { var eventData = e.data; });
		*/
		//STATUS: TESTING
		triggerCustomEvent: function( eventName, eventData ){

			var eoName = Folder.fs === 'Macintosh' ?
				"PlugPlugExternalObject" : "PlugPlugExternalObject.dll";

			//lib has to be instanciated to can construct new CSXSEvent
			var mylib = new ExternalObject('lib:' + eoName);

			var eventObj = new CSXSEvent();

			eventObj.type = eventName;
			//todo: transfer of objects isn't working. why?
			//eventObj.data = JSON.stringify( eventData );
			eventObj.data = eventData;

			eventObj.dispatch();
		},

		//STATUS: OK
		dbg: function ( message ) {
			//this will debug out in special Premiere App panel ( like typical notifications )
			app.setSDKEventMessage( message, "info" );
		},

		//STATUS: OK
		getAppVersion: function(){
			return app.version;
		},

		//STATUS: OK
		/*
		* expects path with "/"-style separators
		*/
		getCleanPath: function( path ){
			var separator = methods.getSeparator();
			return path.replace( /\//g, separator );
		},

		//STATUS: OK
		getSeparator : function() {
			return Folder.fs == 'Macintosh' ? '/' : '\\';
		},

		//STATUS: OK (for the opening with path)
		/*
		* path: full path, optional
		*/
		openProject: function( path ){

			//with dialog
			if( !path ){

				var filterString = "";
				if ( Folder.fs === 'Windows' )
					filterString = "All files:*.*";

				var projToOpen	= File.openDialog ("Choose project:", filterString, false);

				if ( ( projToOpen ) && projToOpen.exists ) {
					path = projToOpen.fsName;

					app.openDocument( path,
						0, // suppress 'Convert Project' dialogs?
						0, // suppress 'Locate Files' dialogs?
						0, // suppress warning dialogs?
						1 // doNotAddToMRUList (avaiable in PPro v12.1+)
					);

					projToOpen.close();
				}

			//without dialog
			} else {

				app.openDocument( path ,
					0, // suppress 'Convert Project' dialogs?
					0, // suppress 'Locate Files' dialogs?
					0, // suppress warning dialogs?
					1 // doNotAddToMRUList (avaiable in PPro v12.1+)
				);
			}

			return path;
		},

		//STATUS: TESTING
		forceCloseProject: function( project, doSave ){
			// closeDocument( 0 = forcing without saving )

			var saveFlag = doSave? 1 : 0;
			var success = project.closeDocument( saveFlag );

			//for developing: if saving is not possible the document isn't closed. So we don't save and close it manually again
			if( !success && doSave )
				success = project.closeDocument( 0 );

			return success;
		},

		//STATUS: OK
		isCurrentProject: function( projectName ){
			var current = false;

			if( app.project.name == projectName )
				current = true;

			return current;
		},

		//STATUS: OK
		getAllOpenProjects: function(){
			var projectsArr = [];

			var projects = app.projects;

			//bug in v12.0: projects array was returning wrong list (only the current project multiple times)
			//fixed in v12.1
			for( var i = 0; i < projects.numProjects; i++ )
				projectsArr.push( projects[ i ] );

			/*
			* projectsArr will hold all opened projects
			*
			* project will have these properties (many more):
			* 	name: fileName
			* 	path: fullPath (starting on Windows with "\\?\C:\Program Files..." )
			* 	...
			*/

			return projectsArr;
		},

		//STATUS: OK
		getOpenProjectByName: function( projectName ){
			var project = false;

			//Get all projects
			var projects = methods.getAllOpenProjects();

			for( var i = 0; i < projects.length; i++ ){
				if( projects[i].name == projectName )
					project = projects[ i ];
			}

			return project;
		},

		//STATUS: OK
		//todo: sequence name could exist multiple times. but each sequence has a unique id
		getSequenceByName: function( sequenceName ){

			var totalSequences = app.project.sequences.numSequences;
			var sequence = false;

			for( var i = 0; i < totalSequences; i++ ){
				if( app.project.sequences[ i ].name === sequenceName ){
					sequence = app.project.sequences[ i ];

					//will return the first found sequence with the name
					break;
				}
			}

			return sequence;
		},

		//STATUS: OK
		getActiveSequence: function(){
			var sequence = app.project.activeSequence;

			//when no active sequence exists it will be undefined
			if(typeof sequence == "undefined")
				sequence = false;

			return sequence;
		},
		
		//STATUS: TESTING
		isActiveSequence: function( sequence ){
			var activeSequence = methods.getActiveSequence();
			return activeSequence && activeSequence.id == sequence;
		},

		//STATUS: OK
		openSequenceByName: function( sequenceName ){
			var sequence = methods.getSequenceByName( sequenceName );

			//will return true even if the sequence was already open
			return app.project.openSequence( sequence.sequenceID );
		},

		//STATUS: OK, needs enableQE
		closeSequenceByName: function( sequenceName ){
			var close = false;

			//problem: 
			//	we can get array of all sequences via app.project.sequences (not available in QE)
			// 	we can call close() only for activeSequence got via QE (not available in standard project.sequences)
			//so we will iterate over the standard sequences, if we detect the target we make it active, select via QE and close it.
			
			var allSequences = methods.getAllSequences();
			var sequence;
			var foundSequence;
			
			for( var i = 0; i < allSequences.length; i++ ){
				sequence = allSequences[ i ];
				if( sequence.name == sequenceName ){
					foundSequence = sequence;
					break;
				}
			}
			
			if( foundSequence ){
				if( !methods.isActiveSequence( foundSequence ) )
					app.project.activeSequence = foundSequence;
				
				app.enableQE();
				var activeSequence = methods.safeGetActiveSequenceQE();

				if( activeSequence ){
					activeSequence.close(); 
					close = true;
				}
			}

			return close;
		},

		//STATUS: OK
		//returns a clean array (instead of app.project.sequences which is mixed array and object)
		getAllSequences: function(){
			var sequences = [];

			var totalSequences = app.project.sequences.numSequences;

			for( var i=0; i < totalSequences; i++ )
				sequences.push( app.project.sequences[ i ] );

			return sequences;
		},

		//STATUS: TESTING, needs enableQE
		//sometimes qe.project.getActiveSequence() isn't initialized properly, so we use this wrapper to catch errors
		safeGetActiveSequenceQE: function(){
			if( typeof qe === "undefined" )
				app.enableQE();

			var activeSequence = false;

			if( typeof qe.project.getActiveSequence !== "undefined" )
				activeSequence = qe.project.getActiveSequence();

			return activeSequence;
		},

		//STATUS: OK, needs enableQE
		/*
		* trackType: "Video", "Audio"
		*/
		//todo: if function getTrackItems is called twice, whole script breaks. why?
		getTrackItems: function( trackType ){
			app.enableQE();

			//todo: another bug: it happened in the past, that qe was available with only a few properties (e.g. not qe.project)
			//	how is this possible and how can we prevent this (only solution in the past: restart Premiere)?
			var activeSequence = methods.safeGetActiveSequenceQE();

			var trackItems = [];
			var trackItem;
			var track;

			if( activeSequence ){

				if( !trackType || trackType == "Video" ){
					var trackCount = activeSequence.numVideoTracks;

					for( var i = 0; i < trackCount; i++ ){
						track = activeSequence.getVideoTrackAt( i );

						if( track ){
							for ( var j = 0; j < track.numItems; j++ ){
								trackItem = track.getItemAt( j );

								if( trackItem && ( trackItem.type != 'Empty' ) ){
									//todo: that's the location: if function getTrackItems is called twice, everything breaks. why?
									// for debugging:
									// trackItems.push( methods.getAllProps( trackItem ) );

									trackItems.push( trackItem );
								}
							}
						}
					}
				}

				if( !trackType || trackType == "Audio" ){
					var trackCount = activeSequence.numAudioTracks;

					for( var i = 0; i < trackCount; i++ ){
						track = activeSequence.getAudioTrackAt( i );

						if( track ){
							for ( var j = 0; j < track.numItems; j++ ){
								trackItem = track.getItemAt( j );

								if( trackItem && ( trackItem.type != 'Empty' ) ){
									//todo: that's the location: if function getTrackItems is called twice, everything breaks. why?
									// error message is: "object is invalid"
									// for debugging:
									// 		trackItems.push( methods.getAllProps( trackItem ) );

									trackItems.push( trackItem );
								}
							}
						}
					}
				}

			} else {
				//dbg( "No active sequence." );
			}

			return trackItems;
		},

		//STATUS: OK, needs enableQE
		/*
		* 	trackType: optional
		*/
		getTrackItemByName: function( trackItemName, trackType ){
			var trackItem = false;
			var trackItems = methods.getTrackItems( trackType );

			for( var i = 0; i < trackItems.length; i++ ){
				if( trackItems[ i ].name == trackItemName ){
					trackItem = trackItems[ i ];
					break;
				}
			}

			return trackItem;
		},

		//STATUS: OK, needs enableQE
		/*
		* 	component is an effect object, e.g. { id: 3, name: "Lumetri Farbe" }
		*/
		getTrackItemComponents: function( trackItem ){
			var components = [];
			var component;

			//with component.getParamList() we could get the list of all params as [{index:paramName}, {index:paramName},... ]

			if( typeof trackItem.numComponents != "undefined" ){
				for( var i = 0; i < trackItem.numComponents; i++ ){
					component = trackItem.getComponentAt( i );
					components.push( component );
				}

			} else if( typeof trackItem.components != "undefined" ) {
				for( var i = 0; i < trackItem.components.numItems; i++ ){
					component = trackItem.components[ i ];
					components.push( component );
				}
			}

			return components;
		},

		//STATUS: OK, needs enableQE
		getComponentValue: function( component, paramIndex ){
			var value = component.getParamValue( component.getParamList()[ paramIndex ] );

			//todo: can we parseFloat ALL values?
			//	and watch out since (some?) values will have comma like "1,3"
			var valueFormatted = parseFloat( value.replace( ",", "." ) );

			return valueFormatted;
		},

		//STATUS: OK, needs enableQE
		getComponentById: function( components, effectId ){
			var component = false;

			for( var i = 0; i < components.length; i++ ){

				if( components[ i ].id == effectId ){
					component = components[ i ];
					break;
				}
			}

			return component;
		},

		//STATUS: OK, needs enableQE
		checkEffectParameter: function( trackItem, effectType, effectId, effectParameter ){

			var value = false;
			var trackType = effectType;

			if( trackItem ){
				var components = methods.getTrackItemComponents( trackItem );
				var component = methods.getComponentById( components, effectId );

				if( component )
					value = methods.getComponentValue( component, effectParameter );
			}

			return value;
		},

		/*
		* trackItem is supposed to be (not necessary) a trackItem object generated via enableQE, which means the method isSelected() is not available for it
		* so we are using an iteration over app.project.activeSequence.videoTracks. this will get a trackItem with an available isSelected() method
		*/
		//it will check only for the first found trackItem with a matching name of trackItem
		//STATUS: OK
		isTrackItemSelectedByName: function( trackItemName ){
			var isSelected = false;
			var found = false;

			var sequence = app.project.activeSequence;
			if( sequence ){
				var tracks = [];

				//walk over all video and audio tracks
				var videoTracks = sequence.videoTracks;
				for( var i = 0; i < videoTracks.numTracks; i++ )
					tracks.push( videoTracks[ i ] );
				var audioTracks = sequence.audioTracks;
				for( var i = 0; i < audioTracks.numTracks; i++ )
					tracks.push( audioTracks[ i ] );

				for( var i = 0; i < tracks.length; i++ ){

					if( !found ){
						var trackItems = tracks[ i ].clips;

						for( var j = 0; j < trackItems.numItems; j++ ){

							if( !found ){
								var trackItem = trackItems[ j ];

								if( trackItem.name == trackItemName ){
									found = true;

									isSelected = trackItem.isSelected();
									break;
								}
							}
						}
					}
				}

			}

			return isSelected;
		},

		//we are not able to receive a QE expanded trackitem when checking for selection
		getSelectedTrackItemName: function(){
			var getSelectedTrackItemName = false;
			var found = false;

			var sequence = app.project.activeSequence;
			if( sequence ){
				var tracks = [];

				//walk over all video and audio tracks
				var videoTracks = sequence.videoTracks;
				for( var i = 0; i < videoTracks.numTracks; i++ )
					tracks.push( videoTracks[ i ] );
				var audioTracks = sequence.audioTracks;
				for( var i = 0; i < audioTracks.numTracks; i++ )
					tracks.push( audioTracks[ i ] );

				for( var i = 0; i < tracks.length; i++ ){

					if( !found ){
						var trackItems = tracks[ i ].clips;

						for( var j = 0; j < trackItems.numItems; j++ ){

							if( !found ){
								var trackItem = trackItems[ j ];

								if( trackItem.isSelected() ){
									found = true;
									getSelectedTrackItemName = trackItem.name;
								}
							}
						}
					}
				}

			}

			return getSelectedTrackItemName;
		},

		//STATUS: OK, needs enableQE
		closeAllSequences: function(){
			app.enableQE();

			var close = true;
			var activeSequence;

			while( close ){
				activeSequence = methods.safeGetActiveSequenceQE();

				//activeSequence.close() is only available in QE
				if( activeSequence )
					activeSequence.close();

				activeSequence = methods.safeGetActiveSequenceQE();
				if( !activeSequence )
					close = false;
			}

			return true;
		},

		//itemType: "sequence", "file" optional
		//STATUS: OK
		getProjectItemsCount: function( itemType ){

			if( !itemType )
				itemType = false;

			var projectItemCount = 0;

			//possible ProjectItemType: CLIP, BIN, ROOT, FILE
			for( var i = 0; i < app.project.rootItem.children.numItems; i++){

				/*
				* todo: bug in Premiere API: we always receive type=1 (CLIP), no matter what file.
				* workaround: we check the metadata and look for "Column.Intrinsic.FilePath" -> this is only available in file items
				*
				*  new in PPro v12.1: there is a method for this ( isSequence() )
				*/
				if( !itemType || ( itemType == "sequence" && app.project.rootItem.children[ i ].getProjectMetadata().indexOf( "Column.Intrinsic.FilePath" ) === -1 ) || ( itemType == "file" ) )
					projectItemCount++;
			}

			return projectItemCount;
		},

		//STATUS: OK
		/*
		* returns the current time of CTI in frames or false
		*/
		getCti: function(){
			var cti = false;

			/*
			//possible different approach: using QE

			app.enableQE();
			var activeSequence = methods.safeGetActiveSequenceQE();
			if( activeSequence )
				cti = activeSequence.CTI.frames;
			*/

			var activeSequence;
			if( app.project )
				activeSequence = app.project.activeSequence;

			if( activeSequence ){
				var ticks = parseInt( app.project.activeSequence.getPlayerPosition().ticks, 10 );
				// getPlayerPosition() only returns ticks and seconds, so we convert to frames
				cti = ticks / 10160640000;
			}

			return cti;
		},

		getSourceCti: function(){
			app.enableQE();
			var cti = qe.source.player.getPosition();

			//todo: convert the timecode to frames to be consistent ( same format as getCti() )
			return cti;
		},

		//STATUS: OK
		changePreferenceByName: function( preferenceName, newValue, allowCreateProperty ){
			var origVal;
			var changed;
			var notFound = false;
			allowCreateProperty = allowCreateProperty || 0;

			var appProperties = app.properties;

			if ( appProperties ){

				var propertyExists = app.properties.doesPropertyExist( preferenceName );
				var propertyIsReadOnly = app.properties.isPropertyReadOnly( preferenceName );
				var oldPropertyValue = app.properties.getProperty( preferenceName );

				origVal = oldPropertyValue;

				//convert string to real bool
				if( origVal === "false" )
					origVal = false;
				else if(origVal === "true" )
					origVal = true;

				//set new val
				// optional third parameter possible: 0 = non-persistent,  1 = persistent (default)
				if( propertyExists || allowCreateProperty ){

					if( !propertyIsReadOnly || allowCreateProperty ){

						appProperties.setProperty( preferenceName, newValue, 1, allowCreateProperty );

						var safetyCheck = app.properties.getProperty( preferenceName );
						changed = safetyCheck != oldPropertyValue;

					} else {

						changed = false;
					}

				} else {

					notFound = true;
				}

			} else {
				notFound = true;
			}

			return {
				origVal: origVal,
				changed: changed,
				notFound: notFound,
				preferenceName: preferenceName
			};
		},

		getMruEntry: function( mruIndex ){
			return [
				app.properties.getProperty( mruDocPathKey + mruIndex ),
				app.properties.getProperty( mruDocSizeKey + mruIndex ),
				app.properties.getProperty( mruDocDateKey + mruIndex )
			];
		},

		setMruEntry: function( mruIndex, mruEntry ){
			// watch out: if there is no mruEntry for the index, there is:
			// 		e.g. mruEntry[ 1 ] = someValue, mruEntry[ 12 ] = someValue, mruEntry[ 0 ] = "", but it is not setable!
			// so we have to detect if setable:

			var indexSetable = app.properties.doesPropertyExist( mruDocPathKey + mruIndex );

			if( indexSetable ){
				app.properties.setProperty( mruDocPathKey + mruIndex, mruEntry[ 0 ] );
				app.properties.setProperty( mruDocSizeKey + mruIndex, mruEntry[ 1 ] );
				app.properties.setProperty( mruDocDateKey + mruIndex, mruEntry[ 2 ] );
			}

			return indexSetable;
		},

		removeFilesFromMru: function( keepFileMatchFilterCallback ){
			var destinationIndex = 0;
			var docPath;
			var keepCurrentFile;
			var totalRemovedFiles = 0;

			for( var sourceIndex = 0; sourceIndex < mruMaxDocuments; sourceIndex++ ){

				docPath = app.properties.getProperty( mruDocPathKey + sourceIndex );
				keepCurrentFile = keepFileMatchFilterCallback( docPath );

				if( keepCurrentFile ){

					if ( destinationIndex != sourceIndex ){
						methods.setMruEntry( destinationIndex, methods.getMruEntry( sourceIndex ) );
						methods.setMruEntry( sourceIndex, [ '', '', '' ] );
					}

					destinationIndex++;

				} else {
					methods.setMruEntry( sourceIndex, [ '', '', '' ] );
					totalRemovedFiles++;
				}
			}

			return totalRemovedFiles;
		},

		//STATUS: TESTING
		getFilenameFromPath: function( path ){
			return path.replace( /^.*[\\\/]/, '' );
		},

		//STATUS: TESTING
		saveProjectAsCopy: function( project ){
			var saved = false;

			if( project ){
				//todo: save the file as a copy without opening the new file. not possible right now!
				saved = project.save();
			}

			return saved;
		},

		//generic function to handle the JS->JSX->JS way
		callFromJs: function( functionId, payload ){

			var callbacks = {

				//STATUS: TESTING
				test: function( payload ){
					var sequenceName = payload.sequenceName;

					return methods.closeSequenceByName( sequenceName );
				},

				//STATUS: OK
				saveCopyAndForceCloseProject: function( payload ){
					var success = false;
					var projectName = payload.projectName;
					var project = methods.getOpenProjectByName( projectName );

					if( project )
						success = methods.forceCloseProject( project, true );

					return success;
				},

				//STATUS: OK
				openProjectAndCloseAndSaveOthers: function( payload ){
					var tutorialProjectNames = payload.tutorialProjectNames;
					var projectPathToOpen = payload.projectPath;
					var onlyOpenTutorialIfNoProjectIsOpen = payload.onlyOpenTutorialIfNoProjectIsOpen || false;

					var projects = methods.getAllOpenProjects();
					var canOpenTutorial = true;

					if( onlyOpenTutorialIfNoProjectIsOpen ){

						canOpenTutorial = false; //default

						var isAnyTutorialOpen = false;
						for( var i = 0; i < projects.length; i++ ){
							var project = projects[ i ];

							if( tutorialProjectNames.indexOf( project.name ) !== -1 ){
								isAnyTutorialOpen = true;
								break;
							}
						}

						if( isAnyTutorialOpen )
							canOpenTutorial = true;
					}

					if( canOpenTutorial )
						//first: open the project (if already open will do nothing)
						methods.openProject( projectPathToOpen );

					//second: try to close others
					for( var i = 0; i < projects.length; i++ ){
						//bug in v12.0: projects array was returning wrong list (only the current project multiple times)
						//fixed in v12.1
						var project = projects[ i ];

						//if project is one of tutorials projects
						if( tutorialProjectNames.indexOf( project.name ) !== -1 ){

							//if project is not the one to be opened: close and save it
							if( project.name != methods.getFilenameFromPath( projectPathToOpen ) )
								methods.forceCloseProject( project, true );
						}
					}

					return true;
				},

				//STATUS: OK
				saveProjectToLocation: function( payload ){

					var dirPath = payload.dirPath;
					var projectName = payload.projectName;

					var newPath = dirPath + "/" + projectName;
					var success;

					var project = app.project;

					//todo: check, if the project is really the one we think it is (the current tutorial project)
					if( project ){
						//save the current project to new file
						success = project.saveAs( newPath );

						/*
						attention: actual behavior from saveAs():
							- current project will be closed, saved project is immediatly opened by Premiere!
							- saved project is generally not added to MRU list!
							- saved project will overwrite all with same name without asking!
						*/
					}

					return {
						success: success
					};
				},

				//STATUS: OK
				bindProjectItemSelected: function(){
					app.bind( 'onSourceClipSelectedInProjectPanel', function(){
						//arguments will be each a projectItem

						//store globally
						totalSelectedProjectItems = arguments.length;
					} );

					return true;
				},

				//STATUS: OK
				unbindProjectItemSelected: function(){

					app.unbind( 'onSourceClipSelectedInProjectPanel' );

					//reset
					totalSelectedProjectItems = 0;

					return true;
				},

				//STATUS: OK
				//todo: the handling of getting the selected project itmes is done via bind("onSourceClipSelectedInProjectPanel") and globally stored items.
				//	could be improved
				getSelectedProjectItems: function(){
					return totalSelectedProjectItems;
				},

				//STATUS: OK
				cleanupMruList: function( payload ){

					/*
					var removeThese = payload.tutorialProjectNames; //array of all tutorial project names

					methods.removeFilesFromMru( function( documentPath ){
						// should return true if file should be kept and false if file should be deleted

						var keep = true;
						if( documentPath != "" ){

							//don't remove the first tutorial project form MRU (it's the only "visible" project there)
							for( var i = 1; i < removeThese.length; i++ )
								if( documentPath.search( removeThese[ i ] ) != -1 ){
									keep = false;
									continue;
								}

						} else {
							keep = false;
						}

						return keep;
					} );
					*/

					return true;
				},

				showOnboarding: function(){
					if( typeof app.showOnboarding !== "undefined" )
						app.showOnboarding( 3 );
					else
						dbg( "app.showOnboarding() not implemented in current build" );

					return true;
				},

				//STATUS: OK
				setPrefImportPath: function( newPath ){
					newPath = methods.getCleanPath( newPath );

					//that's the path which will open the directory when e.g. the user doubleclicks on the project panel to import custom media
					var result = methods.changePreferenceByName( "be.prefs.last.used.directory", newPath, 1 );
					return result;
				},

				//STATUS: OK
				setPrefImportWorkspace: function( newVal ){

					//that's the setting in the menu "Window > Workspaces > Import Workspace from project"
					var result = methods.changePreferenceByName( "FE.Prefs.ImportWorkspace", newVal );
					return result;
				},

				//STATUS: OK
				setPrefRippleSequenceMarkers: function( newVal ){

					//that's the setting in the menu "Markers > Ripple Sequence Markers"
					var result = methods.changePreferenceByName( "FE.Prefs.Timeline.SequenceMarker.Ripple.Mode", newVal );
					return result;
				},

				//STATUS: OK
				setExtensionPersistent: function(){
					var extensionId = "com.adobe.LABORLearningPanel.course";
					app.setExtensionPersistent( extensionId, 1 );

					return true;
				},

				//STATUS: OK
				getCurrProjectName: function(){
					var project = app.project;

					return project ? project.name : false;
				},

				//STATUS: OK
				getCti: function(){
					var cti = methods.getCti();

					return cti;
				},

				//STATUS:
				getSourceCti: function(){
					var cti = methods.getSourceCti();

					return cti;
				},

				//STATUS: OK
				getClipsCount: function( payload ){
					var clipsCount;

					if( payload && payload.clipName )
						clipsCount = methods.getTrackItemByName( payload.clipName, "Video" ) ? 1 : 0;
					else
						clipsCount =  methods.getTrackItems( "Video" ).length;

					return clipsCount;
				},

				//STATUS: OK
				/*
				* payload.clipName: STR, optional. else it will get the selected track item
				*/
				getClipProperties: function( payload ){
					var properties = false;
					var trackItemName = payload && payload.clipName ? payload.clipName : methods.getSelectedTrackItemName();

					if( trackItemName ){
						var trackItem = methods.getTrackItemByName( trackItemName );
						properties = trackItem;
					}

					return properties;
				},

				//STATUS: OK
				getClipsProperties: function(){
					var properties = [];
					var trackItems = methods.getTrackItems( "Video" );

					for( var i = 0; i < trackItems.length; i++)
						properties.push( trackItems[ i ] );

					return properties;
				},

				//STATUS: OK
				closeAllSequences: function(){
					return methods.closeAllSequences();
				},

				//STATUS: OK
				getProjectItemsCount: function( payload ){
					var itemType = payload.itemType;

					return methods.getProjectItemsCount( itemType );
				},

				//STATUS: OK
				getTrackItemEffectValueByName: function( payload ){
					var effectType = payload.effectType;
					var effectId = payload.effectId;
					var effectParameter = payload.effectParameter;
					var trackItemName = payload.trackItemName;

					var trackItem = methods.getTrackItemByName( trackItemName );
					var value = methods.checkEffectParameter( trackItem, effectType, effectId, effectParameter );

					return value;
				},

				//STATUS: OK
				getSelectedTrackItemEffectValue: function( payload ){
					var effectType = payload.effectType;
					var effectId = payload.effectId;
					var effectParameter = payload.effectParameter;

					var trackItem = methods.getTrackItemByName( methods.getSelectedTrackItemName() );
					var value = methods.checkEffectParameter( trackItem, effectType, effectId, effectParameter );

					return value;
				},

				//STATUS: OK
				getAllSequencesNames: function(){
					var sequences = methods.getAllSequences();
					var names = [];

					for( var i=0; i < sequences.length; i++ )
						names.push( sequences[i].name );

					return names;
				},

				//STATUS: OK
				getAllOpenProjectsNames: function(){

					var projects = methods.getAllOpenProjects();
					var names = [];

					//bug in v12.0: projects array was returning wrong list (only the current project multiple times)
					//fixed in v12.1
					for( var i=0; i < projects.length; i++ )
						names.push( projects[i].name );

					return names;
				},

				//STATUS: OK
				openProject: function( payload ){
					/*
					//todo:
					//	possible to set forceImportWorkspace here. but it doesn't make sense since Premiere won't import workspaces if there any projects opened already
					//	(which is the case so the panel could be visible in the first place)
					var forceImportWorkspace = payload.forceImportWorkspace;
					var prefResult;

					//set pref
					if( forceImportWorkspace )
						prefResult = methods.changePreferenceByName( "FE.Prefs.ImportWorkspace", true );
					*/

					var path = payload.projectPath;
					path = methods.getCleanPath( path );
					var project = methods.openProject( path );

					/*
					//reset pref
					if( forceImportWorkspace && prefResult.changed )
						methods.changePreferenceByName( "FE.Prefs.ImportWorkspace", prefResult.origVal );
					*/

					return project;
				},

				//STATUS: OK
				forceCloseProject: function( payload ){
					var success = false;
					var projectName = payload.projectName;
					var project = methods.getOpenProjectByName( projectName );

					if( project )
						success = methods.forceCloseProject( project );

					return success;
				},

				//STATUS: TESTING
				goToMarkerByIndex: function( payload ){

					var sequenceName = payload.sequenceName;
					var index = payload.index;

					return methods.goToMarker(index, sequenceName);

				},

				//STATUS: TESTING
				goToMarkerByName: function( payload ){

					var sequenceName = payload.sequenceName;
					var markerName = payload.markerName;

					return methods.goToMarker( markerName, sequenceName );
				},

				//STATUS: OK
				openSequenceByName: function( payload ){
					var sequenceName = payload.sequenceName;

					return methods.openSequenceByName( sequenceName );
				},

				//STATUS: OK
				closeSequenceByName: function( payload ){
					var sequenceName = payload.sequenceName;

					return methods.closeSequenceByName( sequenceName );
				}

			};

			if( typeof callbacks[ functionId ] !== "undefined" )
				return JSON.stringify( callbacks[ functionId ]( typeof payload !== "undefined" ? ( methods.isJsonString( payload ) ? JSON.parse( payload ) : payload ) : undefined ) );

		}

	}

	//shortcuts
	var dbg = methods.dbg;

	//return public methods & properties
	return {
		callFromJs: methods.callFromJs,
		methods: methods
	};

} )();

//make accessible for JS via $
$.callFromJs = APIModule.callFromJs;
$.methods = APIModule.methods; //for debugging