/**
* 	Premiere Learning Panel
* 	for Premiere Pro v12.1+
*
*	@author: Florian Lange <f.lange@labor.digital>
*	LABOR – Agentur für moderne Kommunikation GmbH, Mainz, Germany
* 	https://labor.digital
*
*/

/*
*	0 = production mode
*	1 = + show "refresh" bar at top
*	2 = + debug out the JSX interactions
*	3 = + show all dev testing controls
*
*/
var DEBUG_LEVEL = 0;

//INSTALL_MODE = "globalExtensionFolder", "applicationFolder"
var INSTALL_MODE = "applicationFolder";



/*
* CEPPanel is an universal module to handle:
*	OS-specific things and paths
*	JSX interactions
*	logging
*	theme/skin appearance
*
*/
var CEPPanel = ( function(){

	var glo = {};

	glo.callFromHtmlCallbacks = {
		onBeforeJSX: {
			//will be set via extendCallFromHtml()
		},
		onAfterJSX: {
			//will be set via extendCallFromHtml()
		}
	};

	var methods = {

		/*
		* generic function to handle the JS->JSX->JS way
		*
		*	functionId: STR. mandatory in JSX. optional in JS
		* 		the function with the name can be called in JS before JSX
		* 		the function with the name will be called in JSX
		* 		the function with the name can be called in JS after JSX
		* 	payload: any data to be passed to JSX
		* 	customCallback: optional custom callback in JS
		*/

		callFromHtml: function( functionId, payload, customCallback ){
			// Get the extension root
			var extensionPath = glo.extensionPath;

			var callbacksOnBeforeJSX = glo.callFromHtmlCallbacks.onBeforeJSX;
			var callbacksOnAfterJSX = glo.callFromHtmlCallbacks.onAfterJSX;

			var defaultCallBackOnAfterJSX = function( data ){
				if( DEBUG_LEVEL >= 2 )
					methods.dbg( "default callback after JSX", functionId, data );
			};

			payload = typeof callbacksOnBeforeJSX[functionId] !== 'undefined' ? callbacksOnBeforeJSX[ functionId ]( payload ) : payload;


			//error check: we cannot JSONify undefined values
			if( typeof payload !== 'undefined' && typeof payload !== 'string' && DEBUG_LEVEL >= 2 )
				$.each( payload, function( keyOrIndex, val ){
					if( typeof val === "undefined" )
						dbg( "undefined value in payload for callFromHtml", keyOrIndex, payload );
				});

			/*
			* if we have payload, we will provide it as string in the eval function
			* watch out to use the correct quotes and doublequotes since JSON.stringify will wrap everything in doublequotes!
			*/
			var evalStr = "$.callFromJs('" + functionId + "'" + (typeof payload !== 'undefined'? ", '"+ (typeof payload != "string"? JSON.stringify( payload ) : payload ) +"'" : "") + ");"

			glo.csInterface.evalScript( evalStr, function( data ){

				typeof callbacksOnAfterJSX[ functionId ] != "undefined" ?
					callbacksOnAfterJSX[ functionId ]( JSON.parse( data ) ) :
					defaultCallBackOnAfterJSX( JSON.parse( data ) );

				if( customCallback )
					customCallback( JSON.parse( data ) );
				} );
		},

		//you can call this func from other modules to extend for more custom callbacks of the callFromHtml() func
		extendCallFromHtml: function( callbacksOnBeforeJSX, callbacksOnAfterJSX ){
			$.extend( glo.callFromHtmlCallbacks.onBeforeJSX, callbacksOnBeforeJSX );
			$.extend( glo.callFromHtmlCallbacks.onAfterJSX, callbacksOnAfterJSX );
		},

		getEnvironmentVars: function(){
			return {
				OSVersion: glo.OSVersion,
				extensionPath: glo.extensionPath,
				csInterface: glo.csInterface
			};
		},

		/*
		*	eventName: "any custom name"
		*	callback: function( data ){}
		*/
		bindJSXListener: function( eventName, callback ){

			glo.csInterface.addEventListener("My Custom Event", function( e ) {
				dbg('Data from the JSX payload: ' + e.data );

				//todo: in progress: JSON parsing not working properly, investigate...
				//callback( JSON.parse( e.data ) );
			});
		},

		//inital call from html
		onLoaded: function(){
			//set environment vars:


			//csInterface instance
			glo.csInterface = new CSInterface();
			// Get the OS
			glo.OSVersion = glo.csInterface.getOSInformation();

			// Get the extension root
			var extensionRootCEP = glo.csInterface.getSystemPath( SystemPath.EXTENSION );

			var extensionRootApp = glo.csInterface.getSystemPath( SystemPath.HOST_APPLICATION );
			//fix: if extension lives in application directory, for Mac we have to go one level up, since the PPro application lives in "/MacOS"
			if( methods.getOS() == "Mac" )
				extensionRootApp = extensionRootApp.replace( "/MacOS", "" );

			extensionRootApp = extensionRootApp.substr( 0, extensionRootApp.lastIndexOf("/") );
			extensionRootApp += "/CEP/extensions/com.adobe.LABORLearningPanel"; //hardcoded path info!

			glo.extensionPath = INSTALL_MODE == "globalExtensionFolder"? extensionRootCEP : extensionRootApp;

			//listen for skin changes
			methods.initSkinHandling();

			//load JSX files
			methods.loadJSX();
		},

		//load all JSX files
		loadJSX: function(){
			// load files
			var extensionPathGeneral = glo.extensionPath + '/jsx/';
			glo.csInterface.evalScript( '$._ext.evalFiles("' + extensionPathGeneral + '")' );

			var extensionPathSpecific = glo.extensionPath + '/jsx/jsx-course/';
			glo.csInterface.evalScript( '$._ext.evalFiles("' + extensionPathSpecific + '")' );
		},

		//debug out in console
		dbg: Function.prototype.bind.call(console.log, console),

		//helper (from first.js)
		makeDiv: function () {

			var
				attribs = arguments[0],
				attrStringified = "";

			/*
			 * if attrib is string: we expect somthing like this:
			 * 		".className.anotherClass"
			 * 	and make it to
			 * 		{ 'class' : "className anotherClass" }
			 */
			if (typeof attribs == "undefined") {
				attribs = {};

			} else if (typeof attribs == "string") {
				attribs = {
					"class": $.trim(attribs.split(".").join(" "))
				};
			}

			$.each(attribs, function (attrName, attrVal) {
				attrStringified += attrName + '="' + attrVal + '" ';
			});

			var contents = $.makeArray(arguments);
			contents.shift();
			contents = contents.join("");

			return '<div ' + attrStringified + '>' + contents + '</div>';
		},

		getSeparator: function(){
			var separator = methods.getOS() == "Mac"? '/' : '\\';
			return separator;
		},

		getOS: function(){
			var currentSystem = glo.OSVersion;
			return currentSystem.indexOf( 'Mac' ) == 0? "Mac" : "Win";
		},

		getCleanPath: function( path ){
            path = path.replace( /\//g, methods.getSeparator() );
			return path;
		},

		initSkinHandling: function(){
			methods.updateThemeWithAppSkinInfo( glo.csInterface.hostEnvironment.appSkinInfo );

			// Update the color of the panel when the theme color of the product changed.
			glo.csInterface.addEventListener( CSInterface.THEME_COLOR_CHANGED_EVENT, methods.onAppThemeColorChanged );
		},

		onAppThemeColorChanged: function( e ){
			methods.updateSkin();
		},

		updateSkin: function(){
			// Should get a latest HostEnvironment object from application.
			var skinInfo = JSON.parse( window.__adobe_cep__.getHostEnvironment() ).appSkinInfo;

			// Gets the style information such as color info from the skinInfo,
			// and redraw all UI controls of your extension according to the style info.
			methods.updateThemeWithAppSkinInfo( skinInfo );
		},

		updateThemeWithAppSkinInfo: function( appSkinInfo ){
			var panelBackgroundColorRaw = appSkinInfo.panelBackgroundColor.color;
			var panelBackgroundColor = methods.toHex( panelBackgroundColorRaw );

			if( !glo.customStylesheet ){
				$("<style></style>").appendTo( "head" );
				glo.customStylesheet = document.styleSheets[ document.styleSheets.length - 1 ];
			}

			//remove all custom rules
			while( glo.customStylesheet.cssRules.length )
				glo.customStylesheet.deleteRule( 0 );

			if( glo.callbackOnThemeUpdate )
				glo.callbackOnThemeUpdate( panelBackgroundColor, panelBackgroundColorRaw, appSkinInfo, methods.addCSSRuleToGlobalSheet, methods.toHex );
		},

		setCallbackOnThemeUpdate: function( callbackOnThemeUpdate ){
			glo.callbackOnThemeUpdate = callbackOnThemeUpdate;
		},

		addCSSRuleToGlobalSheet: function( selector, rules ){
			methods.addCSSRule( glo.customStylesheet, selector, rules );
		},

		addCSSRule: function( sheet, selector, rules ) {
			sheet.insertRule( selector + "{" + rules + "}", 0 );
		},

		/**
		 * Convert the Color object to string in hexadecimal format
		 */
		toHex: function( color, delta ){

			function computeValue(value, delta) {
				var computedValue = !isNaN( delta ) ? value + delta : value;
				if( computedValue < 0 ){
					computedValue = 0;
				} else if ( computedValue > 255 ) {
					computedValue = 255;
				}

				computedValue = Math.round( computedValue ).toString( 16 );
				return computedValue.length == 1 ? "0" + computedValue : computedValue;
			}

			var hex = "";

			if ( color )
				hex = computeValue( color.red, delta ) + computeValue( color.green, delta ) + computeValue( color.blue, delta );

			return "#" + hex;
		},

		triggerEvent: function( name, payload ){
			var event = new CSEvent( name, "APPLICATION" );

			if( payload ){

				//convert all values to string (true => "true", 1 => "1" etc.) for Adobe instrumentation
				$.each( payload, function( key, val ){
					if(typeof val !== "string")
						payload[ key ] = String( val );
				} );

				event.data = payload;
			}

			glo.csInterface.dispatchEvent( event );
		}

	};

	//shortcuts
	var dbg = methods.dbg;


	//return public funcs
	return {
		callFromHtml: methods.callFromHtml,
		extendCallFromHtml: methods.extendCallFromHtml,
		dbg: methods.dbg,
		getEnvironmentVars: methods.getEnvironmentVars,
		getCleanPath: methods.getCleanPath,
		getOS: methods.getOS,
		onLoaded: methods.onLoaded,
		bindJSXListener: methods.bindJSXListener,
		makeDiv: methods.makeDiv,
		setCallbackOnThemeUpdate: methods.setCallbackOnThemeUpdate,
		updateSkin: methods.updateSkin,
		triggerEvent: methods.triggerEvent
	};

} )();













var CEPTutorials = ( function(){

	//shortcuts ;-)
	var dbg = CEPPanel.dbg;
	var makeDiv = CEPPanel.makeDiv;


	var glo = {
		currTutorialIndex: false,
		eventListenerInterval: false,
		panelPayloadsPathRelative: false,
		panelPayloadsPath: false,
		projectPayloadsPath: false,
		preinstalledPayloadsPath: false,
		isBind:{
			projectItemSelected: false
		},
		userPrefs: {},
		translations: {
			endScreenMoveToNextTutorial: "Move on to the next tutorial by clicking on the image below.",
			endScreenBackToOverview: "Back to tutorials overview",
			endScreenStartYourOwnProject: "Start your own project now!",
			startScreenHeadline: "Premiere Pro Tutorials",
			navAllTutorials: "All Tutorials",
			navPrevTutorial: "Back",
			navNextTutorial: "Next",
			navPrevStep: "Back",
			navNextStep: "Next",
			teasersLetsStart: "Let's go!",
			currentTutorial: "Current Tutorial: ",
		}
	};

	//init the specific callbacks from this module
	var callbacksOnBeforeJSX = {

		/*
		* write callbacks like this:
		*
		* myCustomFunc: function( payload ){
		*	//do some
		*
		*	return payload;
		* }
		*
		*/

	};

	var callbacksOnAfterJSX = {

		/*
		* write callbacks like this:
		*
		* myCustomFunc: function( data ){
		* 	//data = return from JSX
		* }
		*
		*/

	};

	CEPPanel.extendCallFromHtml( callbacksOnBeforeJSX, callbacksOnAfterJSX );

	//initial call on doc ready
	$( document ).ready(function() {

		//store global $objs
		glo.$startScreen = $( '*[data-main]' );
		glo.$navigation = $( '*[data-navigation]' );
		glo.$stepNavigation = glo.$navigation.find( ".stepNavigation" )
		glo.$tutorialContentsContainer = $( ".g8_tutorialContentsContainer" );

		//init texts
		glo.$startScreen.find('*[data-startscreen-heading]').html( glo.translations.startScreenHeadline );

		glo.$navigation.hide();

		//the getEnvironmentVars() needs some time to be initied (about 3 millisec)
		var intervalCheckEnvironment = setInterval( function(){
			glo.environmentVars = CEPPanel.getEnvironmentVars();

			if( typeof glo.environmentVars.csInterface !== "undefined"){
				clearInterval( intervalCheckEnvironment );

				methods.initTutorials();
			}
		}, 1 );
	});

	var methods = {

		initTutorials: function(){
			var extensionPath = glo.environmentVars.extensionPath;

			var environmentVars = CEPPanel.getEnvironmentVars();
			var currentSystem = environmentVars.OSVersion;

			glo.panelPayloadsPathRelative = "panel-payloads/"; //hardcoded path info!
			glo.panelPayloadsPath = extensionPath + "/panel-payloads/"; //hardcoded path info!
			glo.projectPayloadsPath = extensionPath + "/project-payloads/"; //hardcoded path info!

			//get the correct path to the preinstalled assets
			//Mac
			if( CEPPanel.getOS() == "Mac" )
				glo.preinstalledPayloadsPath = "/Users/Shared/Adobe/Premiere Pro/12.0/Tutorial/Going Home project/"; //hardcoded path info!

			//Win
			else {

				var userPathPrefix = "C:/Users/"; //default

				//we have to distinguish where the user directoy is (bootpartition C: or D: etc.)
				var userDataDir = environmentVars.csInterface.getSystemPath( SystemPath.USER_DATA ); //will return something like "C:/Users/myUserName/AppData/Roaming"
				var strMatch = userDataDir.match( /.*Users\// );
				if( strMatch )
					userPathPrefix = strMatch[ 0 ];

				glo.preinstalledPayloadsPath = userPathPrefix + "Public/Documents/Adobe/Premiere Pro/12.0/Tutorial/Going Home project/"; //hardcoded path info!
			}

			//set extension persistent, so any workspace changes won't kill it
			CEPPanel.callFromHtml( 'setExtensionPersistent' );

			methods.initSkin();

			//todo: add global listener to be able to listen to JSX events
			/*
			CEPPanel.bindJSXListener( "My Custom Event", function( data ){
				dbg("we received from JSX:", data );
			});
			*/

			//initial bind navigation
			glo.$navBtnNextStep = glo.$navigation.find( '*[data-navigation-next-step]' ).on( 'click', methods.onClickStepNext ).html( glo.translations.navNextStep );
			glo.$navBtnPrevStep = glo.$navigation.find( '*[data-navigation-prev-step]' ).on( 'click', methods.onClickStepPrev ).html( glo.translations.navPrevStep );
			glo.$navBtnNextTutorial = glo.$navigation.find( '*[data-navigation-next-tutorial]' ).on( 'click', methods.onClickTutorialNext ).html( '<span class="responsiveHideSmall">' + glo.translations.navNextTutorial + ' </span><i class="fa fa-chevron-right" aria-hidden="true"></i>' );
			glo.$navBtnPrevTutorial = glo.$navigation.find( '*[data-navigation-prev-tutorial]' ).on( 'click', methods.onClickTutorialPrev ).html( '<i class="fa fa-chevron-left iconLeft" aria-hidden="true"></i><span class="responsiveHideSmall"> ' + glo.translations.navPrevTutorial + '</span>' );
			glo.$navBtnTutorialOverview = glo.$navigation.find( '*[data-navigation-tutorials-overview]' ).on( 'click', methods.onClickTutorialsOverview ).html( '<i class="fa fa-bars iconLeft" aria-hidden="true"></i><span class="responsiveHideSmall"> ' + glo.translations.navAllTutorials + '</span>' );

			//read config from json (containing overview on all tutorials in order)
			var contents = methods.readFile( glo.panelPayloadsPath + "tutorials-config.json" );
			glo.tutorials = JSON.parse( contents );

			glo.$groups = [];
			var $groupsContainer = $(".g8_groups");
			$.each( glo.tutorials, function( tutorialIndex, tutorialConfig){

				//extend config
				glo.tutorials[ tutorialIndex ].tutorialIndex = tutorialIndex; //keep the order since tutorialIndex is needed by the tutorialObj
				var tutorialObj = new Tutorial( tutorialConfig );
				glo.tutorials[ tutorialIndex ].tutorialObj = tutorialObj;
				if( tutorialConfig.projectUrl )
					glo.tutorials[ tutorialIndex ].projectUrl = methods.resolveShortcutPaths( tutorialConfig.projectUrl );

				//create/add to group
				var $group = $groupsContainer.find( ".topic" ).filterByData( "groupTitle", tutorialConfig.groupTitle );

				//create group if it not exist yet
				if( !$group.length ){
					// todo: temp disabled. put in toggled groups again in later releases with more than one tutorial group
					/*
					$group = $( makeDiv(".topic") ).appendTo( $groupsContainer ).data( "groupTitle", tutorialConfig.groupTitle );
					var $toggler = $( makeDiv( ".topic__toggler",
						makeDiv( {"class":"topic__toggler__thumb", "style":"background-image:url('" + tutorialConfig.groupThumbUrl + "');" } ),
						makeDiv( ".topic__toggler__title", tutorialConfig.groupTitle )
					) );
					$toggler.on( "click", methods.onToggleGroupContent);
					$group.append( $toggler );
					$group.append( makeDiv( ".topic__toggled" ) );

					glo.$groups.push( $group );
					*/

					$group = $( makeDiv(".topic.oneGroupOnly") ).appendTo( $groupsContainer ).data( "groupTitle", tutorialConfig.groupTitle );
					var $toggler = $( makeDiv( ".topic__toggler",
						makeDiv( ".topic__toggler__title", tutorialConfig.groupTitle ),
						makeDiv( ".topic__toggler__subtitle", tutorialConfig.groupSubtitle )
					) );
					$group.append( $toggler );
					$group.append( makeDiv( ".topic__toggled" ) );

					glo.$groups.push( $group );
				}

				//create teaser
				var $teaser = $( makeDiv( ".teaser" ) ).appendTo( $group.find( ".topic__toggled" ) ).data( "tutorialIndex", tutorialIndex ).data( "tutorialObj", tutorialObj );
				$teaser.append( makeDiv( { "class":"teaser__img", "style":"background-image:url('" + tutorialConfig.thumbUrl + "');" }, makeDiv( ".btnHolder", makeDiv( ".btn", glo.translations.teasersLetsStart ) ) ) );
				$teaser.append( makeDiv( ".teaser__title", makeDiv( ".title__heading", tutorialConfig.title ), makeDiv( ".title__time", tutorialConfig.tutorialDuration ) ) );

				//bind click start tutorial
				$teaser.on( "click", methods.onClickTeaserTutorialStart );
			} );

			//initial show/hide
			methods.showStartScreen();
		},

		initSkin: function(){

			//add custom callback
			CEPPanel.setCallbackOnThemeUpdate( function( panelBackgroundColor, panelBackgroundColorRaw, appSkinInfo, addCSSRule, toHex ){
				//colors:
				var panelBackgroundColorLighter = toHex( panelBackgroundColorRaw, 10 );
				var panelBackgroundColorDarker = toHex( panelBackgroundColorRaw, -30 );
				var isPanelThemeLight = panelBackgroundColorRaw.red > 60;

				addCSSRule( "body, .navigation", "background-color:" + panelBackgroundColor );

				if( isPanelThemeLight ){

					addCSSRule( ".navigation", "border-color: #313131" );
					addCSSRule( ".preHeading", "color: #888" );
					addCSSRule( ".button.btn--secondary:hover, .btn--secondary:hover", "background-color: #373737" );

				} else {

					addCSSRule( ".navigation", "border-color: #4D4D4D" );
					addCSSRule( ".preHeading", "color:#777" );
					addCSSRule( ".button.btn--secondary:hover, .btn--secondary:hover", "background-color: #444" );
				}
			});

			//trigger initial update
			CEPPanel.updateSkin();
		},

		resolveShortcutPaths: function( path ){
			return !path? "" : path.replace( "{{ProjectPayloadsPath}}", glo.projectPayloadsPath ).replace( "{{PreinstalledPayloadsPath}}", glo.preinstalledPayloadsPath );
		},

		showStartScreen: function(){
			//always open the first project on startscreen
			CEPPanel.callFromHtml( 'openProjectAndCloseAndSaveOthers', {
				projectPath: methods.getAllTutorialProjectsPaths()[ 0 ],
				tutorialProjectNames: methods.getAllTutorialProjectsNames(),
				onlyOpenTutorialIfNoProjectIsOpen: true
			 } );

			glo.$navigation.hide();

			//hide other tutorials
			$( ".screen" ).hide();

			glo.$startScreen.show();
		},

		showOnboarding: function(){
			//trigger event for instrumentation
			CEPPanel.triggerEvent( "com.adobe.learn.events.ClickedShowOnboarding", {
				TutorialName: this.getCurrentTutorialObj().tutorialConfig.title
			} );

			CEPPanel.callFromHtml( 'showOnboarding' );
		},

		getCurrentTutorialObj: function(){
			return glo.tutorials[ glo.currTutorialIndex ].tutorialObj;
		},

		onToggleGroupContent: function( e ){
			//todo: streamline this whole $ selection of group, toggler etc. ;-)
			var $groups = glo.$groups;

			var $toggler = $( this );
			var $groupContent = $( this ).next( '.topic__toggled' );

			var show = !$groupContent.is( ':visible' );

			show ? $groupContent.slideDown( 300 ) : $groupContent.slideUp( 300 );
			$toggler.toggleClass( "untoggled", show );

			//collapse other groups
			if( show ){
				$.each( $groups, function( index, $group ){
					if( $toggler.closest( ".topic" )[ 0 ] != $group[ 0 ] ){
						$group.find( '.topic__toggled' ).slideUp( 300 );
						$group.find( ".topic__toggler" ).removeClass( "untoggled" );
					}
				}) ;
			}
		},

		onClickTeaserTutorialStart: function( e ) {
			e.preventDefault();
			var $teaser = $( this );

			var tutorialObj = $teaser.data( 'tutorialObj' );
			tutorialObj.startTutorial();
		},

		onClickNextTeaserTutorialStart: function( e ){
			e.preventDefault();
			var tutorialObj = methods.getCurrentTutorialObj();

			tutorialObj.onUnloadTutorial( function(){
				var newTutorialObj = e.data.tutorialObj;
				newTutorialObj.startTutorial();
			});
		},

		onClickStepNext: function( e ){
			e.preventDefault();
			var tutorialObj = methods.getCurrentTutorialObj();
			tutorialObj.stepNext();
		},

		onClickStepPrev: function( e ){
			e.preventDefault();
			var tutorialObj = methods.getCurrentTutorialObj();
			tutorialObj.stepPrev();
		},

		onClickTutorialNext: function( e ){
			e.preventDefault();
			var tutorialObj = methods.getCurrentTutorialObj();

			tutorialObj.onUnloadTutorial( function(){
				var newTutorialObj = glo.tutorials[ glo.currTutorialIndex + 1 ].tutorialObj;
				newTutorialObj.startTutorial();
			} );
		},

		onClickTutorialPrev: function( e ){
			e.preventDefault();
			var tutorialObj = methods.getCurrentTutorialObj();

			tutorialObj.onUnloadTutorial( function(){
				var newTutorialObj = glo.tutorials[ glo.currTutorialIndex - 1 ].tutorialObj;
				newTutorialObj.startTutorial();
			} );
		},

		onClickTutorialsOverview: function(e ){
			e.preventDefault();
			var tutorialObj = methods.getCurrentTutorialObj();

			//trigger event for instrumentation
			//only trigger "TutorialExited" when we are NOT on the last screen of a tutorial
			if( tutorialObj.currentStep < tutorialObj.tutorialConfig.steps.length - 1 )
				CEPPanel.triggerEvent( "com.adobe.learn.events.TutorialExited", {
					TutorialName: tutorialObj.tutorialConfig.title,
					TutorialPageNumber: tutorialObj.currentStep + 1
				} );

			tutorialObj.onUnloadTutorial( function(){
				methods.showStartScreen();
			} );
		},

		toggleDisplay: function( $obj, show){
			show? $obj.show() : $obj.hide();
		},

		readFile: function( path ){
            path = CEPPanel.getCleanPath( path );
			var result = window.cep.fs.readFile( path );

			//success
			if( result.err === 0 ){
				//result.data is file content
				return result.data;
			} else {
				dbg( "could not open file", path);
				return false;
			}
		},

		getFilenameFromPath: function( path ){
			return path.replace( /^.*[\\\/]/, '' );
		},

		//returns the project names of all tutorial projects
		getAllTutorialProjectsNames: function(){
			var projectsNames = [];

			$.each( glo.tutorials, function( index, tutorialConfig ){
				if( tutorialConfig.projectUrl )
					projectsNames.push( methods.getFilenameFromPath( tutorialConfig.projectUrl ) );
			} );

			return projectsNames;
		},

		//returns the project paths of all tutorial projects
		getAllTutorialProjectsPaths: function(){
			var projectsPaths = [];

			$.each( glo.tutorials, function( index, tutorialConfig ){
				if( tutorialConfig.projectUrl )
					projectsPaths.push( tutorialConfig.projectUrl );
			} );

			return projectsPaths;
		}

	};


	var SpriteAnimation = function( $object ){
		var that = this;
		this.$object = $object;
		this.totalSteps = $object.data( "spriteAnimationSteps" );
		this.stepDura = 1000/24;

		this.$object.addClass( "spriteAnimation-0" );
		this.$object.on( "runSpriteAnimation", function(){
			that.run();
		} );
	};

	SpriteAnimation.prototype = {

		reset: function(){
			for( var i = 0; i < this.totalSteps; i++ )
				this.$object.removeClass( "spriteAnimation-" + i );

			this.$object.addClass( "spriteAnimation-0" );
		},

		run: function(){
			var that = this;
			var i = 0;

			this.reset();

			if( this.aniInterval )
				clearInterval( this.aniInterval );

			var interval = setInterval( function(){

				if( i < that.totalSteps - 1 ){
					that.$object.removeClass( "spriteAnimation-" + i );
					i++;
					that.$object.addClass( "spriteAnimation-" + i );

				} else {
					clearInterval( this.aniInterval );
				}

			}, this.stepDura );
		}
	};


	var Tutorial = function( tutorialConfig ){
		var that = this;

		this.$steps = [];
		this.nextTutorial = false;
		this.prevTutorial = false;
		this.tutorialConfig = tutorialConfig;

		$.each( glo.tutorials, function( index, otherTutorialConfig ){
			if( otherTutorialConfig.tutorialIndex == tutorialConfig.tutorialIndex ){
				that.nextTutorialConfig = index + 1 < glo.tutorials.length ? glo.tutorials[ index + 1 ] : false;
				that.prevTutorialConfig = index - 1 >= 0 ? glo.tutorials[ index - 1 ] :  false;
			}
		});
	};

	Tutorial.prototype = {

		startTutorial: function(){
			var that = this;
			glo.currTutorialIndex = this.tutorialConfig.tutorialIndex;

			//create markup
			glo.$tutorialContentsContainer.empty();
			var $tutorialContent = $( makeDiv( ".screen.tutorial-steps" ) ).appendTo( glo.$tutorialContentsContainer );
			this.$steps = []; //reset

			//trigger event for instrumentation
			CEPPanel.triggerEvent( "com.adobe.learn.events.TutorialStarted", {
				TutorialName: this.tutorialConfig.title
			} );

			//create each step
			$.each( this.tutorialConfig.steps, function( stepIndex, stepConfig ){

				//store actionable items
				//todo: refactor $steps[] to steps[].$step and put actionItems in steps[].actionItems
				that.tutorialConfig.steps[ stepIndex ].actionItems = [];

				var $step = $( makeDiv( ".screen.tutorial-steps" ) ).appendTo( $tutorialContent ).hide();

				var contents = "";

				contents += '<div class="tutorialHeading"><div class="tutorialHeading__inner">' + glo.translations.currentTutorial + that.tutorialConfig.title + '</div></div>';

				var tutorialHeadingSet = false;

				$.each( stepConfig.stepItems, function( index2, stepItem ){
					if( stepItem.type === "imageFull" ){
						contents += '<div class="step__thumb" style="background-image:url(\'' + stepItem.imgUrl + '\');"></div>';
					}

					else if( stepItem.type === "imageFullLarge" ){
						contents += '<div class="step__img"><img src="' + stepItem.imgUrl + '"/></div>';
					}

					else if( stepItem.type === "heading" ){
						//prepend the tutorial heading before the first heading
						if( !tutorialHeadingSet ){
							contents += '<div class="padded-horizontal"><div class="preHeading">' + that.tutorialConfig.title + '</div></div>';
							tutorialHeadingSet = true;
						}

						contents += '<div class="padded-horizontal"><div class="heading">' + ( stepIndex + 1 ) + ". " + stepItem.text + '</div></div>';
					}

					else if( stepItem.type === "text" ){
						contents += '<div class="padded-horizontal"><p>' + stepItem.text + '</p></div>';
					}

					else if( stepItem.type === "listSubsteps" ){
						contents += '<div class="padded-horizontal"><ul class="step__list">';
						$.each( stepItem.listItems, function( index3, listItem ){

							//store action items
							var actionItemMarkup = "";
							if( listItem.checkUserInteractions ){
								that.tutorialConfig.steps[ stepIndex ].actionItems.push( listItem );
								actionItemMarkup = ' class="actionable"';
							}

							contents += '<li ' + actionItemMarkup + '>' + listItem.text + '</li>';
						} );
						contents += '</ul></div>';
					}

					else if( stepItem.type === "list" ){
						contents += '<div class="padded-horizontal"><ul>';
						$.each( stepItem.listItems, function( index3, listItem ){
							contents += '<li>' + listItem.text + '</li>';
						} );
						contents += '</ul></div>';
					}

					else if( stepItem.type === "video" ){
						var autoplay = stepItem.autoplay? "autoplay " : "";
						var loop = stepItem.loop? "loop " : "";

						//don't show controls for looping animations
						var controls = !stepItem.loop? "controls " : "";
						//don't add paddings for looping animations
						var padded = !stepItem.loop? "padded-horizontal" : "";
						//we have a different ratio for the looping animations
						var additionalClasses = stepItem.loop? " video__wrapper--fullwidth" : "";

						contents += '<div class="' + padded + '"><div class="video__wrapper' + additionalClasses + '"><video ' + controls + autoplay + loop + '><source src="' + stepItem.videoUrl + '" type="video/mp4"></video></div></div>';
					}

					else if( stepItem.type === "well" ){

						//default wellType = "info"
						var icon = "fa-info-circle";
						if( stepItem.wellType == "beware" )
							icon = "fa-exclamation-triangle";
						else if( stepItem.wellType == "hint" )
							icon = "fa-lightbulb-o";

						contents += '<div class="padded"><div class="well ' + stepItem.wellType + '">';
						if( stepItem.heading )
							contents += '<div class="heading"><i class="fa ' + icon + '" aria-hidden="true"></i>' + stepItem.heading + '</div>';

						$.each( stepItem.texts, function( index3, text ){
							contents += '<p>' + text + '</p>';
						} );
						contents += '</div></div>';
					}

					else if( stepItem.type === "endScreen" ){
						contents += '<div class="padded-horizontal centered"><div class="heading heading--large">' + stepItem.heading + '</div><div class="iconWrapper"><div class="spriteAnimation spriteAnimation--checkmark" data-sprite-animation-steps="12"></div></div><div class="tutorialEndCta" data-success-next-tutorial-tile data-endscreen-custom-cta="' + ( stepItem.cta? stepItem.cta.replace(/"/g, "&quot;") : "" ) + '" data-endscreen-additional="' + ( stepItem.additional? stepItem.additional.replace(/"/g, "&quot;") : "" ) + '"></div></div>';
					}
				});

				//resolve shortcodes
				contents = contents.replace(/<\$/g, '<span class="appLink">');
				contents = contents.replace(/\$>/g, '</span>');
				contents = contents.replace(/<§/g, '<span class="appLabel">');
				contents = contents.replace(/§>/g, '</span>');
				contents = contents.replace(/<%/g, '<span class="appKey">');
				contents = contents.replace(/%>/g, '</span>');

				//don't put in the project name directly since the user can change the project within the tutorial
				contents = contents.replace(/{{CurrProjectName}}/g, '<span data-curr-project-name>Your Project</span>');

				contents = contents.replace(/{{PanelPayloadsPath}}/g, '<span class="srcpath">'  + glo.panelPayloadsPath + '</span>');
				contents = contents.replace(/{{ProjectPayloadsPath}}/g, '<span class="srcpath">'  + glo.projectPayloadsPath + '</span>');

				$step.append( contents );

				that.$steps.push( $step );
			} );

			//open project if configured in config
			if( typeof this.tutorialConfig.projectUrl !== 'undefined' )
				//open project, then close the other tutorial projects (in this order due to prevent display bug in timeline panel)
				CEPPanel.callFromHtml( 'openProjectAndCloseAndSaveOthers', { projectPath: this.tutorialConfig.projectUrl, tutorialProjectNames: methods.getAllTutorialProjectsNames() } );

			// Add to the last screen info and link of the next tutorial:

			var $currentTutorialEndTile = this.$steps[ this.$steps.length - 1 ].find('*[data-success-next-tutorial-tile]');

			// add custom or default cta-text before the teaser
			var customCta = $currentTutorialEndTile.data( "endscreenCustomCta" );
			$currentTutorialEndTile.empty().append('<p class="p">' + ( customCta? customCta : glo.translations.endScreenMoveToNextTutorial ) + '</p>');

			if( this.nextTutorialConfig ){
				// add teaser, if there's a next tutorial

				var $teaserToNext = $( makeDiv( ".teaser" ) );
				$teaserToNext.append( makeDiv( { "class":"teaser__img", "style":"background-image:url('" + this.nextTutorialConfig.thumbUrl + "');" }, makeDiv( ".btnHolder", makeDiv( ".btn", glo.translations.teasersLetsStart ) ) ) );
				$teaserToNext.append( makeDiv( ".teaser__title", this.nextTutorialConfig.title ) );

				$teaserToNext.on( "click", { tutorialObj: this.nextTutorialConfig.tutorialObj }, methods.onClickNextTeaserTutorialStart );

				$currentTutorialEndTile.append( $teaserToNext );

			}

			if( !this.nextTutorialConfig ){

				$( '<a href="#" class="btn">' + glo.translations.endScreenStartYourOwnProject + '</a>' ).appendTo( $currentTutorialEndTile )
				.on( "click", function(){
					methods.showOnboarding();
				});
			}

			//add info on additional resources like web links
			$currentTutorialEndTile.append( makeDiv(".cta--silent", $currentTutorialEndTile.data( "endscreenAdditional" ) ) );

			if( !this.nextTutorialConfig ){

				$( '<a href="#" class="btn btn--secondary ctaBackToOverview">' + glo.translations.endScreenBackToOverview + '</a>' ).appendTo( $currentTutorialEndTile )
				.on( "click", function(){
					methods.showStartScreen();
				});

			} else {

				$( '<a href="#" class="btn btn--secondary ctaBackToOverview">' + glo.translations.endScreenStartYourOwnProject + '</a>' ).appendTo( $currentTutorialEndTile )
				.on( "click", function(){
					methods.showOnboarding();
				});
			}

			//binds:

			//Open links in browser
			$tutorialContent.find('a[href^="http"]').on( 'click', function( e ){
				e.preventDefault();

				var linkUrl = $( this ).attr( 'href' );

				//trigger event for instrumentation
				CEPPanel.triggerEvent( "com.adobe.learn.events.ClickedHyperlink", {
					TutorialName: that.tutorialConfig.title,
					TutorialPageNumber: that.currentStep + 1,
					HyperlinkDestination: linkUrl
				} );

				glo.environmentVars.csInterface.openURLInDefaultBrowser( linkUrl );
			} );

			//animate sprites
			$tutorialContent.find( '.spriteAnimation' ).each( function(){
				new SpriteAnimation( $( this ) );
			} );

			glo.$startScreen.hide();

			//show first step
			this.showStep( 0 );
		},

		showStep: function( newIndex ){
			var that = this;

			this.cleanupEventListeners();

			var stepConfig = this.tutorialConfig.steps[ newIndex ];
			var $step = this.$steps[ newIndex ];

			//store (before navi!)
			this.currentStep = newIndex;

			//trigger events for instrumentation
			CEPPanel.triggerEvent( "com.adobe.learn.events.TutorialPageDisplayed", {
				TutorialName: this.tutorialConfig.title,
				TutorialPageNumber: this.currentStep + 1
			} );

			//only trigger "TutorialSuccessfullyCompleted" if we are on last step and all steps were solved
			if( newIndex == this.tutorialConfig.steps.length - 1 && this.getAllStepsSolvedState() )
				CEPPanel.triggerEvent( "com.adobe.learn.events.TutorialSuccessfullyCompleted", {
					TutorialName: this.tutorialConfig.title
				} );

			//update navi
			this.updateNavigation();

			//hide other steps
			$.each( this.$steps, function( stepIndex, $step ){
				if( stepIndex != newIndex)
					$step.hide();
			} );

			//handle step preference mods. these will be changed on step show and re-changed back on unload step
			var stepPreferenceMods = stepConfig.stepPreferenceMods || false;
			if( stepPreferenceMods, stepPreferenceMods ){
				$.each( stepPreferenceMods, function( index, stepPreferenceMod ){

					if( typeof stepPreferenceMod.newVal === "string" )
						stepPreferenceMod.newVal = methods.resolveShortcutPaths( stepPreferenceMod.newVal );

					//store orig val to stepConfig.stepPreferenceMods.origVal
					CEPPanel.callFromHtml( stepPreferenceMod.actionName, stepPreferenceMod.newVal, function( prefReturn ){

						var origVal = prefReturn.origVal;

						if( typeof origVal === "string" )
							origVal = origVal.replace( /\\/g, "/" );

						stepPreferenceMods[ index ].origVal = origVal;
					} );

				} );
			}

			//handle step action
			var stepActionsOnShow = stepConfig.stepActionsOnShow || false;
			if( stepActionsOnShow ){

				var i = 0;
				var stepActionsOnShowCallback = function(){
					var stepAction = stepActionsOnShow[ i ];
					CEPPanel.callFromHtml( stepAction.actionName, stepAction.payload, function(){
						i++;

						//recursion
						if( i < stepActionsOnShow.length )
							stepActionsOnShowCallback();
					} );
				};
				stepActionsOnShowCallback();
			}

			//handle item action
			var actionableItems = this.tutorialConfig.steps[ this.currentStep ].actionItems;

			if( actionableItems.length ){
				//reset solved items
				var $actionItems = that.$steps[ that.currentStep ].find( ".actionable" );
				$actionItems.removeClass( "solved" );

				//reset: set step to unsolved
				that.setCurrentStepSolvedState( false );

				this.listenForUserInteractionSequential( actionableItems );
			}

			//replace app vars with current values
			CEPPanel.callFromHtml( 'getCurrProjectName', {}, function( data ){
				if( data )
					$step.find( "*[data-curr-project-name]" ).text( data.replace(/\.[^/.]+$/, "") );
			} );

			//show step
			$step.show();

			//handle autostarts
			$step.find( "video[autoplay]" ).each( function(){
				var video = this;

				//restart video
				video.currentTime = 0;
				//todo: here sometimes a chrome bug appears when very fast skipping through the videos ("The play() request was interrupted by a call to pause()."). no real problem
				video.play();
			} );

			$step.find( ".spriteAnimation" ).each( function(){
				var $sprite = $( this );
				$sprite.trigger( "runSpriteAnimation" );
			} );

		},

		listenForUserInteractionSequential: function( actionableItems, index ){
			var that = this;

			if( typeof index === "undefined")
				index = 0;

			this.cleanupEventListeners();

			var checkUserInteractions = actionableItems[ index ].checkUserInteractions;

			this.listenForUserInteraction( checkUserInteractions, function( autoStepOnComplete ){
				//the actionitem [ index ] was solved

				that.cleanupEventListeners();

				//mark item as solved
				var $actionItem = that.$steps[ that.currentStep ].find( ".actionable" ).eq( index );
				$actionItem.addClass( "solved" );

				//recursive calling: call listening for next action item
				if( index < actionableItems.length - 1 )
					that.listenForUserInteractionSequential( actionableItems, index + 1 );

				//all actionitems are solved. step is acomplished
				else {

					//solve: set step to solved
					that.setCurrentStepSolvedState( true );

					//if the flag autoStepOnComplete is set, then we skip to the next step
					if( autoStepOnComplete )
						that.stepNext();
				}
			} );
		},

		extendPayloadForEffectCheck: function( payload ){

			var paramsMap = {
				"lumetri": {
					//from basic panel:
					"tint": 11,
					"exposure": 14,
					"contrast": 15,
					"highlights": 16,
					"shadows": 17,
					"whites": 18,
					"blacks": 19,
					"hdrSpecular": 20,
					"saturation": 24
					//add here more mappings...
					//use in JSX trackItem.getComponentAt( i ).getParamList() to get a comprehensive list of effects (applied effects only)
				}
			};

			var typeMap = {
				"lumetri": "Video"
				//add here more mappings...
			}

			var effectIdMap = {
				"lumetri": 3
				//add here more mappings...
				//maybe the number depends on the position of the effect in the effects panel? but it is an id, not an index
			};

			var newPayload = {
				effectType: typeMap[ payload.effectName ],
				effectId: effectIdMap[ payload.effectName ],
				effectParameter: paramsMap[ payload.effectName ][ payload.effectParameter ]
			};
			newPayload = $.extend( {}, payload, newPayload );

			return newPayload;
		},

		listenForUserInteraction: function( actions, callbackOnSuccess ){
			/*
			 possible values:

				checkUserInteractions = [
				{
					"check": "checkAudioVideoEffect",
					"autoStepOnComplete": true,
					"payload": {
						"effectName": "lumetri",
						"effectParameter": "alpha",
						"effectParameterValue": 80,
						"effectParameterOperator": "<="

					"check": "projectItemAdded",
					"payload": {
						"itemType": "sequence" //"file"
					}

					"check": "clipAdded",
					"payload": {
						"clipName": "my name"
					}

					"check": "clipMoved",
					"payload": {
						"clipName": "my name"
					}

					"check": "clipsMoved"

					"check": "clipLengthChanged",
					"payload": {
						"clipName": "my name"
					}

					"check": "clipsLengthChanged"

					"check": "clipRemoved",
					"payload": {
						"clipName": "my name"

					"check": "ctiMoved"

					"check": "ctiStopped"

					"check":	"projectItemSelectedMultiple"

					...
				]
			*/

			var that = this;
			var checkResults = [];

			var onBeforeCheck = function(){

				$.each( actions, function( index, action ){

					var payload = action.payload || {};

					//check for audio or video effect
					if( action.check == "checkAudioVideoEffect" ){

						//nothing

					} else if( action.check == "projectItemAdded" ){

						CEPPanel.callFromHtml( 'getProjectItemsCount', { itemType: payload.itemType || false }, function( data ){
							checkResults[ index ] = {
								totalItems: data
							};
						} );

					} else if( action.check == "clipAdded" ){

						CEPPanel.callFromHtml( 'getClipsCount', { clipName: payload.clipName }, function( data ){
							checkResults[ index ] = {
								totalItems: data
							};
						} );

					} else if( action.check == "clipMoved" ){

						CEPPanel.callFromHtml( 'getClipProperties', { clipName: payload.clipName }, function( data ){
							checkResults[ index ] = {
								clipProperties: data
							};
						} );

					} else if( action.check == "clipsMoved" ){

						CEPPanel.callFromHtml( 'getClipsProperties', {}, function( data ){
							checkResults[ index ] = {
								clipsProperties: data
							};
						} );

					} else if( action.check == "clipLengthChanged" ){

						CEPPanel.callFromHtml( 'getClipProperties', { clipName: payload.clipName }, function( data ){
							checkResults[ index ] = {
								clipProperties: data
							};
						} );

					} else if( action.check == "clipsLengthChanged" ){

						CEPPanel.callFromHtml( 'getClipsProperties', {}, function( data ){
							checkResults[ index ] = {
								clipsProperties: data
							};
						} );

					} else if( action.check == "clipRemoved" ){

						CEPPanel.callFromHtml( 'getClipsCount', { clipName: payload.clipName }, function( data ){
							checkResults[ index ] = {
								clipsCount: data
							};
						} );

					} else if( action.check == "ctiMoved" ){

						CEPPanel.callFromHtml( 'getCti', {}, function( data ){
							checkResults[ index ] = {
								ctiInFrames: data
							};
						} );

					} else if( action.check == "ctiStopped" ){

						CEPPanel.callFromHtml( 'getCti', {}, function( data ){
							checkResults[ index ] = {
								ctiInFrames: data
							};
						} );

					} else if( action.check == "sourceCtiMoved" ){

						CEPPanel.callFromHtml( 'getSourceCti', {}, function( data ){
							checkResults[ index ] = {
								ctiInTimecode: data
							};
						} );

					} else if( action.check == "sourceCtiStopped" ){

						CEPPanel.callFromHtml( 'getSourceCti', {}, function( data ){
							checkResults[ index ] = {
								ctiInTimecode: data
							};
						} );

					} else if( action.check == "projectItemSelectedMultiple" ){

						CEPPanel.callFromHtml( 'bindProjectItemSelected', {} );
						glo.isBind.projectItemSelected = true;
					}


				});

			};

			var finalCallback = function( index, totalSuccess ){
				if( index == actions.length - 1 && totalSuccess )
					callbackOnSuccess( actions[ index ].autoStepOnComplete );
			};

			var onEveryCheck = function(){

				var totalSuccess = true;

				//we can have multiple actions to be checked on a step which all have to be true
				$.each( actions, function( index, action ){

					var payload = action.payload || {};

					//check for audio or video effect
					if( action.check == "checkAudioVideoEffect" ){

						payload = that.extendPayloadForEffectCheck( action.payload );

						CEPPanel.callFromHtml( 'getSelectedTrackItemEffectValue', payload, function( data ){

							if( data !== false ){
								var value = data;
								var effectParameterValue = action.payload.effectParameterValue;
								var effectParameterOperator = action.payload.effectParameterOperator;

								if( effectParameterOperator == '<=' && value > effectParameterValue )
									totalSuccess = false;
								else if( effectParameterOperator == '>=' && value < effectParameterValue )
									totalSuccess = false;
								else if(effectParameterOperator == '==' && value != effectParameterValue )
									totalSuccess = false;

							} else {
								totalSuccess = false;
							}

							finalCallback( index, totalSuccess );
						} );

					//check if we have a new project item, e.g. media import or a new sequence created
					} else if( action.check == "projectItemAdded" ){

						CEPPanel.callFromHtml( 'getProjectItemsCount', { itemType: payload.itemType || false }, function( data ){
							var newTotalItems = data;
							var oldTotalItems = checkResults[ index ].totalItems;

							if( newTotalItems <= oldTotalItems )
								totalSuccess = false;

							finalCallback( index, totalSuccess );
						} );

					//check if we added a clip to the timeline
					} else if( action.check == "clipAdded" ){

						CEPPanel.callFromHtml( 'getClipsCount', { clipName: payload.clipName }, function( data ){
							var newTotalItems = data;
							var oldTotalItems = checkResults[ index ].totalItems;

							if( newTotalItems <= oldTotalItems )
								totalSuccess = false;

							finalCallback( index, totalSuccess );
						} );

					//check if we changed clip position on the timeline
					} else if( action.check == "clipMoved" ){

						CEPPanel.callFromHtml( 'getClipProperties', { clipName: payload.clipName }, function( data ){
							var newProperties = data;
							var oldProperties = checkResults[ index ].clipProperties;

							if( !oldProperties || !newProperties ||
								( oldProperties.start.frames == newProperties.start.frames ) )
								totalSuccess = false;

							finalCallback( index, totalSuccess );
						} );

					//check if we changed any clip position on the timeline
					} else if( action.check == "clipsMoved" ){

						CEPPanel.callFromHtml( 'getClipsProperties', {}, function( data ){
							var newClipsProperties = data;
							var oldClipsProperties = checkResults[ index ].clipsProperties;

							totalSuccess = false;

							if( newClipsProperties && oldClipsProperties ){

								for( var j = 0; j < newClipsProperties.length; j++){

									//make sure we have correct values to check for
									if( oldClipsProperties[ j ] && newClipsProperties[ j ] ){

										if( oldClipsProperties[ j ].start.frames != newClipsProperties[ j ].start.frames ){
											totalSuccess = true;
											break;
										}
									}
								}
							}

							finalCallback( index, totalSuccess );
						} );

					//check if we changed clip duration on the timeline
					} else if( action.check == "clipLengthChanged" ){

						CEPPanel.callFromHtml( 'getClipProperties', { clipName: payload.clipName }, function( data ){
							var newProperties = data;
							var oldProperties = checkResults[ index ].clipProperties;

							if( !oldProperties || !newProperties ||
								( oldProperties.start.frames - oldProperties.end.frames == newProperties.start.frames - newProperties.end.frames ) )
								totalSuccess = false;

							finalCallback( index, totalSuccess );
						} );

					//check if we changed any clip duration on the timeline
					} else if( action.check == "clipsLengthChanged" ){

						CEPPanel.callFromHtml( 'getClipsProperties', {}, function( data ){
							var newClipsProperties = data;
							var oldClipsProperties = checkResults[ index ].clipsProperties;

							totalSuccess = false;

							if( newClipsProperties && oldClipsProperties ){
								for( var j = 0; j < newClipsProperties.length; j++){

									//make sure we have correct values to check for
									if( oldClipsProperties[ j ] && newClipsProperties[ j ] ){

										if( oldClipsProperties[ j ].start.frames - oldClipsProperties[ j ].end.frames != newClipsProperties[ j ].start.frames - newClipsProperties[ j ].end.frames ){
											totalSuccess = true;
											break;
										}
									}
								}
							}

							finalCallback( index, totalSuccess );
						} );

					//check if we removed a clip from the timeline
					} else if( action.check == "clipRemoved" ){

						CEPPanel.callFromHtml( 'getClipsCount', { clipName: payload.clipName }, function( data ){
							var oldClipsCount = checkResults[ index ].clipsCount;
							var newClipsCount = data;

							if( oldClipsCount == newClipsCount )
								totalSuccess = false;

							finalCallback( index, totalSuccess );
						} );

					//check if we moved the playhead
					} else if( action.check == "ctiMoved" ){

						CEPPanel.callFromHtml( 'getCti', {}, function( data ){
							var oldCtiInFrames = checkResults[ index ].ctiInFrames;
							var newCtiInFrames = data;

							//no timeline available
							if( newCtiInFrames === false )
								totalSuccess = false;

							//it was no timeline available, no it is: store
							else if( oldCtiInFrames === false ){
								oldCtiInFrames = newCtiInFrames;
								totalSuccess = false;
							}

							//make sure we have correct values to check for
							if( oldCtiInFrames !== false && newCtiInFrames !== false )

								if( oldCtiInFrames == newCtiInFrames )
									totalSuccess = false;

							finalCallback( index, totalSuccess );
						} );


					//check if we stopped the playhead
					} else if( action.check == "ctiStopped" ){

						CEPPanel.callFromHtml( 'getCti', {}, function( data ){
							var oldCtiInFrames = checkResults[ index ].ctiInFrames;
							var newCtiInFrames = data;

							//make sure we have correct values to check for
							if( oldCtiInFrames !== false && newCtiInFrames !== false )

								if( oldCtiInFrames != newCtiInFrames ){
									totalSuccess = false;
									//store new position
									checkResults[ index ].ctiInFrames = newCtiInFrames;
								}


							finalCallback( index, totalSuccess );
						} );

					//check if we moved the source playhead
					} else if( action.check == "sourceCtiMoved" ){

						CEPPanel.callFromHtml( 'getSourceCti', {}, function( data ){
							var oldCtiInTimecode = checkResults[ index ].ctiInTimecode;
							var newCtiInTimecode = data;

							//make sure we have correct values to check for
							if( oldCtiInTimecode !== false && newCtiInTimecode !== false )

								if( oldCtiInTimecode == newCtiInTimecode )
									totalSuccess = false;

							finalCallback( index, totalSuccess );
						} );


					//check if we stopped the source playhead
					} else if( action.check == "sourceCtiStopped" ){

						CEPPanel.callFromHtml( 'getSourceCti', {}, function( data ){
							var oldCtiInTimecode = checkResults[ index ].ctiInTimecode;
							var newCtiInTimecode = data;

							//make sure we have correct values to check for
							if( oldCtiInTimecode !== false && newCtiInTimecode !== false )

								if( oldCtiInTimecode != newCtiInTimecode ){
									totalSuccess = false;
									//store new position
									checkResults[ index ].ctiInTimecode = newCtiInTimecode;
								}


							finalCallback( index, totalSuccess );

						} );

					//check if there were multiple project items selected
					} else if( action.check == "projectItemSelectedMultiple" ){

						CEPPanel.callFromHtml( 'getSelectedProjectItems', {}, function( data ){
							if( data < 2 )
								totalSuccess = false;

							else {
								CEPPanel.callFromHtml( 'unbindProjectItemSelected', {} );
								glo.isBind.projectItemSelected = false;
							}

							finalCallback( index, totalSuccess );
						} );

					}

				});

			};

			//direct call
			onBeforeCheck();

			this.cleanupEventListeners( true ); //keep the binds!

			glo.eventListenerInterval = setInterval( function(){
				onEveryCheck();
			}, 200 );
		},

		updateNavigation: function(){
			glo.$navigation.show();

			glo.$stepNavigation.html( ( this.currentStep + 1 ) + "/" + this.$steps.length );

			methods.toggleDisplay( glo.$navBtnNextTutorial, this.currentStep == this.$steps.length - 1 && this.nextTutorialConfig );
			methods.toggleDisplay( glo.$navBtnPrevTutorial, this.currentStep == 0 && this.prevTutorialConfig );
			methods.toggleDisplay( glo.$navBtnNextStep, this.currentStep < this.$steps.length - 1 );
			methods.toggleDisplay( glo.$navBtnPrevStep, this.currentStep > 0 );
		},

		cleanupEventListeners: function( keepBinds ){
			//cleanup if there were listeners
			if( glo.eventListenerInterval )
				clearInterval( glo.eventListenerInterval );

			if( !keepBinds )
				//since we are partly using JSX.bind() to listen for things: unbind it
				if( glo.isBind.projectItemSelected )
					CEPPanel.callFromHtml( 'unbindProjectItemSelected', {} );
		},

		onUnloadTutorial: function( callback ){
			this.onUnloadStep();

			this.cleanupEventListeners();

			callback();
		},

		onUnloadStep: function(){
			this.cleanupEventListeners();

			//reset step preference mods
			var stepConfig = this.tutorialConfig.steps[ this.currentStep ];
			var stepPreferenceMods = stepConfig.stepPreferenceMods || false;
			if( stepPreferenceMods ){

				$.each( stepPreferenceMods, function( index, stepPreferenceMod ){
					if( typeof stepPreferenceMod.origVal !== "undefined" )
						CEPPanel.callFromHtml( stepPreferenceMod.actionName, stepPreferenceMod.origVal );
				} );
			}

			//handle step actions
			var stepActionsOnUnload = stepConfig.stepActionsOnUnload || false;
			if( stepActionsOnUnload ){
				$.each( stepActionsOnUnload, function( index, stepAction ){
					CEPPanel.callFromHtml( stepAction.actionName, stepAction.payload );
				} );
			}

			//stop all videos
			var $step = this.$steps[ this.currentStep ];
			$step.find( "video" ).each( function(){
				var video = this;
				video.pause();
			} );
		},

		stepNext: function(){
			//trigger event for instrumentation
			CEPPanel.triggerEvent( "com.adobe.learn.events.ClickedNext", {
				TutorialName: this.tutorialConfig.title,
				TutorialPageNumber: this.currentStep + 1,
				PageSuccessfullyCompleted: this.getCurrentStepSolvedState()
			} );

			this.onUnloadStep();
			this.showStep( this.currentStep + 1 );
		},

		stepPrev: function(){
			this.onUnloadStep();
			this.showStep( this.currentStep - 1 );
		},

		//solved = true, false
		setCurrentStepSolvedState: function( solved ){
			this.tutorialConfig.steps[ this.currentStep ].solved = solved;
		},

		// returns true, if current step is solved or has no solvable actions
		getCurrentStepSolvedState: function(){
			var stepConfig = this.tutorialConfig.steps[ this.currentStep ];
			return typeof stepConfig.solved !== "undefined" ? stepConfig.solved : true;
		},

		// returns true, if all steps are solved or have no solvable actions
		getAllStepsSolvedState: function(){
			var allSolved = true;

			$.each( this.tutorialConfig.steps, function( index, stepConfig ){
				if( typeof stepConfig.solved !== "undefined" && stepConfig.solved === false ){
					allSolved = false;
					return false;
				}
			} );

			return allSolved;
		}

	};


	//helper for jQuery
	$.fn.filterByData = function( prop, val ) {
		return this.filter(
			function() {
				return $( this ).data( prop ) == val;
			}
		);
	}


	return {
		resolveShortcutPaths: methods.resolveShortcutPaths,
		getCurrentTutorialObj: methods.getCurrentTutorialObj,
		getAllTutorialProjectsNames: methods.getAllTutorialProjectsNames,
		getAllTutorialProjectsPaths: methods.getAllTutorialProjectsPaths,
		getFilenameFromPath: methods.getFilenameFromPath
	};

} )();