// Copyright 2000-2021 JetBrains s.r.o.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.jetbrains.php.lang.psi.resolve.types;

import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.NlsSafe;
import com.intellij.openapi.util.Ref;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiElement;
import com.intellij.util.containers.CollectionFactory;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.text.CaseInsensitiveStringHashingStrategy;
import com.jetbrains.php.PhpClassHierarchyUtils;
import com.jetbrains.php.PhpIndex;
import com.jetbrains.php.lang.psi.elements.PhpClass;
import com.jetbrains.php.lang.psi.elements.PhpTypedElement;
import com.jetbrains.php.lang.psi.elements.Variable;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.intellij.openapi.util.text.StringUtil.startsWithChar;

public class PhpType {
  private static final Logger LOG = Logger.getInstance(PhpType.class);

  public static final @NlsSafe String PHPSTORM_HELPERS = "___PHPSTORM_HELPERS";
  public static final @NlsSafe String _PHPSTORM_HELPERS_FQN = "\\___PHPSTORM_HELPERS";

  public static final @NlsSafe String _OBJECT_FQN = "\\" + PHPSTORM_HELPERS + "\\object"; //TODO - USAGES?
  //public static final @NlsSafe String _PHPSTORM_HELPERS_STATIC = "\\" + PHPSTORM_HELPERS + "\\static";
  //public static final @NlsSafe String _PHPSTORM_HELPERS_$THIS = "\\" + PHPSTORM_HELPERS + "\\this";

  public static final @NlsSafe String _OBJECT = "\\object";
  public static final @NlsSafe String _MIXED = "\\mixed";
  public static final @NlsSafe String _VOID = "\\void";
  public static final @NlsSafe String _NULL = "\\null";
  public static final @NlsSafe String _ARRAY = "\\array";
  public static final @NlsSafe String _ITERABLE = "\\iterable";
  public static final @NlsSafe String _INT = "\\int";
  public static final @NlsSafe String _INTEGER = "\\integer";
  public static final @NlsSafe String _BOOL = "\\bool";
  public static final @NlsSafe String _BOOLEAN = "\\boolean";
  public static final @NlsSafe String _TRUE = "\\true";
  public static final @NlsSafe String _FALSE = "\\false";
  public static final @NlsSafe String _STRING = "\\string";
  public static final @NlsSafe String _FLOAT = "\\float";
  public static final @NlsSafe String _DOUBLE = "\\double";
  public static final @NlsSafe String _CLOSURE = "\\Closure";
  /**
   * use CALLABLE in our code, but doc literal should be OK
   */
  public static final @NlsSafe String _CALLBACK = "\\callback";
  public static final @NlsSafe String _CALLABLE = "\\callable";
  public static final @NlsSafe String _NUMBER = "\\number";
  public static final @NlsSafe String _RESOURCE = "\\resource";
  public static final @NlsSafe String _EXCEPTION = "\\Exception";
  public static final @NlsSafe String _THROWABLE = "\\Throwable";

  //primitive types
  public static final PhpType EMPTY = builder().build();
  /**
   * This type can be used to indicate uncertainty about inferred type.
   * Most of the type-based inspections won't (and should not) highlight any error on the element with this type.
   */
  public static final PhpType MIXED = builder().add(_MIXED).build();
  public static final PhpType NULL = builder().add(_NULL).build();
  public static final PhpType STRING = builder().add(_STRING).build();
  /**
   * @see #isBoolean()
   */
  public static final PhpType BOOLEAN = builder().add(_BOOL).build();
  public static final PhpType FALSE = builder().add(_FALSE).build();
  public static final PhpType TRUE = builder().add(_TRUE).build();
  public static final PhpType INT = builder().add(_INT).build();
  public static final PhpType FLOAT = builder().add(_FLOAT).build();
  public static final PhpType OBJECT = builder().add(_OBJECT).build();
  public static final PhpType CLOSURE = builder().add(_CLOSURE).build();
  public static final PhpType CALLABLE = builder().add(_CALLABLE).build();
  public static final PhpType RESOURCE = builder().add(_RESOURCE).build();
  /**
   * Represents an array of elements with unknown type, {@link #elementType()} on this type will produce {@link #MIXED}.
   * Array of elements with specific types can be represented using {@link #pluralise()}.
   * Consider to use {@link #isArray(PhpType)} to check both conditions: whether the type is exactly this instance or plural one.
   */
  public static final PhpType ARRAY = builder().add(_ARRAY).build();
  public static final PhpType ITERABLE = builder().add(_ITERABLE).build();

  //aliases
  public static final PhpType NUMBER = builder().add(_NUMBER).build();

  //fake types
  public static final PhpType VOID = builder().add(_VOID).build();

  //complex types
  public static final PhpType NUMERIC = builder().add(STRING).add(INT).build();
  public static final PhpType SCALAR = builder().add(INT).add(FLOAT).add(STRING).add(BOOLEAN).add(FALSE).build();
  public static final PhpType FLOAT_INT = builder().add(FLOAT).add(INT).build();

  public static final PhpType UNSET = builder().add("unset").build();
  public static final PhpType STATIC = builder().add(PhpClass.STATIC).build();
  public static final PhpType EXCEPTION = builder().add(_EXCEPTION).build();
  public static final PhpType THROWABLE = builder().add(_THROWABLE).build();
  public static final PhpType $THIS = builder().add(Variable.$THIS).build();

  private static final @NlsSafe String _TRAVERSABLE = "\\Traversable";
  public static final PhpType TRAVERSABLE = new PhpType().add(_TRAVERSABLE);
  public static final PhpType ARRAY_TRAVERSABLE_TYPE = new PhpType().add(_ARRAY).add(_TRAVERSABLE);
  public static final String EXCLUDED_INCOMPLETE_TYPE_SEPARATOR = "∆";

  public PhpType createImmutableType() {
    return this instanceof ImmutablePhpType ? this : new ImmutablePhpType().addInternal(this);
  }

  @NotNull
  public PhpType map(@NotNull Function<@NotNull String, @NotNull String> typeMapper) {
    if (types == null || isEmpty()) {
      return this;
    }
    PhpType res = new PhpType();
    for (String type : types) {
      res.add(typeMapper.apply(type));
    }
    return res;
  }

  public static @NlsSafe String unpluralize(@NotNull  @NlsSafe String type, int dimension) {
    return type.substring(0, type.length() - dimension * 2);
  }

  public static @NlsSafe String pluralise(@NotNull @NlsSafe String type, int dimension) {
    return type + StringUtil.repeat("[]", dimension);
  }

  public static boolean isPrimitiveClassAccess(@NlsSafe String subject) {
    String originalSubject = subject;
    while (startsWithChar(subject, '#')) {
      char key = subject.charAt(1);
      if (PhpTypeSignatureKey.FIELD.is(key) || PhpTypeSignatureKey.METHOD.is(key)) {
        int i = subject.lastIndexOf('.');
        if (i < 0) {
          LOG.warn(String.format("Invalid subject: %s\nProcessed subject: %s", originalSubject, subject));
          return false;
        }
        subject = subject.substring(2, i);
      }
      else {
        break;
      }
    }
    return startsWithChar(subject, '#') && PhpTypeSignatureKey.CLASS.is(subject.charAt(1)) && isPrimitiveType(subject.substring(2));
  }

  public static @NotNull String unpluralize(@NotNull String type) {
    return unpluralize(type, getPluralDimension(type));
  }

  public interface PhpTypeExclusion {
    boolean isNotApplicableType(@Nullable Project project, @NotNull String type);

    @NlsSafe String getCode();

    default boolean filterOnlyUnresolved() {
      return true;
    }
  }

  public enum ExcludeCode implements PhpTypeExclusion {
    NOT_NULL("n") {
      @Override
      public boolean isNotApplicableType(@Nullable Project project, @NotNull @NlsSafe String type) {
        return isNull(type);
      }
    },
    NOT_PRIMITIVE("p") {
      @Override
      public boolean isNotApplicableType(@Nullable Project project, @NotNull @NlsSafe String type) {
        return !isMixedType(type) && (isArray(type) || isNotExtendablePrimitiveType(type));
      }
    }

    ,NOT_FALSE("f") {
      @Override
      public boolean isNotApplicableType(@Nullable Project project, @NotNull @NlsSafe String type) {
        return _FALSE.equals(type);
      }
    }

    ,NOT_MIXED("m") {
      @Override
      public boolean isNotApplicableType(@Nullable Project project, @NotNull @NlsSafe String type) {
        return isMixedType(type);
      }
    }

    ,NOT_OBJECT("o") {
      @Override
      public boolean isNotApplicableType(@Nullable Project project, @NotNull @NlsSafe String type) {
        return isObject(type);
      }
    }
    ;

    @NotNull
    private final String myCode;

    ExcludeCode(@NotNull String code) {
      myCode = code;
    }
    @Override
    public abstract boolean isNotApplicableType(@Nullable Project project, @NotNull @NlsSafe String type);

    @Override
    public @NotNull String getCode() {
      return myCode;
    }
  }

  @Nullable
  public static PhpType.PhpTypeExclusion fromCode(@NotNull String code) {
    return ContainerUtil.find(ExcludeCode.values(), v -> v.getCode().equals(code));
  }

  @Nullable
  private Set<String> types;
  private boolean isComplete = true;
  private boolean dirty = false;
  private @NlsSafe String myStringResolved;
  private @NlsSafe String myString;

  public PhpType() {
  }

  public boolean isBoolean() {
    return !isEmpty() && filterOut(s -> _BOOL.equals(s) || _TRUE.equals(s) || _FALSE.equals(s)).isEmpty();
  }

  public static boolean isArray(@Nullable PhpType type) {
    if (type == null) return false;
    if (type.isEmpty()) return false;
    return type.getTypes().stream().allMatch(t -> isArray(t) || isPluralType(t));
  }

  @NotNull
  public static PhpTypeBuilder builder() {
    return new PhpTypeBuilder();
  }

  @NotNull
  public PhpType add(@Nullable @NlsSafe String aClass) {
    if (aClass == null || aClass.length() <= 0) {
      return this;
    }

    if (aClass.length() > 1 && aClass.charAt(0) == '#') {
      isComplete = false;
    }

    String trimmed = aClass;
    while (StringUtil.endsWith(trimmed, "[]")) {
      trimmed = StringUtil.trimEnd(trimmed, "[]");
    }
    if (isPrimitiveType(trimmed) && !aClass.startsWith("\\")) {
      aClass = "\\" + aClass; //intern will help here
    }
    if (aClass.equalsIgnoreCase(_INTEGER)) {
      aClass = _INT;
    }
    else if (aClass.equals(_STRING)) {
      aClass = _STRING;
    }
    else if (aClass.equalsIgnoreCase(_ARRAY)) {
      aClass = _ARRAY;
    }
    else if (aClass.equalsIgnoreCase(_BOOL)) {
      aClass = _BOOL;
    }
    else if (aClass.equalsIgnoreCase(_MIXED)) {
      aClass = _MIXED;
    }
    else if (aClass.equalsIgnoreCase(_BOOLEAN)) {
      aClass = _BOOL;
    }
    else if (aClass.equalsIgnoreCase(_CALLBACK)) {
      aClass = _CALLABLE;
    }
    else if (aClass.equalsIgnoreCase(_ITERABLE)) {
      aClass = _ITERABLE;
    }
    else if (aClass.equalsIgnoreCase(_DOUBLE)) {
      aClass = _FLOAT;
    }
    if (types == null) {
      types = CollectionFactory.createCaseInsensitiveStringSet();
    }
    if (types.size() > 50) {
      if (ApplicationManager.getApplication().isInternal()) {
        LOG.warn("too much type variants: " + types);
      }
    }
    else {
      types.add(aClass);
    }
    dirty = true;
    return this;
  }

  @NotNull
  public PhpType add(@Nullable PsiElement other) {
    if(other instanceof PhpTypedElement) {
      final PhpType type = ((PhpTypedElement)other).getType();
      add(type);
    }
    return this;
  }

  public @NotNull PhpType add(PhpType type) {
    if (type == null || type.types == null || type.types.size() <= 0) {
      return this;
    }

    try {
      isComplete &= type.isComplete;
      if (types == null) {
        types = CollectionFactory.createCaseInsensitiveStringSet(type.types);
      }
      else {
        types.addAll(type.types);
        if (types.size() > 50 && ApplicationManager.getApplication().isInternal()) {
          LOG.warn("too much type variants: " + types);
        }
      }
      dirty = true;
    }
    catch (NoSuchElementException e) {
      throw new RuntimeException("NSEE @" + type.types, e);
    }
    return this;
  }

  public int size() {
    return types!=null ? types.size() : 0;
  }

  public boolean isAmbiguous() {
    return isEmpty() || hasUnknown() || intersects(this, MIXED);
  }

  /**
   * @deprecated Use {@link #isAmbiguous()}
   */
  @Deprecated
  public boolean isUndefined() {
    return isAmbiguous();
  }

  /**
   * @return UNORDERED set of contained type literals
   */
  @NotNull
  public Set<@NlsSafe String> getTypes() {
    return removeFalseIfNeeded(types);
  }

  @NotNull
  private static Set<@NlsSafe String> removeFalseIfNeeded(Set<@NlsSafe String> types) {
    if(types != null) {
      if (types.contains(_FALSE) && types.contains(_BOOL)) {
        Set<String> typesCopy = new HashSet<>(types);
        typesCopy.remove(_FALSE);
        return typesCopy;
      }
      return types;
    } else return Collections.emptySet();
  }

  /**
   * @return more costly, ORDERED set of contained type literals (cached), should be avoided
   * TODO - factor usages out?
   */
  public @NotNull Set<@NlsSafe String> getTypesSorted() {
    if (types == null) {
      return Collections.emptySet();
    }
    else {
      sortIfNeeded();
      assert types != null;
      return removeFalseIfNeeded(types);
    }
  }

  public String toString() {
    if(types==null) return "";
    if(!dirty && myString!=null) return myString;
    String typesString = sortedResolvedTypes().collect(Collectors.joining("|"));
    return myString = isComplete ? typesString : typesString + "|?";
  }

  /**
   * @param type string to trim-root-ns-if-required
   * @return trimmed string
   */
  public static String toString(@NlsSafe String type) {
    return (isPrimitiveType(StringUtil.trimEnd(type, "[]")) && type.startsWith("\\")) ? type.substring(1) : type;
  }

  private void sortIfNeeded() {
    if (types != null && !(types instanceof SortedSet)) {
      Set<String> sorted = new TreeSet<>(String.CASE_INSENSITIVE_ORDER);
      sorted.addAll(types);
      types = sorted;
    }
  }

  /**
   * NOTE: apply to .global() type only!
   * @return presentable name
   */
  public @NlsSafe String toStringResolved() {
    //PhpContractUtil.assertCompleteType(this);
    if (!dirty && myStringResolved != null) {
      return myStringResolved;
    }
    myStringResolved = toStringRelativized(null);
    return myStringResolved;
  }

  /**
   * NOTE: apply to .global() type only!
   * @return relativized presentable name
   */
  public @NlsSafe String toStringRelativized(@Nullable @NlsSafe final String currentNamespaceName) {
    //PhpContractUtil.assertCompleteType(this);
    if (types == null) {
      return "";
    }
    final StringBuilder builder = new StringBuilder();
    for (String type : getSortedTypesStrings(currentNamespaceName)) {
      builder.append(type).append('|');
    }
    if (builder.length() > 1) {
      builder.setLength(builder.length() - 1);
    }
    if (!isComplete) {
      builder.append("|?");
    }
    return builder.toString();
  }

  @NotNull
  public Collection<@NlsSafe String> getSortedTypesStrings(@Nullable @NlsSafe String currentNamespaceName) {
    return sortedResolvedTypes()
      .map(t -> currentNamespaceName != null && t.startsWith(currentNamespaceName) ? t.substring(currentNamespaceName.length()) : t)
      .collect(Collectors.toList());
  }

  @NotNull
  private Stream<@NlsSafe String> sortedResolvedTypes() {
    return getTypesSorted().stream()
      .filter(t -> !startsWithChar(t, '?'))
      .map(PhpType::toString);
  }

  public boolean isConvertibleFromGlobal(Project project, @NotNull PhpType type) {
    if (isConvertibleLocal(project, type)) return true;
    return global(project).isConvertibleFrom(type.global(project), PhpIndex.getInstance(project));
  }

  public boolean isConvertibleLocal(Project project, @NotNull PhpType type) {
    if (hasUnknown() || type.hasUnknown()) {
      PhpType t1 = filterUnknown();
      PhpType t2 = type.filterUnknown();
      if (!t1.isEmpty() && !t2.isEmpty() && t1.isConvertibleFrom(type, PhpIndex.getInstance(project))) {
        return true;
      }
    }
    return false;
  }


  public boolean isConvertibleFrom(@NotNull PhpType otherType, @NotNull PhpIndex index) {
    if(isAmbiguous() || otherType.isAmbiguous()) return true;
    if(this.equals(NULL)) return true;
    Set<String> otherTypes = otherType.types;
    if(otherTypes == null) return types == null;
    for (String other : otherTypes) {
      if (types != null) {
        for (final String my : types) {
          if (isPluralType(my) && isPluralType(other)) {
            boolean elementTypesConvertible =
              new PhpType().add(my).unpluralize().isConvertibleFrom(new PhpType().add(other).unpluralize(), index);
            if (elementTypesConvertible) return true;
          }
          if (my.equalsIgnoreCase(other)
             || isPluralType(my) && other.equalsIgnoreCase(_ARRAY)
             || isPluralType(other) && (my.equalsIgnoreCase(_ARRAY) || my.equalsIgnoreCase(_ITERABLE))
             || (my.equalsIgnoreCase(_STRING) && !other.equalsIgnoreCase(_ARRAY) && !isPluralType(other) && !nonPrimitiveWithoutToString(other, index))
             || (other.equalsIgnoreCase(_STRING) && my.equalsIgnoreCase(_CALLABLE))
             || (other.equalsIgnoreCase(_ARRAY) && my.equalsIgnoreCase(_CALLABLE))
             || (isPluralType(other) && my.equalsIgnoreCase(_CALLABLE))
             || (other.equalsIgnoreCase(_STRING) && my.equalsIgnoreCase(_INT))
             || (other.equalsIgnoreCase(_STRING) && my.equalsIgnoreCase(_FLOAT))
             || (other.equalsIgnoreCase(_STRING) && my.equalsIgnoreCase(_BOOL))
             || (other.equalsIgnoreCase(_STRING) && my.equalsIgnoreCase(_FALSE))
             || my.equalsIgnoreCase(_ITERABLE) && other.equalsIgnoreCase(_ARRAY)
             || my.equalsIgnoreCase(_ITERABLE) && isPluralType(other)
             || isBidi(my, other, _TRUE, _BOOL)
             || isBidi(my, other, _FALSE, _BOOL)
             || isBidi(my, other, _INT, _FLOAT)
             || isBidi(my, other, _BOOL, _INT)
             || isBidi(my, other, _FALSE, _INT)
             || isBidi(my, other, _BOOL, _FLOAT)
             || isBidi(my, other, _FALSE, _FLOAT)
             || isBidi(my, other, _NUMBER, _FLOAT)
             || isBidi(my, other, _NUMBER, _INT)
             || isBidi(my, other, _OBJECT, "\\stdClass")
            ) return true;
          if (!my.equalsIgnoreCase(_CALLABLE) && !other.equalsIgnoreCase(_CALLABLE) && isPrimitiveType(my) && isPrimitiveType(other)) {
            continue;
          }
          if (findSuper(my, other, index) || isSuperWithOnlyPolymorphicTarget(my, other, index)) return true;
          if (!isPluralType(other) && !isPrimitiveType(other) && _OBJECT.equalsIgnoreCase(my)) return true;
          if (my.equalsIgnoreCase(_CALLABLE) && checkInvoke(other, index)
              || other.equalsIgnoreCase(_CALLABLE) && checkInvoke(my, index)) {
            return true;
          }
        }
      }
    }
    //System.out.println("\tisConvertibleFrom == false");
    return false;
  }

  private static boolean isSuperWithOnlyPolymorphicTarget(String my, String other, @NotNull PhpIndex index) {
    Collection<PhpClass> classes = index.getAnyByFQN(other);
    if (!classes.isEmpty() && ContainerUtil.and(classes, PhpClass::isAbstract)) {
      List<PhpClass> subClasses = index.getAllSubclasses(other).stream()
        .filter(c -> !c.isAbstract())
        .limit(2).collect(Collectors.toList());
      PhpClass onlySubClass = ContainerUtil.getOnlyItem(subClasses);
      return onlySubClass != null && findSuper(my, onlySubClass.getFQN(), index);
    }
    return false;
  }

  @ApiStatus.Internal
  public static boolean nonPrimitiveWithoutToString(String fqn, @NotNull PhpIndex index) {
    if (isPluralType(fqn) || isNotExtendablePrimitiveType(fqn)) {
      return false;
    }
    Collection<PhpClass> classes = index.getAnyByFQN(fqn);
    return !classes.isEmpty() && !hasToString(classes) && !hasToString(index.getAllSubclasses(fqn));
  }

  private static boolean hasToString(Collection<PhpClass> classes) {
    return ContainerUtil.exists(classes, c -> c.findMethodByName(PhpClass.TO_STRING) != null);
  }

  private static boolean isBidi(@NlsSafe String my, @NlsSafe String other, @NlsSafe String type1, @NlsSafe String type2) {
    return (my.equalsIgnoreCase(type1) && other.equalsIgnoreCase(type2)) ||
           (other.equalsIgnoreCase(type1) && my.equalsIgnoreCase(type2));
  }

  @NotNull
  public PhpType filterUnknown() {
    return filterOut(s -> s.startsWith("#") || s.startsWith("?"));
  }

  @NotNull
  public PhpType filterPrimitives() {
    return filterOut(PhpType::isPrimitiveType);
  }

  public boolean hasUnknown() {
    if(types==null) return false;
    for (String type : types) {
      if (StringUtil.startsWith(type, "?")) {
        return true;
      }
    }
    return false;
  }

  private static boolean checkInvoke(@NotNull @NlsSafe String some, @NotNull PhpIndex index) {
    final Collection<PhpClass> candidates = index.getAnyByFQN(some);
    for (PhpClass candidate : candidates) {
      if(candidate.findMethodByName(PhpClass.INVOKE)!=null) return true;
    }
    return false;
  }

  //TODO move to hierarchy utils
  public static boolean findSuper(@NotNull @NlsSafe String mySuper, @Nullable @NlsSafe String otherChild, @NotNull PhpIndex index) {
    if (otherChild == null) return false;
    if (mySuper.endsWith("[]") != otherChild.endsWith("[]")) return false;
    mySuper = StringUtil.trimEnd(mySuper, "[]");
    otherChild = StringUtil.trimEnd(otherChild, "[]");
    if (!mySuper.startsWith("\\")) mySuper = "\\" + mySuper;
    if (!otherChild.startsWith("\\")) otherChild = "\\" + otherChild;
    if (mySuper.equalsIgnoreCase(otherChild)) {
      return true;
    }
    Collection<PhpClass> mes = index.getAnyByFQN(mySuper);
    Ref<Boolean> result = new Ref<>(false);
    return index.getAnyByFQN(otherChild).stream().anyMatch(phpClass -> {
      PhpClassHierarchyUtils.processSuperWithoutMixins(phpClass, true, true, aSuper -> {
        for (final PhpClass me : mes) {
          if (PhpClassHierarchyUtils.classesEqual(me, aSuper)) {
            result.set(true);
            break;
          }
        }
        return !result.get();
      });
      return result.get();
    });
  }

  /**
   * @return type info is complete from all possible sources/is incomplete - non local stuff omitted. Defaults to true.
   */
  public boolean isComplete() {
    return isComplete;
  }

  /**
   * @param context - important for resolution of SELF, THIS, STATIC
   * @deprecated just use .global
   * @return resolved type
   */
  @Deprecated
  public PhpType globalLocationAware(@NotNull PsiElement context) {
    return global(context.getProject());
  }

  public PhpType global(@NotNull Project p) {
    try {
      return PhpIndex.getInstance(p).completeType(p, this, null);
    }
    catch (StackOverflowError e) {
      LOG.warn("SOE in PhpType.global @ " + this);
      return EMPTY;
    }
  }

  public boolean isEmpty() {
    return types==null || types.isEmpty();
  }

  @NotNull
  public PhpType elementType() {
    return elementType(PhpTypeSignatureKey.ARRAY_ELEMENT);
  }

  @ApiStatus.Internal
  @NotNull
  public PhpType elementType(PhpTypeKey typeKey) {
    final PhpType elementType = new PhpType();
    PhpType keysType = new PhpType();
    if (types != null) {
      for (String type : types) {
        if (PhpKeyTypeProvider.isArrayKeySignature(type)) {
          String unsignedKeyType = type.substring(2);
          if (PhpKeyTypeProvider.isArrayKeySignature(unsignedKeyType)) {
            keysType.add(unsignedKeyType);
          }
          continue;
        }
        if (isPrimitiveClassAccess(type)) {
          elementType.add(_MIXED);
          continue;
        }
        if (type.equalsIgnoreCase(_ARRAY)) {
          elementType.add(MIXED);
        }
        else if (isPluralType(type) && !type.contains("#π")) {
          elementType.add(type.substring(0, type.length() - 2));
        }
        else if (isPrimitiveType(type)) {
          if (type.equals(_STRING)) {
            elementType.add(STRING);
          }
          else {
            elementType.add(MIXED);
          }
        }
        else {
          elementType.add(typeKey.sign(PhpTypeSignatureKey.CLASS.signIfUnsigned(type)));
        }
      }
    }
    if (elementType.isEmpty()) elementType.add(_MIXED);
    return elementType.add(keysType);
  }

  @NotNull
  public PhpType pluralise() {
    return pluralise(1);
  }

  public PhpType pluralise(int c) {
    if (c == 0) {
      return this;
    }
    PhpType elementType = new PhpType();
    if (types != null) {
      for (String type : types) {
        String pluralise = pluraliseMixedAware(type, c);
        elementType.add(pluralise);
      }
    }
    if (elementType.isEmpty()) {
      elementType.add(ARRAY);
    }
    return elementType;
  }

  @NotNull
  public static String pluraliseMixedAware(String type, int c) {
    return _MIXED.equals(type) ? pluralise(_ARRAY, c - 1) : pluralise(type, c);
  }

  @NotNull
  public PhpType unpluralize() {
    if (ContainerUtil.isEmpty(types)) return getEmpty();
    final PhpType unpluralized = new PhpType();
    for (final String type : types) {
      unpluralized.add(_ARRAY.equalsIgnoreCase(type) ? _MIXED : StringUtil.trimEnd(type, "[]"));
    }
    return unpluralized;
  }

  public static boolean isPrimitiveType(@Nullable String type) {
    if(type == null) return true;
    if(type.length() < 3 || type.length() > 11) return false;
    if(type.charAt(0) == '#') return false;
    if(!type.startsWith("\\")) type = "\\" + type;
    return isNotExtendablePrimitiveType(type) ||
           isArray(type) ||
           _OBJECT.equalsIgnoreCase(type) ||
           _CALLABLE.equalsIgnoreCase(type) ||
           _ITERABLE.equalsIgnoreCase(type);
  }

  /**
   * Some primitive types are not exclusive, e.g. variable can be an {@code array} and {@code ArrayAccess} class at the same time
   * or {@code callable} and any other class type with implemented {@code __invoke} method. <br/><br/>
   *
   * This method is indented to check if type consists of exclusive primitive types,
   * i.e. types that can't be extended by class types
   * @return {@code true} if type consists of exclusive primitive types, {@code false} otherwise
   */
  public boolean isNotExtendablePrimitiveType() {
    if (isEmpty()) return false;
    return getTypes().stream().allMatch(PhpType::isNotExtendablePrimitiveType);
  }

  public static boolean isNotExtendablePrimitiveType(@Nullable @NlsSafe String type) {
    if(type == null) return true;
    if(type.length() < 3 || type.length() > 11) return false;
    if(type.charAt(0) == '#') return false;
    if(!type.startsWith("\\")) type = "\\" + type;
    return
      _MIXED.equalsIgnoreCase(type) ||
      _STRING.equalsIgnoreCase(type) ||
      _INT.equalsIgnoreCase(type) ||
      _INTEGER.equalsIgnoreCase(type) ||
      _NUMBER.equalsIgnoreCase(type) ||
      _BOOL.equalsIgnoreCase(type) ||
      _BOOLEAN.equalsIgnoreCase(type) ||
      _TRUE.equalsIgnoreCase(type) ||
      _FALSE.equalsIgnoreCase(type) ||
      _FLOAT.equalsIgnoreCase(type) ||
      _NULL.equalsIgnoreCase(type) ||
      _RESOURCE.equalsIgnoreCase(type) ||
      _VOID.equalsIgnoreCase(type) ||
      _DOUBLE.equalsIgnoreCase(type)
      ;
  }

  public static boolean isArray(@NotNull final @NlsSafe String type){
    return _ARRAY.equals(type);
  }

  public static boolean isString(@NotNull final @NlsSafe String type){
    return _STRING.equals(type);
  }

  public static boolean isObject(@NotNull @NlsSafe String type){
    return _OBJECT.equals(type);
  }

  public static boolean isMixedType(@NotNull @NlsSafe String type){
    return _MIXED.equals(type);
  }

  public static boolean isCallableType(@NotNull @NlsSafe String type){
    return _CALLABLE.equals(type);
  }

  public static boolean isPluralType(@NotNull @NlsSafe String type){
    return type.endsWith("[]");
  }

  public static int getPluralDimension(@NotNull @NlsSafe String type) {
    int res = 0;
    while (StringUtil.endsWith(type, 0, type.length() - res * 2, "[]")) {
      res++;
    }
    return res;
  }

  public static boolean isPluralPrimitiveType(@NotNull @NlsSafe String type){
    return type.endsWith("[]") && isPrimitiveType(type.substring(0, type.length()-2));
  }

  public static boolean isAnonymousClass(@NotNull @NlsSafe String type) {
    return StringUtil.startsWith(type, PhpClass.ANONYMOUS);
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (!(o instanceof PhpType)) return false;

    PhpType phpType = (PhpType)o;
    return isComplete == phpType.isComplete && Objects.equals(types, phpType.types);
  }

  @Override
  public int hashCode() {
    int result = types != null ? computeHashCode(types) : 0;
    result = 31 * result + (isComplete ? 1 : 0);
    return result;
  }

  private static int computeHashCode(@NotNull Set<@NlsSafe String> types) {
    int result = 0;
    for (final String type : types) {
      result += CaseInsensitiveStringHashingStrategy.INSTANCE.computeHashCode(type); // the same as THashSet does
    }
    return result;
  }

  public static boolean isUnresolved(@NotNull @NlsSafe String type){
    return type.indexOf('#')!=-1;
  }

  public static boolean isNull(@NotNull @NlsSafe String type){
    return _NULL.equalsIgnoreCase(type);
  }

  public boolean isNullable() {
    return getTypes().stream().anyMatch(PhpType::isNull);
  }

  private static boolean isScalar(@NotNull @NlsSafe String type){
    return _BOOL.equals(type)     ||
           _BOOLEAN.equals(type)  ||
           _FLOAT.equals(type)    ||
           _STRING.equals(type)   ||
           _INT.equals(type)      ||
           _INTEGER.equals(type);
  }

  public static boolean isScalar(@NotNull final PhpType type, @NotNull final Project project){
    final PhpType completedType = type.global(project);
    Set<String> types = completedType.types;
    if(types==null) return true;
    for(String curType : types){
      if(!isScalar(curType)){
        return false;
      }
    }
    return true;
  }

  public boolean hasUnresolved() {
    if(types==null) return false;
    for (String type : types) {
      if (startsWithChar(type, '#')) {
        return true;
      }
    }
    return false;
  }

  public static boolean intersectsGlobal(Project project, @NotNull PhpType f, @NotNull PhpType s) {
    return intersectsLocal(f, s) || intersects(f.global(project), s.global(project));
  }

  private static boolean intersectsLocal(@NotNull PhpType f, @NotNull PhpType s) {
    if (f.hasUnknown() || s.hasUnknown()) {
      PhpType f1 = f.filterUnknown();
      PhpType s1 = s.filterUnknown();
      if (!f1.isEmpty() && !s1.isEmpty() && intersects(f1, s1)) {
        return true;
      }
    }
    return false;
  }

  public static boolean intersects(@NotNull PhpType phpType1,@NotNull PhpType phpType2){
    //PhpContractUtil.assertCompleteType(phpType1, phpType2);
    final Set<String> phpTypeSet1 = phpType1.types;
    final Set<String> phpTypeSet2 = phpType2.types;
    if(phpTypeSet1==null || phpTypeSet2==null) return phpTypeSet1==phpTypeSet2;
    for(String type1 : phpTypeSet1){
      if (phpTypeSet2.contains(type1) ||
          (_FALSE.equals(type1) || _TRUE.equals(type1)) && phpTypeSet2.contains(_BOOL) ||
          _BOOL.equals(type1) && (phpTypeSet2.contains(_FALSE) || phpTypeSet2.contains(_TRUE))) {
        return true;
      }
    }
    return false;
  }

  public static boolean isSubType(@NotNull PhpType phpType1, @NotNull PhpType phpType2){
    //PhpContractUtil.assertCompleteType(phpType1, phpType2);
    final Set<String> typeSet1 = phpType1.types;
    if(typeSet1==null || typeSet1.size() == 0){
      return false;
    }
    final Set<String> typeSet2 = phpType2.types;
    if(typeSet2==null || typeSet2.size() == 0){
      return false;
    }
    for(String type1 : typeSet1){
      if(!typeSet2.contains(type1)){
        return false;
      }
    }
    return true;
  }

  @NotNull
  public static PhpType and(@NotNull PhpType phpType1, @NotNull PhpType phpType2) {
    final Set<String> phpTypeSet1 = phpType1.types;
    final Set<String> phpTypeSet2 = phpType2.types;
    if(phpTypeSet1==null || phpTypeSet2==null) return EMPTY;
    final PhpType phpType = new PhpType();
    for (String type1 : phpTypeSet1) {
      if (phpTypeSet2.contains(type1)) {
        phpType.add(type1);
      }
    }
    return phpType;
  }

  @NotNull
  public static PhpType or(@NotNull PhpType phpType1, @NotNull PhpType phpType2) {
    return new PhpType().add(phpType1).add(phpType2);
  }

  @NotNull
  public PhpType filterNull() {
    return filterOutIncompleteTypesAware(ExcludeCode.NOT_NULL);
  }

  public PhpType filterScalarPrimitives() {
    return filterOutIncompleteTypesAware(ExcludeCode.NOT_PRIMITIVE);
  }

  public PhpType filterFalse() {
    return filterOutIncompleteTypesAware(ExcludeCode.NOT_FALSE);
  }

  @ApiStatus.Internal
  public PhpType filterOutIncompleteTypesAware(PhpTypeExclusion excludeCodeForIncompleteType) {
    if (types == null || types.isEmpty()) {
      return this instanceof ImmutablePhpType ? getEmpty() : new PhpType();
    }
    final PhpType phpType = new PhpType();
    for (String type : types) {
      if (excludeCodeForIncompleteType.isNotApplicableType(null, type)) {
        continue;
      }
      if (!PhpKeyTypeProvider.isArrayKeySignature(type) && (!excludeCodeForIncompleteType.filterOnlyUnresolved() || startsWithChar(type,'#')) && !StringUtil.containsChar(type,'|')) {
        phpType.add(filteredIncompleteType(type, excludeCodeForIncompleteType));
      } else {
        phpType.add(type);
      }
    }
    return phpType;
  }

  protected PhpType getEmpty() {
    return new PhpType();
  }

  private static String filteredIncompleteType(@NlsSafe String type, PhpTypeExclusion code) {
    int i = 0;
    String suffixToAdd = EXCLUDED_INCOMPLETE_TYPE_SEPARATOR + code.getCode();
    while (type.startsWith("#-", i) ) {
      if (StringUtil.endsWith(type, 0, type.length() - i, suffixToAdd)) {
        return type;
      }
      i += 2;
    }
    return String.format("#-%s%s", type, suffixToAdd);
  }

  @NotNull
  public PhpType filterMixed() {
    return filterOutIncompleteTypesAware(ExcludeCode.NOT_MIXED);
  }

  @ApiStatus.Internal
  public PhpType filterObject() {
    return filterOutIncompleteTypesAware(ExcludeCode.NOT_OBJECT);
  }

  @NotNull
  public PhpType filterPlurals() {
    return filterOut(PhpType::isPluralType);
  }

  @NotNull
  public PhpType filterOut(@NotNull Predicate<String> typeExcludePredicate) {
    if(types == null || types.isEmpty()) return getEmpty();
    final PhpType phpType = new PhpType();
    for (String type : types) {
      if(!typeExcludePredicate.test(type)) {
        phpType.add(type);
      }
    }
    return phpType;
  }

  @NotNull
  public PhpType filter(@NotNull final PhpType sieve) {
    if (ContainerUtil.isEmpty(types)) return getEmpty();
    final Set<String> sieveTypes = sieve.types;
    if (ContainerUtil.isEmpty(sieveTypes)) return new PhpType().add(this);
    final PhpType phpType = new PhpType();
    if (sieveTypes.size() == 1) {
      final String sieveType = ContainerUtil.getFirstItem(sieveTypes);
      assert sieveType != null;
      types.stream().filter(type -> !sieveType.equals(type)).forEach(phpType::add);
    }
    else {
      types.stream().filter(type -> !sieveTypes.contains(type)).forEach(phpType::add);
    }
    return phpType;
  }

  private static class ImmutablePhpType extends PhpType {

    @NotNull
    @Override
    public PhpType add(@Nullable @NlsSafe String aClass) {
      throw getException();
    }

    @NotNull
    @Override
    public PhpType add(@Nullable PsiElement other) {
      throw getException();
    }

    @NotNull
    @Override
    public PhpType add(@Nullable PhpType type) {
      throw getException();
    }

    @Override
    protected PhpType getEmpty() {
      return EMPTY;
    }

    private PhpType addInternal(@Nullable PhpType type) {
      return super.add(type);
    }

    @NotNull
    private static RuntimeException getException() {
      return new UnsupportedOperationException("This PHP type is immutable");
    }
  }

  public static class PhpTypeBuilder {

    private final PhpType temp = new PhpType();

    @NotNull
    public PhpTypeBuilder add(@Nullable @NlsSafe String aClass) {
      temp.add(aClass);
      return this;
    }

    @NotNull
    public PhpTypeBuilder add(@Nullable @NlsSafe PsiElement other) {
      temp.add(other);
      return this;
    }

    @NotNull
    public PhpTypeBuilder add(@Nullable final PhpType type) {
      temp.add(type);
      return this;
    }

    @NotNull
    public PhpTypeBuilder merge(@NotNull final PhpTypeBuilder builder) {
      temp.add(builder.temp);
      return this;
    }

    @NotNull
    public PhpType build() {
      final PhpType type = new ImmutablePhpType();
      if (temp.types != null) {
        switch (temp.types.size()) {
          case 0:
            type.types = Collections.emptySet();
            break;
          case 1:
            type.types = Collections.unmodifiableSet(CollectionFactory.createCaseInsensitiveStringSet(temp.types));
            break;
          default:
            type.types = temp.types;
            type.dirty = temp.dirty;
        }
      }
      type.isComplete = temp.isComplete;
      return type;
    }
  }
}
