// Copyright 2000-2021 JetBrains s.r.o.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.jetbrains.php;

import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.Condition;
import com.intellij.openapi.util.Ref;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiElement;
import com.intellij.util.Processor;
import com.intellij.util.containers.ContainerUtil;
import com.jetbrains.php.lang.psi.elements.*;
import com.jetbrains.php.lang.psi.resolve.types.PhpType;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.BiFunction;

import static com.intellij.util.containers.ContainerUtil.addIfNotNull;
import static java.util.Arrays.stream;
import static java.util.stream.Collectors.toCollection;

public final class PhpClassHierarchyUtils {
  private static final NextElementsAppender<PhpClass> SUPER_CLASS_APPENDER_NOT_AMBIGUITY =
    (element, phpClasses) -> addIfNotNull(phpClasses, element.getSuperClass());
  private static final NextElementsAppender<PhpClass> SUPER_CLASS_APPENDER_AMBIGUITY =
    (element, phpClasses) -> phpClasses.addAll(element.getSuperClasses());
  private static final NextElementsAppender<PhpClass> SUPER_INTERFACE_APPENDER_NOT_AMBIGUITY = (element, phpClasses) -> {
    final PhpIndex phpIndex = PhpIndex.getInstance(element.getProject());
    for (String interfaceName : element.getInterfaceNames()) {
      final Collection<PhpClass> interfacesByFQN = phpIndex.getInterfacesByFQN(interfaceName);
      if (interfacesByFQN.size() == 1) {
        phpClasses.add(interfacesByFQN.iterator().next());
      }
    }
  };
  private static final NextElementsAppender<PhpClass> SUPER_INTERFACE_APPENDER_AMBIGUITY = (element, phpClasses) -> {
    final PhpIndex phpIndex = PhpIndex.getInstance(element.getProject());
    for (String interfaceName : element.getInterfaceNames()) {
      phpClasses.addAll(phpIndex.getInterfacesByFQN(interfaceName));
    }
  };

  private static final NextElementsAppender<PhpClass> MIXINS_APPENDER =
    (element, phpClasses) -> ContainerUtil.addAll(phpClasses, element.getMixins());
  private static final NextElementsAppender<PhpClass> SUPER_TRAIT_APPENDER_NOT_AMBIGUITY = (element, phpClasses) -> {
    final PhpIndex phpIndex = PhpIndex.getInstance(element.getProject());
    for (String name : element.getTraitNames()) {
      final Collection<PhpClass> classes = phpIndex.getTraitsByFQN(name);
      if (classes.size() == 1) {
        phpClasses.add(classes.iterator().next());
      }
    }
  };
  private static final NextElementsAppender<PhpClass> SUPER_TRAIT_APPENDER_AMBIGUITY = (element, phpClasses) -> {
    final PhpIndex phpIndex = PhpIndex.getInstance(element.getProject());
    final String[] names = element.getTraitNames();
    for (String name : names) {
      phpClasses.addAll(phpIndex.getTraitsByFQN(name));
    }
  };


  private static final NextElementsAppender<PhpClass> SUPER_APPENDER_NOT_AMBIGUITY =
    new CompositeNextElementsAppender<>(SUPER_TRAIT_APPENDER_NOT_AMBIGUITY, SUPER_CLASS_APPENDER_NOT_AMBIGUITY, SUPER_INTERFACE_APPENDER_NOT_AMBIGUITY);

  private static final NextElementsAppender<PhpClass> SUPER_APPENDER_AMBIGUITY =
    new CompositeNextElementsAppender<>(SUPER_TRAIT_APPENDER_AMBIGUITY, SUPER_CLASS_APPENDER_AMBIGUITY, SUPER_INTERFACE_APPENDER_AMBIGUITY);

  private static final class CompositeNextElementsAppender<T> implements NextElementsAppender<T> {

    private final NextElementsAppender<T>[] myAppenders;

    private CompositeNextElementsAppender(NextElementsAppender<T>... appenders) {
      myAppenders = appenders;
    }

    @Override
    public final void appendNextElements(@NotNull T element, @NotNull Collection<? super T> collection) {
      for (NextElementsAppender<T> appender : myAppenders) {
        appender.appendNextElements(element, collection);
      }
    }
  }

  private PhpClassHierarchyUtils() {
  }

  @FunctionalInterface
  private interface NextElementsAppender<T> {

    void appendNextElements(@NotNull T element, @NotNull Collection<? super T> collection);

  }

  private static <T> void process(@NotNull final T initialElement,
                                 boolean processSelf,
                                 @NotNull final Processor<? super T> processor,
                                 @NotNull final NextElementsAppender<T> appender) {
    final Set<T> processed = new HashSet<>();
    final Deque<T> processorPool = new ArrayDeque<>();
    if (processSelf) {
      processorPool.add(initialElement);
    }
    else {
      appender.appendNextElements(initialElement, processorPool);
    }
    while (processorPool.size() > 0) {
      final T first = processorPool.getFirst();
      if (processed.add(first)) {
        if (processor.process(first)) {
          appender.appendNextElements(first, processorPool);
        }
        else {
          return;
        }
      }
      processorPool.removeFirst();
    }
  }

  public static void processSuperClasses(@NotNull final PhpClass clazz,
                                         boolean processSelf,
                                         final boolean allowAmbiguity,
                                         @NotNull final Processor<? super PhpClass> processor) {
    process(clazz, processSelf, processor, allowAmbiguity ? SUPER_CLASS_APPENDER_AMBIGUITY : SUPER_CLASS_APPENDER_NOT_AMBIGUITY);
  }

  public static void processSuperInterfaces(@NotNull final PhpClass clazz,
                                         boolean processSelf,
                                         final boolean allowAmbiguity,
                                         @NotNull final Processor<? super PhpClass> processor) {
    process(clazz, processSelf, processor, allowAmbiguity ? SUPER_INTERFACE_APPENDER_AMBIGUITY : SUPER_INTERFACE_APPENDER_NOT_AMBIGUITY);
  }

  /**
   * Process {@link PhpClass#getMixins()} as well
   * @see #processSuperWithoutMixins(PhpClass, boolean, boolean, Processor)
   */
  public static void processSupers(@NotNull final PhpClass clazz,
                                         boolean processSelf,
                                         final boolean allowAmbiguity,
                                         @NotNull final Processor<? super PhpClass> processor) {
    process(clazz, processSelf, processor, allowAmbiguity ? SUPER_APPENDER_AMBIGUITY_WITH_MIXINS : SUPER_APPENDER_NOT_AMBIGUITY_WITH_MIXINS);
  }

  public static void processSuperWithoutMixins(@NotNull final PhpClass clazz,
                                   boolean processSelf,
                                   final boolean allowAmbiguity,
                                   @NotNull final Processor<? super PhpClass> processor) {
    process(clazz, processSelf, processor, allowAmbiguity ? SUPER_APPENDER_AMBIGUITY : SUPER_APPENDER_NOT_AMBIGUITY);
  }

  private static final NextElementsAppender<PhpClass> SUPER_APPENDER_NOT_AMBIGUITY_WITH_MIXINS =
    new CompositeNextElementsAppender<>(SUPER_TRAIT_APPENDER_NOT_AMBIGUITY, MIXINS_APPENDER, SUPER_CLASS_APPENDER_NOT_AMBIGUITY, SUPER_INTERFACE_APPENDER_NOT_AMBIGUITY);

  private static final NextElementsAppender<PhpClass> SUPER_APPENDER_AMBIGUITY_WITH_MIXINS =
    new CompositeNextElementsAppender<>(SUPER_TRAIT_APPENDER_AMBIGUITY, MIXINS_APPENDER, SUPER_CLASS_APPENDER_AMBIGUITY, SUPER_INTERFACE_APPENDER_AMBIGUITY);

  public static boolean isSuperClass(@NotNull final PhpClass superClass, @NotNull final PhpClass subClass, boolean allowAmbiguity) {
    final Ref<Boolean> isSuperClassRef = new Ref<>(false);
    processSuperClasses(subClass, false, allowAmbiguity, curClass -> {
      if (classesEqual(curClass, superClass)) {
        isSuperClassRef.set(true);
      }
      return !isSuperClassRef.get();
    });
    return isSuperClassRef.get();
  }

  public static void processMethods(PhpClass phpClass,
                                    PhpClass initialClass,
                                    @NotNull HierarchyMethodProcessor methodProcessor,
                                    boolean processOwnMembersOnly) {
    processMembersInternal(phpClass, new HashSet<>(), null, initialClass, methodProcessor, Method.INSTANCEOF,
                           processOwnMembersOnly);
  }

  public static void processFields(PhpClass phpClass,
                                   PhpClass initialClass,
                                   @NotNull HierarchyFieldProcessor fieldProcessor,
                                   boolean processOwnMembersOnly) {
    processMembersInternal(phpClass, new HashSet<>(), null, initialClass, fieldProcessor, Field.INSTANCEOF, processOwnMembersOnly);
  }

  private static <T extends PhpClassMember> boolean processMembersInternal(@Nullable PhpClass phpClass,
                                                                           @NotNull Set<? super PhpClass> visited,
                                                                           @Nullable Map<String, PhpTraitUseRule> conflictResolution,
                                                                           PhpClass initialClass,
                                                                           @NotNull TypedHierarchyMemberProcessor<T> processor,
                                                                           Condition<PsiElement> condition,
                                                                           boolean processOwnMembersOnly) {
    if (phpClass == null || !visited.add(phpClass)) {
      return true;
    }
    //System.out.println(phpClass.getFQN() + " conflictResolution = " + conflictResolution);
    boolean processMethods = condition == Method.INSTANCEOF;
    for (PhpClassMember member : processMethods ? phpClass.getOwnMethods() : phpClass.getOwnFields()) {
      PhpTraitUseRule rule = conflictResolution != null ? conflictResolution.get(member.getFQN()) : null;
      //System.out.println("member = " + member.getFQN());
      //System.out.println("rule = " + (rule!=null?rule.getText():"null") );
      if (rule != null && rule.getAlias() == null && processMethods) {
        //System.out.println("*** OVERRIDE");
      }
      else {
        if (processMethods ?
            !((HierarchyMethodProcessor)processor).process((Method)member, phpClass, initialClass) :
            !((HierarchyFieldProcessor)processor).process((Field)member, phpClass, initialClass)
          ) return false;
      }
    }

    if(!processOwnMembersOnly && phpClass.hasTraitUses()) {
      List<PhpTraitUseRule> rules = phpClass.traitUseRules().toList();
      Map<String, PhpTraitUseRule> newConflictResolution = getTraitUseRulesConflictResolutions(rules);
      if (processMethods) {
        for (PhpTraitUseRule rule : rules) {
          if (!rule.isInsteadOf()) {
            for (Method member : rule.getMethods()) {
              if (member.isValid() && !((HierarchyMethodProcessor)processor).process(member, phpClass, initialClass)) return false;
            }
          }
        }
      }
      // @link http://www.php.net/manual/en/language.oop5.traits.php
      // An inherited member from a base class is overridden by a member inserted by a Trait.
      // The precedence order is that members from the current class override Trait methods, which in return override inherited methods.
      for (PhpClass trait : phpClass.getTraits()) {
        if (!processMembersInternal(trait, visited, newConflictResolution, initialClass, processor, condition, processOwnMembersOnly)) return false;
      }
    }

    if (!processOwnMembersOnly){
      for (PhpClass superClass : phpClass.getSuperClasses()) {
        if (!processMembersInternal(superClass, visited, null, initialClass, processor, condition, processOwnMembersOnly)) return false;
      }
      for (PhpClass phpInterface : phpClass.getImplementedInterfaces()) {
        if (!processMembersInternal(phpInterface, visited, null, initialClass, processor, condition, processOwnMembersOnly)) return false;
      }
      for (PhpClass mixin : phpClass.getMixins()) {
        if (!processMembersInternal(mixin, visited, null, initialClass, processor, condition, processOwnMembersOnly)) return false;
      }
    }

    return true;
  }

  @NotNull
  public static Map<String, PhpTraitUseRule> getTraitUseRulesConflictResolutions(Collection<PhpTraitUseRule> rules) {
    Map<String, PhpTraitUseRule> newConflictResolution = new HashMap<>();
    for (PhpTraitUseRule rule : rules) {
      for (String overridesFqn : rule.getOverriddenMethodFqns()) {
        newConflictResolution.put(overridesFqn, rule);
      }
    }
    return newConflictResolution;
  }

  public static boolean processOverridingFields(@NotNull Field field, TypedHierarchyMemberProcessor<? super Field> memberProcessor) {
    PhpIndex phpIndex = PhpIndex.getInstance(field.getProject());
    final PhpClass me = field.getContainingClass();
    if (me != null) {
      final String fieldName = field.getName();
      final Collection<PhpClass> allSubclasses = phpIndex.getAllSubclasses(me.getFQN());
      for (PhpClass myChild : allSubclasses) {
        boolean isConstant = field.isConstant();
        final Field overriddenField = myChild.findOwnFieldByName(fieldName, isConstant);
        if (overriddenField != null) {
          if (!memberProcessor.process(overriddenField, myChild, me)) return false;
        }
      }
      return true;
    }
    return false;
  }

  public static boolean processOverridingMethods(@NotNull Method method, TypedHierarchyMemberProcessor<? super Method> memberProcessor) {
    PhpIndex phpIndex = PhpIndex.getInstance(method.getProject());
    if (!methodCanHaveOverride(method)) return true;
    final String methodName = method.getName();
    PhpClass me = method.getContainingClass();
    if (me != null && me.isValid()) {
      final Collection<PhpClass> allSubclasses = new ArrayList<>(phpIndex.getAllSubclasses(me.getFQN()));
      if (me.isTrait()) {
        allSubclasses.addAll(phpIndex.getTraitUsages(me));
      }
      collectUsedTraits(allSubclasses).stream().filter(trait -> me != trait).forEach(allSubclasses::add);
      for (PhpClass myChild : allSubclasses) {
        if (myChild!=null) {
          Method overriddenMethod = myChild.findOwnMethodByName(methodName);
          if (overriddenMethod != null) {
            if (!memberProcessor.process(overriddenMethod, myChild, me)) return false;
          }
        }
      }
    }
    return true;

  }

  public static boolean processOverridingMembers(PhpClassMember member, HierarchyClassMemberProcessor memberProcessor){
    if (member instanceof Method){
      return processOverridingMethods((Method)member, (method, subClass, baseClass) -> memberProcessor.process(method,subClass,baseClass));
    } else {
      return processOverridingFields((Field)member, (field, subClass, baseClass) -> memberProcessor.process(field,subClass,baseClass));
    }
  }

  @NotNull
  private static Collection<PhpClass> collectUsedTraits(@NotNull Collection<PhpClass> classes) {
    Set<PhpClass> result = new HashSet<>();
    Deque<PhpClass> traitsToProcess = classes.stream().flatMap(e -> stream(e.getTraits())).collect(toCollection(ArrayDeque::new));
    while (!traitsToProcess.isEmpty()) {
      PhpClass trait = traitsToProcess.pollFirst();
      if (trait == null || !result.add(trait)) continue;
      for (PhpClass innerTrait : trait.getTraits()) {
        traitsToProcess.addLast(innerTrait);
      }
    }
    return result;
  }


  public static boolean methodCanHaveOverride(@NotNull Method element) {
    PhpClass containingClass = element.getContainingClass();
    return containingClass != null && !containingClass.isFinal() && !element.getAccess().isPrivate() && !element.isFinal();
  }

  private static <T extends PhpClassMember> boolean processSuperMembersInternal(@NotNull T member, @NotNull BiFunction<PhpClass, T, Collection<T>> memberFounder, @NotNull TypedHierarchyMemberProcessor<T> memberProcessor) {
    Set<PhpClass> processed = new HashSet<>();
    final Queue<PhpClassMember> members = new ArrayDeque<>();
    members.add(member);
    while (!members.isEmpty()) {
      final PhpClass me = members.poll().getContainingClass();
      if (me == null) return false;
      List<PhpClass> parents = getImmediateParents(me);
      if (me.isTrait()) {
        for (PhpClass usage : collectClassesWithTraitUsage(me)) {
          parents.addAll(usage.getSuperClasses());
          Collections.addAll(parents, usage.getImplementedInterfaces());
        }
      }
      Collection<String> overriddenClassesFqns = new HashSet<>();
      if (member instanceof Method) {
        String methodName = member.getName();
        me.traitUseRules()
          .filter(PhpTraitUseRule::isInsteadOf)
          .flatMap(e -> e.getOverriddenMethodFqns())
          .filter(fqn -> fqn.substring(fqn.lastIndexOf('.') + 1).equalsIgnoreCase(methodName))
          .map(fqn -> fqn.substring(0, fqn.lastIndexOf('.')))
          .forEach(overriddenClassesFqns::add);
      }
      boolean res = true;
      for (PhpClass mySuper : parents) {
        if (!processed.add(mySuper) || mySuper.isTrait() && overriddenClassesFqns.contains(mySuper.getFQN())) continue;
        for (T foundMember : memberFounder.apply(mySuper, member)) {
          if (foundMember != null) {
            PhpModifier modifier = foundMember.getModifier();
            if (!modifier.isPrivate() || foundMember instanceof Method && modifier.isAbstract() && Optional.ofNullable(foundMember.getContainingClass()).map(PhpClass::isTrait).orElse(false)) {
              members.add(foundMember);
              res = memberProcessor.process(foundMember, me, mySuper) && res;
            }
          }
        }
      }
      if (!res) {
        return false;
      }
    }
    return true;

  }

  public static boolean processSuperMethods(@NotNull Method member, @NotNull HierarchyMethodProcessor memberProcessor) {
    return processSuperMembersInternal(member, (superClass, method) -> superClass.findMethodsByName(member.getName()), memberProcessor);
  }

  public static boolean processSuperFields(@NotNull Field member, @NotNull final HierarchyFieldProcessor memberProcessor) {
    return processSuperMembersInternal(member, (superClass, field) -> Collections.singleton(superClass.findFieldByName(member.getName(), member.isConstant())),
                                       memberProcessor);
  }

  public static boolean processSuperMembers(@NotNull PhpClassMember member, @NotNull final HierarchyClassMemberProcessor memberProcessor) {
    if (member instanceof Method) {
      return processSuperMethods(((Method)member), (member1, subClass, baseClass) -> memberProcessor.process(member1, subClass, baseClass));
    } else {
      return processSuperFields(((Field)member), (member1, subClass, baseClass) -> memberProcessor.process(member1, subClass, baseClass));
    }
  }

  @NotNull
  private static Collection<PhpClass> collectClassesWithTraitUsage(@NotNull PhpClass trait) {
    PhpIndex index = PhpIndex.getInstance(trait.getProject());
    Deque<PhpClass> usages = new ArrayDeque<>(index.getTraitUsages(trait));
    Set<PhpClass> result = new HashSet<>();
    while (!usages.isEmpty()) {
      PhpClass classWithTraitUsage = usages.pollFirst();
      if (classWithTraitUsage == null || !result.add(classWithTraitUsage)) continue;
      if (classWithTraitUsage.isTrait()) usages.addAll(index.getTraitUsages(classWithTraitUsage));
    }
    return result;
  }

  @NotNull
  public static Collection<PhpClass> getSuperClasses(@NotNull PhpClass aClass) {
    String superName = aClass.getSuperFQN();
    final PhpIndex phpIndex = PhpIndex.getInstance(aClass.getProject());
    return aClass.isInterface() ?
           phpIndex.getInterfacesByFQN(superName) :
           phpIndex.getClassesByFQN(superName);
  }

  public static List<PhpClass> getImmediateParents(PhpClass me) {
    List<PhpClass> parents = new ArrayList<>(getSuperClasses(me));
    ContainerUtil.addAll(parents, me.getImplementedInterfaces());
    ContainerUtil.addAll(parents, me.getTraits());
    return parents;
  }

  @FunctionalInterface
  public interface HierarchyClassMemberProcessor extends HierarchyMemberProcessor {
    boolean process(PhpClassMember classMember, PhpClass subClass, PhpClass baseClass);
  }

  @FunctionalInterface
  public interface HierarchyMethodProcessor extends TypedHierarchyMemberProcessor<Method> {

  }

  @FunctionalInterface
  public interface HierarchyFieldProcessor extends TypedHierarchyMemberProcessor<Field> {

  }

  public interface HierarchyMemberProcessor {
  }

  public interface TypedHierarchyMemberProcessor<T extends PhpClassMember> {
    boolean process(T member, PhpClass subClass, PhpClass baseClass);
  }

  public static Collection<PhpClass> getDirectSubclasses(@NotNull PhpClass psiClass) {
    if (psiClass.isFinal()) {
      return Collections.emptyList();
    }

    final PhpIndex phpIndex = PhpIndex.getInstance(psiClass.getProject());
    return phpIndex.getDirectSubclasses(psiClass.getFQN());
  }

  public static Collection<PhpClass> getAllSubclasses(@NotNull PhpClass psiClass) {
    if (psiClass.isFinal()) {
      return Collections.emptyList();
    }

    final PhpIndex phpIndex = PhpIndex.getInstance(psiClass.getProject());
    return phpIndex.getAllSubclasses(psiClass.getFQN());
  }

  @Nullable
  public static PhpClass getObject(@NotNull final Project project) {
    Iterator iterator = PhpIndex.getInstance(project).getClassesByFQN(PhpType._OBJECT_FQN).iterator();
    return iterator.hasNext() ? (PhpClass)iterator.next() : null;
  }

  public static boolean isMyTrait(@NotNull final PhpClass me, @NotNull final PhpClass trait, @Nullable Collection<? super PhpClass> visited) {
    if (!trait.isTrait()) {
      return false;
    }
    if (visited == null) {
      visited = new HashSet<>();
    }
    for (final PhpClass candidate : me.getTraits()) {
      if (!visited.add(candidate)) {
        continue;
      }
      if (classesEqual(trait, candidate) || isMyTrait(candidate, trait, visited)) {
        return true;
      }
    }
    return false;
  }

  public static boolean classesEqual(@Nullable PhpClass one, @Nullable PhpClass another) {
    if (one != null && another != null) {
      if (one == another) {
        return true;
      }
      else if (one instanceof PhpClassAlias || another instanceof PhpClassAlias) {
        if (one instanceof PhpClassAlias) one = ((PhpClassAlias)one).getOriginal();
        if (another instanceof PhpClassAlias) another = ((PhpClassAlias)another).getOriginal();
        return classesEqual(one, another);
      }
      else if (StringUtil.equalsIgnoreCase(one.getFQN(), another.getFQN())) {
        return true;
      }
    }
    return false;
  }

}
