/*
 * Copyright 2000-2018 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.spi;

import com.intellij.codeInsight.completion.JavaLookupElementBuilder;
import com.intellij.codeInsight.daemon.EmptyResolveMessageProvider;
import com.intellij.codeInsight.lookup.LookupElement;
import com.intellij.jam.JamConverter;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.roots.ModuleRootManager;
import com.intellij.openapi.util.TextRange;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.psi.*;
import com.intellij.spring.SpringApiBundle;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.List;

/**
 * Provides class reference to valid values configured via given config key in {@code spring.factories}.
 *
 * @see SpringSpiManager#getClassesListValue(boolean, String)
 */
public class SpringSpiClassesListJamConverter extends JamConverter<PsiClass> {

  private final String myConfigKey;

  public SpringSpiClassesListJamConverter(String configKey) {
    myConfigKey = configKey;
  }

  @Nullable
  @Override
  public PsiClass fromString(@Nullable final String s, JamStringAttributeElement<PsiClass> context) {
    if (StringUtil.isEmptyOrSpaces(s)) return null;

    PsiLanguageInjectionHost host = context.getLanguageInjectionHost();
    if (host == null) return null;

    final PsiReference[] references = createReferences(context, host);
    if (references.length != 1) return null;
    return (PsiClass)references[0].resolve();
  }

  @Override
  public PsiReference @NotNull [] createReferences(@NotNull final JamStringAttributeElement<PsiClass> context,
                                                   @NotNull PsiLanguageInjectionHost injectionHost) {
    return new PsiReference[]{new SpringSpiClassReference(myConfigKey, injectionHost, context.getStringValue())};
  }


  public static class SpringSpiClassReference extends PsiReferenceBase<PsiElement> implements EmptyResolveMessageProvider {

    private final String myConfigKey;
    private final String myText;

    public SpringSpiClassReference(String configKey, @NotNull PsiElement literal, String text) {
      super(literal);
      myConfigKey = configKey;
      myText = text;
    }

    public SpringSpiClassReference(String configKey, PsiElement element, TextRange rangeInElement, String text) {
      super(element, rangeInElement);
      myConfigKey = configKey;
      myText = text;
    }

    @Nullable
    @Override
    public PsiElement resolve() {
      if (StringUtil.isEmptyOrSpaces(myText)) return null;

      return ContainerUtil.find(getRelevantClasses(getElement()), psiClass -> myText.equals(psiClass.getQualifiedName()));
    }

    @Override
    public Object @NotNull [] getVariants() {
      final List<PsiClass> classes = getRelevantClasses(getElement());
      return ContainerUtil.map2Array(classes, LookupElement.class, psiClass ->
        JavaLookupElementBuilder.forClass(psiClass, psiClass.getQualifiedName(), true)
                                .withPresentableText(StringUtil.notNullize(psiClass.getName())));
    }

    private List<PsiClass> getRelevantClasses(PsiElement literal) {
      if (literal == null) return Collections.emptyList();

      final Module module = ModuleUtilCore.findModuleForPsiElement(literal);
      if (module == null) return Collections.emptyList();

      VirtualFile containingFile = literal.getContainingFile().getOriginalFile().getVirtualFile();
      final boolean isInTest = containingFile != null &&
                               ModuleRootManager.getInstance(module).getFileIndex().isInTestSourceContent(containingFile);
      return SpringSpiManager.getInstance(module).getClassesListValue(isInTest, myConfigKey);
    }

    @NotNull
    @Override
    public String getUnresolvedMessagePattern() {
      return SpringApiBundle.message("SpringSpiClassesListJamConverter.unresolved.message.pattern", myText, myConfigKey);
    }
  }
}
