// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.spi;

import com.intellij.openapi.util.Ref;
import com.intellij.openapi.util.text.DelimitedListProcessor;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiReference;
import com.intellij.psi.PsiReferenceProvider;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.JavaClassReferenceProvider;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.JavaClassReferenceSet;
import com.intellij.util.ArrayUtil;
import com.intellij.util.ProcessingContext;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;

/**
 * Provides class references for values specified in {@code spring.factories}.
 */
public final class SpringSpiClassReferenceProvider extends PsiReferenceProvider {
  public static final SpringSpiClassReferenceProvider INSTANCE = new SpringSpiClassReferenceProvider();

  public static final JavaClassReferenceProvider CLASS_REFERENCE_PROVIDER = new JavaClassReferenceProvider() {
    @Override
    public boolean isSoft() {
      return true;
    }
  };

  @NonNls
  public static final String VALUE_DELIMITERS = ",\\ ";

  static {
    CLASS_REFERENCE_PROVIDER.setOption(JavaClassReferenceProvider.ALLOW_DOLLAR_NAMES, Boolean.TRUE);
  }

  private SpringSpiClassReferenceProvider() {
  }

  @Override
  public PsiReference @NotNull [] getReferencesByElement(@NotNull final PsiElement element, @NotNull ProcessingContext context) {
    final String text = element.getText();

    final Ref<PsiReference[]> allReferences = Ref.create(PsiReference.EMPTY_ARRAY);
    new DelimitedListProcessor(VALUE_DELIMITERS) {  // skip '\' and whitespaces
      @Override
      protected void processToken(int start, int end, boolean delimitersOnly) {
        JavaClassReferenceSet classReferenceSet =
          new JavaClassReferenceSet(text.substring(start, end), element, start, true, CLASS_REFERENCE_PROVIDER) {
            @Override
            public boolean isAllowDollarInNames() {
              return true;
            }
          };
        allReferences.set(ArrayUtil.mergeArrays(allReferences.get(), classReferenceSet.getAllReferences()));
      }
    }.processText(text);
    return allReferences.get();
  }
}
