// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.references;

import com.intellij.codeInsight.ExpectedTypeInfo;
import com.intellij.codeInsight.ExpectedTypesProvider;
import com.intellij.psi.*;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.psi.util.PsiTypesUtil;
import com.intellij.util.ProcessingContext;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class SpringBeanNamesReferenceProvider extends PsiReferenceProvider {

  public static final String[] METHODS = new String[]{"containsBean", "getBean", "isSingleton", "getType", "getAliases"};

  @Override
  public PsiReference @NotNull [] getReferencesByElement(@NotNull PsiElement element, @NotNull final ProcessingContext context) {
    if (element instanceof PsiLiteral) {
      final PsiLiteral literalExpression = (PsiLiteral)element;
      if (literalExpression.getValue() instanceof String) {
        return new PsiReference[]{
          new SpringBeanReference(literalExpression,
                                  ElementManipulators.getValueTextRange(element),
                                  determineRequiredClass(element),
                                  false)
        };
      }
    }
    return PsiReference.EMPTY_ARRAY;
  }

  /**
   * Tries to determine required bean class by analyzing surrounding expression.
   *
   * @param element Current element.
   * @return Expected bean class or {@code null} if not determinable.
   */
  @Nullable
  public static PsiClass determineRequiredClass(PsiElement element) {
    final PsiExpression expression = PsiTreeUtil.getParentOfType(element, PsiExpression.class);
    if (expression == null) {
      return null;
    }

    final ExpectedTypeInfo[] types = ExpectedTypesProvider.getExpectedTypes(expression, true);
    if (types.length != 1) {
      return null;
    }

    final PsiType type = types[0].getType();
    return PsiTypesUtil.getPsiClass(type);
  }
}
