// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.profiles;

import com.intellij.ide.presentation.Presentation;
import com.intellij.ide.util.PsiNavigationSupport;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.Segment;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.pom.PomRenameableTarget;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiFile;
import com.intellij.psi.SmartPointerManager;
import com.intellij.psi.SmartPsiElementPointer;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Objects;

import static com.intellij.spring.constants.SpringCorePresentationConstants.SPRING_PROFILE;

@Presentation(typeName = SPRING_PROFILE)
public class SpringProfileTarget implements PomRenameableTarget<Object> {
  private final SmartPsiElementPointer<PsiElement> myElementPointer;
  private final int myNameOffset;
  private String myName;

  public SpringProfileTarget(@NotNull PsiElement element, @NotNull String name, int nameOffset) {
    PsiFile containingFile = element.getContainingFile();
    Project project = containingFile == null ? element.getProject() : containingFile.getProject();
    myElementPointer = SmartPointerManager.getInstance(project).createSmartPsiElementPointer(element, containingFile);
    myNameOffset = nameOffset;
    myName = name;
  }

  @Override
  public boolean isWritable() {
    return true;
  }

  @Override
  public Object setName(@NotNull String name) {
    myName = name;
    return null;
  }

  @Override
  public String getName() {
    return myName;
  }

  @Override
  public boolean isValid() {
    return myElementPointer.getElement() != null;
  }

  @Override
  public void navigate(boolean requestFocus) {
    Segment elementRange = myElementPointer.getRange();
    if (elementRange == null) return;

    int offset = elementRange.getStartOffset();
    if (myNameOffset < elementRange.getEndOffset() - offset) {
      offset += myNameOffset;
    }
    VirtualFile virtualFile = myElementPointer.getVirtualFile();
    if (virtualFile != null && virtualFile.isValid()) {
      PsiNavigationSupport.getInstance().createNavigatable(myElementPointer.getProject(), virtualFile, offset).navigate(requestFocus);
    }
  }

  @Override
  public boolean canNavigate() {
    return canNavigateToSource();
  }

  @Override
  public boolean canNavigateToSource() {
    if (myNameOffset < 0) return false;

    PsiElement element = myElementPointer.getElement();
    return element != null && PsiNavigationSupport.getInstance().canNavigate(element);
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;

    if (o == null || getClass() != o.getClass()) return false;

    SpringProfileTarget target = (SpringProfileTarget)o;
    return Objects.equals(myName, target.myName);
  }

  @Override
  public int hashCode() {
    return myName == null ? 0 : myName.hashCode();
  }

  @Nullable
  public PsiElement getElement() {
    return myElementPointer.getElement();
  }
}
