// Copyright 2000-2021 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.profiles;

import com.intellij.codeInsight.lookup.LookupElementBuilder;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.roots.ModuleRootManager;
import com.intellij.openapi.util.TextRange;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.pom.references.PomService;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiFile;
import com.intellij.psi.PsiReferenceBase;
import com.intellij.spring.SpringApiIcons;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;

import java.util.LinkedHashSet;
import java.util.Set;

/**
 * @author Sergey Vasiliev
 * @see SpringProfilesFactory#getProfilesReferences(Module, PsiElement, String, int, String, boolean)
 */
public class SpringProfilePsiReference extends PsiReferenceBase<PsiElement> {
  private final String myProfileName;
  private final Module myModule;
  private final boolean myIsDefinition;

  public SpringProfilePsiReference(@NotNull PsiElement element,
                                   @NotNull TextRange range,
                                   @NotNull Module module,
                                   boolean definition) {
    super(element, range);
    myIsDefinition = definition;
    myProfileName = range.substring(element.getText());
    myModule = module;
  }

  @Override
  public PsiElement resolve() {
    if (StringUtil.isEmptyOrSpaces(myProfileName)) return myElement;

    SpringProfileTarget target;
    if (myIsDefinition) {
      target = new SpringProfileTarget(getElement(), myProfileName, getRangeInElement().getStartOffset());
    }
    else {
      target = SpringProfilesFactory.getInstance().findProfileTarget(myModule, includeTestScope(), myProfileName);
    }
    return target == null ? null : PomService.convertToPsi(getElement().getProject(), target);
  }

  @Override
  public boolean isSoft() {
    return true;
  }

  @Override
  public Object @NotNull [] getVariants() {
    Set<String> names = new LinkedHashSet<>();
    for (SpringProfileTarget target : SpringProfilesFactory.getInstance().findProfileTargets(myModule, includeTestScope())) {
      names.add(target.getName());
    }
    return ContainerUtil.map2Array(names, Object.class, name -> LookupElementBuilder.create(name).withIcon(SpringApiIcons.SpringProfile));
  }

  private boolean includeTestScope() {
    PsiFile file = myElement.getContainingFile();
    if (file == null) return false;

    VirtualFile virtualFile = file.getVirtualFile();
    if (virtualFile == null) return false;

    return ModuleRootManager.getInstance(myModule).getFileIndex().isInTestSourceContent(virtualFile);
  }
}
