// Copyright 2000-2021 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.profiles;

import com.intellij.codeInsight.completion.CompletionParameters;
import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.Computable;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.spring.SpringApiIcons;
import com.intellij.spring.model.SpringProfile;
import com.intellij.spring.model.utils.SpringProfileUtils;
import com.intellij.ui.TextFieldWithAutoCompletionListProvider;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.containers.FactoryMap;
import org.jetbrains.annotations.NotNull;

import javax.swing.*;
import java.util.*;

/**
 * provides list of defined profiles
 */
public final class SpringProfileCompletionProvider extends TextFieldWithAutoCompletionListProvider<String> {
  private final boolean myIncludeTests;
  private final Map<Module, Collection<String>> myProfiles = FactoryMap.create(this::getProfiles);
  private Collection<Module> myContext = Collections.emptyList();

  public SpringProfileCompletionProvider(boolean includeTests) {
    super(null);
    myIncludeTests = includeTests;
  }

  /**
   * @param modules collection of modules to search profiles for completion
   */
  public void setContext(@NotNull Collection<Module> modules) {
    myContext = modules;
  }

  @NotNull
  @Override
  protected String getLookupString(@NotNull String item) {
    return item;
  }

  @Override
  protected Icon getIcon(@NotNull String item) {
    return SpringApiIcons.SpringProfile;
  }

  @NotNull
  @Override
  public Collection<String> getItems(String prefix, boolean cached, CompletionParameters parameters) {
    if (prefix == null || myContext.isEmpty()) return Collections.emptyList();

    String text = filterOutCompletingProfile(parameters.getEditor().getDocument().getText(), parameters.getOffset());
    Set<String> activatedProfiles = SpringProfileUtils.profilesFromString(text);

    Set<String> profiles = new HashSet<>();
    for (Module module : myContext) {
      profiles.addAll(myProfiles.get(module));
    }
    profiles.removeAll(activatedProfiles);
    profiles.remove(SpringProfile.DEFAULT_PROFILE_NAME);
    List<String> result = new ArrayList<>(profiles);
    result.sort(StringUtil::naturalCompare);
    return result;
  }

  @Override
  public String getPrefix(@NotNull String text, int offset) {
    int space = text.lastIndexOf(' ', offset - 1) + 1;
    int comma = text.lastIndexOf(',', offset - 1) + 1;
    return text.substring(Math.max(space, comma), offset);
  }

  private Collection<String> getProfiles(Module module) {
    return ApplicationManager.getApplication().runReadAction((Computable<Collection<String>>)() -> {
      List<SpringProfileTarget> targets = SpringProfilesFactory.getInstance().findProfileTargets(module, myIncludeTests);
      return ContainerUtil.map2Set(targets, SpringProfileTarget::getName);
    });
  }

  private static String filterOutCompletingProfile(String text, int offset) {
    int space = text.lastIndexOf(' ', offset - 1) + 1;
    int comma = text.lastIndexOf(',', offset - 1) + 1;
    int before = Math.max(space, comma);
    String result = text.substring(0, before);

    space = text.indexOf(' ', offset);
    comma = text.indexOf(',', offset);
    int after = space < 0 ? comma : comma < 0 ? space : Math.min(space, comma);
    if (after >= 0) {
      result += text.substring(after);
    }
    return result;
  }
}
