// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.values.converters;

import com.intellij.openapi.util.Condition;
import com.intellij.psi.PsiClass;
import com.intellij.psi.util.InheritanceUtil;
import com.intellij.psi.util.PsiTypesUtil;
import com.intellij.spring.model.converters.SpringConverterUtil;
import com.intellij.spring.model.xml.DomSpringBean;
import com.intellij.spring.model.xml.beans.ConstructorArg;
import com.intellij.spring.model.xml.beans.SpringProperty;
import com.intellij.util.xml.DomElement;
import com.intellij.util.xml.GenericDomValue;
import org.jetbrains.annotations.NotNull;

public final class SpringValueConditionFactory {

  public static Condition<GenericDomValue> createPropertyNameCondition(@NotNull final String propertyName) {
    return genericDomValue -> checkPropertyName(genericDomValue, propertyName);
  }

  public static Condition<GenericDomValue> createBeanPropertyCondition(@NotNull final String beanClass,
                                                                       final String @NotNull ... propertyNames) {
    return genericDomValue -> {
      for (String propertyName : propertyNames) {
        if (checkPropertyName(genericDomValue, propertyName)) {
          return checkBeanClass(genericDomValue, beanClass);
        }
      }
      return false;
    };
  }

  public static Condition<GenericDomValue> createBeanClassCondition(@NotNull final String beanClass) {
    return genericDomValue -> checkBeanClass(genericDomValue, beanClass);
  }

  public static Condition<GenericDomValue> createBeanClassConstructorArgCondition(@NotNull final String beanClass) {
    return genericDomValue -> isConstructorArg(genericDomValue) && checkBeanClass(genericDomValue, beanClass);
  }

  private static boolean isConstructorArg(@NotNull GenericDomValue genericDomValue) {
    return genericDomValue.getParentOfType(ConstructorArg.class, false) != null;
  }

  private static boolean checkBeanClass(@NotNull DomElement element, @NotNull String beanClassName) {
    DomSpringBean springBean = SpringConverterUtil.getCurrentBean(element);
    if (springBean == null) return false;

    final PsiClass beanClass = PsiTypesUtil.getPsiClass(springBean.getBeanType());
    return InheritanceUtil.isInheritor(beanClass, beanClassName);
  }

  private static boolean checkPropertyName(@NotNull DomElement element, @NotNull String propertyName) {
    final SpringProperty springProperty = element.getParentOfType(SpringProperty.class, false);
    return springProperty != null && propertyName.equals(springProperty.getName().getStringValue());
  }
}
