// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.values.converters;

import com.intellij.codeInsight.completion.JavaLookupElementBuilder;
import com.intellij.codeInsight.daemon.EmptyResolveMessageProvider;
import com.intellij.codeInsight.lookup.LookupElement;
import com.intellij.openapi.util.TextRange;
import com.intellij.psi.*;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.JavaClassReferenceProvider;
import com.intellij.spring.SpringApiBundle;
import com.intellij.util.ArrayUtil;
import com.intellij.util.IncorrectOperationException;
import com.intellij.util.xml.ConvertContext;
import com.intellij.util.xml.Converter;
import com.intellij.util.xml.CustomReferenceConverter;
import com.intellij.util.xml.GenericDomValue;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;

/**
 * @author Yann C&eacute;bron
 */
public abstract class FieldRetrievingFactoryBeanConverter extends Converter<String> implements CustomReferenceConverter<String> {

  protected final boolean mySoft;

  protected FieldRetrievingFactoryBeanConverter() {
    this(true);
  }

  protected FieldRetrievingFactoryBeanConverter(boolean soft) {
    mySoft = soft;
  }

  protected boolean requireFieldReference() {
    return false;
  }

  @Override
  public PsiReference @NotNull [] createReferences(final GenericDomValue<String> genericDomValue,
                                                   final PsiElement element,
                                                   final ConvertContext context) {
    return createReferences(genericDomValue, element);
  }

  @Override
  public String fromString(@Nullable @NonNls String s, final ConvertContext context) {
    return s;
  }

  @Override
  public String toString(@Nullable String s, final ConvertContext context) {
    return s;
  }


  public PsiReference[] createReferences(final GenericDomValue<String> genericDomValue, final PsiElement element) {
    final String stringValue = genericDomValue.getStringValue();
    if (stringValue == null) {
      return PsiReference.EMPTY_ARRAY;
    }

    List<PsiReference> collectedReferences = new ArrayList<>();

    final JavaClassReferenceProvider provider = new JavaClassReferenceProvider();
    provider.setSoft(mySoft);
    provider.setOption(JavaClassReferenceProvider.ALLOW_DOLLAR_NAMES, Boolean.TRUE);
    final PsiReference[] javaClassReferences = provider.getReferencesByElement(element);

    PsiClass psiClass = null;
    for (PsiReference reference : javaClassReferences) {
      final PsiElement psiElement = reference.resolve();
      if (psiElement == null) break;


      collectedReferences.add(reference);
      if (psiElement instanceof PsiClass) {
        psiClass = (PsiClass)psiElement;
      }
    }

    if (psiClass == null ||
        !requireFieldReference() && psiClass.getQualifiedName() != null && stringValue.endsWith(psiClass.getQualifiedName())) {
      return javaClassReferences;
    }

    collectedReferences.add(createFieldReference(psiClass, element, stringValue, genericDomValue));

    return collectedReferences.toArray(PsiReference.EMPTY_ARRAY);
  }

  private PsiReference createFieldReference(final PsiClass psiClass,
                                            final PsiElement element,
                                            final String stringValue,
                                            final GenericDomValue<String> genericDomValue) {
    final String className = psiClass.getName();
    assert className != null;
    final int fieldNameIdx = stringValue.lastIndexOf(className) + className.length();
    final String fieldName = stringValue.substring(Math.min(stringValue.length(), fieldNameIdx + 1)).trim();

    final TextRange textRange;
    if (fieldName.isEmpty()) {
      textRange = TextRange.from(element.getText().indexOf(className) + className.length() + 1, 0);
    }
    else {
      textRange = TextRange.from(Math.max(0, element.getText().lastIndexOf(fieldName)), fieldName.length());
    }

    return new FieldReference(element, textRange, fieldName, psiClass, genericDomValue);
  }


  protected class FieldReference extends PsiReferenceBase<PsiElement> implements EmptyResolveMessageProvider {

    private final String myFieldName;
    private final PsiClass myPsiClass;
    private final GenericDomValue<String> myGenericDomValue;

    protected FieldReference(PsiElement element,
                             TextRange textRange,
                             String fieldName,
                             PsiClass psiClass,
                             GenericDomValue<String> genericDomValue) {
      super(element, textRange, FieldRetrievingFactoryBeanConverter.this.mySoft);
      myFieldName = fieldName;
      myPsiClass = psiClass;
      myGenericDomValue = genericDomValue;
    }

    @Override
    public PsiElement resolve() {
      if (myFieldName.length() != 0) {
        final PsiField[] psiFields = myPsiClass.getAllFields();
        for (PsiField psiField : psiFields) {
          if (psiField.hasModifierProperty(PsiModifier.PUBLIC) &&
              psiField.hasModifierProperty(PsiModifier.STATIC) &&
              myFieldName.equals(psiField.getName())) {
            return psiField;
          }
        }
      }
      return null;
    }

    @Override
    public PsiElement bindToElement(@NotNull final PsiElement element) throws IncorrectOperationException {
      if (element instanceof PsiField) {
        final PsiField field = (PsiField)element;
        final PsiClass containingClass = field.getContainingClass();
        if (containingClass != null) {
          myGenericDomValue.setStringValue(containingClass.getQualifiedName() + "." + field.getName());
        }
      }
      return getElement();
    }

    @Override
    public Object @NotNull [] getVariants() {
      List<LookupElement> staticFields = new ArrayList<>();
      final PsiField[] psiFields = myPsiClass.getFields();
      for (PsiField psiField : psiFields) {
        if (psiField.hasModifierProperty(PsiModifier.PUBLIC) && psiField.hasModifierProperty(PsiModifier.STATIC)) {
          staticFields.add(JavaLookupElementBuilder.forField(psiField, psiField.getName(), myPsiClass)
                             .withTypeText(psiField.getType().getPresentableText()));
        }
      }
      return ArrayUtil.toObjectArray(staticFields);
    }

    @NotNull
    @Override
    public String getUnresolvedMessagePattern() {
      final String fieldName = getValue();
      if (fieldName.isEmpty() || fieldName.equals(".")) {
        return SpringApiBundle.message("FieldRetrievingFactoryBeanConverter.field.name.expected");
      }
      return SpringApiBundle.message("FieldRetrievingFactoryBeanConverter.cannot.resolve.field", fieldName);
    }
  }

  /**
   * Highlights missing field reference.
   */
  public static class FieldReferenceRequired extends FieldRetrievingFactoryBeanConverter {

    @Override
    protected boolean requireFieldReference() {
      return true;
    }
  }
}
