// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.values;

import com.intellij.lang.injection.InjectedLanguageManager;
import com.intellij.psi.PsiEllipsisType;
import com.intellij.psi.PsiLanguageInjectionHost;
import com.intellij.psi.PsiType;
import com.intellij.psi.xml.XmlAttribute;
import com.intellij.psi.xml.XmlElement;
import com.intellij.spring.model.xml.beans.TypeHolder;
import com.intellij.spring.model.xml.beans.TypeHolderUtil;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.Converter;
import com.intellij.util.xml.DomElement;
import com.intellij.util.xml.GenericDomValue;
import com.intellij.util.xml.WrappingConverter;
import com.intellij.util.xml.converters.values.GenericDomValueConvertersRegistry;
import org.jetbrains.annotations.NotNull;

import java.util.Collections;
import java.util.List;

/**
 * @author Yann C&eacute;bron
 */
public abstract class PropertyValueConverter extends WrappingConverter {

  @Override
  public Converter getConverter(@NotNull final GenericDomValue domElement) {
    final List<Converter> converters = getConverters(domElement);
    return converters.isEmpty() ? null : converters.get(0);
  }

  @Override
  @NotNull
  public List<Converter> getConverters(@NotNull final GenericDomValue element) {
    XmlElement xmlElement = element.getXmlElement();
    if (xmlElement instanceof XmlAttribute) {
      PsiLanguageInjectionHost host = (PsiLanguageInjectionHost)((XmlAttribute)xmlElement).getValueElement();
      if (host == null || InjectedLanguageManager.getInstance(host.getProject()).getInjectedPsiFiles(host) != null) {
        return Collections.emptyList();
      }
    }
    final GenericDomValueConvertersRegistry registry = SpringValueConvertersRegistry.getInstance();

    final List<PsiType> types = getValueTypes(element);
    if (types.isEmpty()) {
      final Converter converter = registry.getConverter(element, null);
      return ContainerUtil.createMaybeSingletonList(converter);
    }

    final List<Converter> converters = new SmartList<>();
    for (PsiType type : types) {
      final Converter converter = registry.getConverter(element, type instanceof PsiEllipsisType ? ((PsiEllipsisType)type).getComponentType() : type);
      if (converter != null) {
        converters.add(converter);
      }
      else {
        return Collections.emptyList();
      }
    }
    return converters;
  }

  @NotNull
  public List<PsiType> getValueTypes(final GenericDomValue element) {
    if (element instanceof TypeHolder) {
      final List<PsiType> psiTypes = TypeHolderUtil.getRequiredTypes(((TypeHolder)element));
      if (!psiTypes.isEmpty()) {
        return psiTypes;
      }
    }

    final DomElement parent = element.getParent();
    return parent instanceof TypeHolder ? TypeHolderUtil.getRequiredTypes(((TypeHolder)parent)) : Collections.emptyList();
  }
}
