// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.utils.resources;

import com.intellij.openapi.fileTypes.FileTypes;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.Condition;
import com.intellij.openapi.util.Conditions;
import com.intellij.openapi.util.NotNullLazyValue;
import com.intellij.psi.*;
import com.intellij.psi.impl.RenameableFakePsiElement;
import com.intellij.util.Function;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;
import java.util.Collection;

public final class SpringResourcesBuilder {
  @NotNull private final PsiElement myElement;
  @NotNull private final String myText;
  private boolean myFromRoot;
  private boolean myFromCurrent;
  private int myOffset;
  private boolean mySoft = true;
  @NotNull private Condition<PsiFileSystemItem> myFilter = Conditions.alwaysTrue();
  private boolean myEndingSlashNotAllowed = true;
  @Nullable private Function<PsiFile, Collection<PsiFileSystemItem>> myCustomDefaultPathEvaluator = null;
  private Module @NotNull [] myModules = Module.EMPTY_ARRAY;
  @Nullable private String myTemplateName;

  private SpringResourcesBuilder(final @NotNull PsiElement element,
                                 final @NotNull String text) {
    this(element, text, ElementManipulators.getOffsetInElement(element));
  }

  private SpringResourcesBuilder(final @NotNull PsiElement element,
                                 final @NotNull String text,
                                 int offset) {
    myElement = element;
    myText = text;
    myOffset = offset;
  }

  public static SpringResourcesBuilder create(final @NotNull PsiElement element,
                                              final @NotNull String s) {
    return new SpringResourcesBuilder(element, s);
  }

  public static SpringResourcesBuilder create(final @NotNull PsiElement element,
                                              final @NotNull String s,
                                              int offset) {
    return new SpringResourcesBuilder(element, s, offset);
  }

  public static SpringResourcesBuilder create(final @NotNull Module module,
                                              final @NotNull String s,
                                              int offset) {
    return new SpringResourcesBuilder(new DummyPsiElement(module), s, offset);
  }

  public SpringResourcesBuilder fromRoot(boolean fromRoot) {
    myFromRoot = fromRoot;
    return this;
  }

  public SpringResourcesBuilder fromCurrent(boolean fromCurrent) {
    myFromCurrent = fromCurrent;
    return this;
  }

  public SpringResourcesBuilder offset(int offset) {
    myOffset = offset;
    return this;
  }

  public SpringResourcesBuilder soft(boolean soft) {
    mySoft = soft;
    return this;
  }

  public SpringResourcesBuilder endingSlashNotAllowed(boolean endingSlashNotAllowed) {
    myEndingSlashNotAllowed = endingSlashNotAllowed;
    return this;
  }

  public SpringResourcesBuilder filter(@NotNull Condition<PsiFileSystemItem> filter) {
    myFilter = filter;
    return this;
  }

  public SpringResourcesBuilder customDefaultPathEvaluator(@Nullable Function<PsiFile, Collection<PsiFileSystemItem>> customDefaultPathEvaluator) {
    myCustomDefaultPathEvaluator = customDefaultPathEvaluator;
    return this;
  }

  public SpringResourcesBuilder modules(Module @NotNull ... modules) {
    myModules = modules;
    return this;
  }

  /**
   * Sets template name to use when creating not existing file.
   *
   * @param templateName File template name.
   * @return This.
   * @see com.intellij.ide.fileTemplates.FileTemplateManager
   */
  public SpringResourcesBuilder newFileTemplateName(String templateName) {
    myTemplateName = templateName;
    return this;
  }

  @NotNull
  PsiElement getElement() {
    return myElement;
  }

  @NotNull
  String getText() {
    return myText;
  }

  boolean isFromRoot() {
    return myFromRoot;
  }

  boolean isFromCurrent() {
    return myFromCurrent;
  }

  int getOffset() {
    return myOffset;
  }

  boolean isSoft() {
    return mySoft;
  }

  @NotNull
  Condition<PsiFileSystemItem> getFilter() {
    return myFilter;
  }

  boolean isEndingSlashNotAllowed() {
    return myEndingSlashNotAllowed;
  }

  @Nullable
  Function<PsiFile, Collection<PsiFileSystemItem>> getCustomDefaultPathEvaluator() {
    return myCustomDefaultPathEvaluator;
  }

  Module @NotNull [] getModules() {
    return myModules;
  }

  @Nullable
  String getTemplateName() {
    return myTemplateName;
  }

  /**
   * It is a hack to deceive the {@link SpringResourcesUtil} employing it's power to resolve Spring-syntax references (like classpath-prefixed)
   * even when there is no PsiElement is available.
   * This {@link DummyPsiElement} provides the access to {@link Module} and {@link Project} info which is required for resolve in {@link SpringResourcesUtil}
   */
  private static final class DummyPsiElement extends RenameableFakePsiElement {
    private final NotNullLazyValue<PsiFile> myFileNotNullLazyValue;
    private final Module myModule;

    private DummyPsiElement(Module module) {
      super(null);
      myModule = module;
      myFileNotNullLazyValue = NotNullLazyValue.lazy(() -> {
        PsiFile psiFile = PsiFileFactory.getInstance(getProject()).createFileFromText("DummyFile.txt", FileTypes.PLAIN_TEXT, "");
        psiFile.putUserData(ModuleUtilCore.KEY_MODULE, myModule);
        return psiFile;
      });
    }

    @NotNull
    @Override
    public Project getProject() {
      return myModule.getProject();
    }

    @Override
    public PsiFile getContainingFile() {
      return myFileNotNullLazyValue.getValue();
    }

    @Override
    public String getName() {
      return null;
    }

    @Nullable
    @Override
    public String getTypeName() {
      return null;
    }

    @Nullable
    @Override
    public Icon getIcon() {
      return null;
    }
  }

}
