// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.utils

import com.intellij.openapi.progress.ProgressManager
import com.intellij.spring.model.CommonSpringBean
import com.intellij.spring.model.SpringBeanPointer
import com.intellij.util.Processor
import com.intellij.util.SmartList
import com.intellij.util.containers.FactoryMap

/**
 * Processor which collects [SpringBeanPointer]s ignoring beans from "ComponentScan" if they were explicitly redefined
 * in Java or XML configuration.
 */
class ExplicitRedefinitionAwareBeansCollector<T : CommonSpringBean> : Processor<SpringBeanPointer<T>> {

  private val beansByNameMap = FactoryMap.createMap<String, MutableList<SpringBeanPointer<T>>>(
    { SmartList() },
    { linkedMapOf() }
  )

  val result: Set<SpringBeanPointer<T>>
    get() = beansByNameMap.values.flatMapTo(linkedSetOf<SpringBeanPointer<T>>()) { mixedBeans ->
      mixedBeans.takeIf { it.size <= 1 } ?:
      mixedBeans.groupBy { it.beanClass?.qualifiedName ?: "" }.values.flatMap { mixedBeansOfOneType ->
        mixedBeansOfOneType.takeIf { it.size <= 1 } ?:
        mixedBeansOfOneType.filter { isExplicitlyDefined(it) }.takeIf { it.isNotEmpty() } ?:
        mixedBeansOfOneType
      }
    }

  override fun process(pointer: SpringBeanPointer<T>): Boolean {
    ProgressManager.checkCanceled()
    beansByNameMap.getValue(pointer.name ?: "").add(pointer)
    return true
  }

  private fun isExplicitlyDefined(pointer: SpringBeanPointer<T>) = pointer.psiElement?.isPhysical ?: false
}