// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.jam.utils.filters;

import com.intellij.jam.JamService;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.JavaPsiFacade;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiPackage;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.search.PackageScope;
import com.intellij.psi.search.searches.AnnotatedElementsSearch;
import com.intellij.spring.model.jam.JamPsiMemberSpringBean;
import com.intellij.spring.model.jam.stereotype.CustomSpringComponent;
import com.intellij.spring.model.jam.stereotype.SpringStereotypeElement;
import com.intellij.util.CommonProcessors;
import com.intellij.util.Processor;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Set;

public class SpringContextIncludeAnnotationFilter extends SpringContextFilter.IncludeExpression {

  public SpringContextIncludeAnnotationFilter(@Nullable String expression) {
    super(expression);
  }

  @NotNull
  @Override
  public Set<SpringStereotypeElement> includeStereotypes(@NotNull Module module, @NotNull Set<PsiPackage> packages) {
    final String annotation = getExpression();
    if (!StringUtil.isEmptyOrSpaces(annotation)) {
      GlobalSearchScope searchScope = GlobalSearchScope.moduleWithDependenciesAndLibrariesScope(module);
      final PsiClass annotationClass = JavaPsiFacade.getInstance(module.getProject()).findClass(annotation, searchScope);

      return getAnnotatedStereotypes(annotationClass, searchScope, packages);
    }
    return Collections.emptySet();
  }

  @NotNull
  public static Set<SpringStereotypeElement> getAnnotatedStereotypes(@Nullable PsiClass annotationClass,
                                                                     @NotNull GlobalSearchScope searchScope,
                                                                     @NotNull Set<PsiPackage> packages) {
    if (annotationClass == null || !annotationClass.isAnnotationType()) return Collections.emptySet();

    final Set<SpringStereotypeElement> components = new LinkedHashSet<>();
    final Set<PsiClass> annotatedClasses = getAnnotatedClasses(annotationClass, packages, searchScope);
    for (PsiClass annotatedClass : annotatedClasses) {
      JamPsiMemberSpringBean bean = JamService.getJamService(annotatedClass.getProject())
        .getJamElement(JamPsiMemberSpringBean.PSI_MEMBER_SPRING_BEAN_JAM_KEY, annotatedClass);
      if (bean instanceof SpringStereotypeElement) {
        components.add((SpringStereotypeElement)bean);
      } else {
        components.add(new CustomSpringComponent(annotationClass.getQualifiedName(), annotatedClass));
      }
    }

    return components;
  }

  private static Set<PsiClass> getAnnotatedClasses(@NotNull PsiClass annotationClass,
                                                   @NotNull Set<PsiPackage> packages,
                                                   @NotNull GlobalSearchScope searchScope) {
    final Set<PsiClass> annotatedClasses = new LinkedHashSet<>();
    Processor<PsiClass> processor = new CommonProcessors.CollectProcessor<>(annotatedClasses);

    for (PsiPackage psiPackage : packages) {
      final GlobalSearchScope scope = searchScope.intersectWith(PackageScope.packageScope(psiPackage, true));
      AnnotatedElementsSearch.searchPsiClasses(annotationClass, scope).forEach(processor);
    }
    return annotatedClasses;
  }
}
