/*
 * Copyright 2000-2013 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.utils.filters;

import com.intellij.openapi.module.Module;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiPackage;
import com.intellij.spring.model.jam.stereotype.SpringStereotypeElement;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.Set;

public abstract class SpringContextFilter {

  public static abstract class Exclude extends SpringContextFilter {

    public static final Exclude EMPTY_EXCLUDE = new SpringContextFilter.Exclude() {
      @Override
      public boolean exclude(@NotNull PsiClass psiClass) {
        return false;
      }
    };

    public abstract boolean exclude(@NotNull PsiClass psiClass);
  }

  public static abstract class ExcludeExpression extends Exclude {
    @Nullable private final String myExpression;

    protected ExcludeExpression(@Nullable String expression) {
      myExpression = expression;
    }

    @Nullable
    public String getExpression() {
      return myExpression;
    }
  }

  public static abstract class ExcludeClasses extends Exclude {
    private final Collection<PsiClass> myClasses;

    protected ExcludeClasses(@NotNull Collection<PsiClass> classes) {
      myClasses = classes;
    }

    public Collection<PsiClass> getClasses() {
      return myClasses;
    }
  }

  public static abstract class Include extends SpringContextFilter {
    public static final SpringContextFilter.Include EMPTY_INCLUDE = new SpringContextFilter.Include() {
      @NotNull
      @Override
      public Set<SpringStereotypeElement> includeStereotypes(@NotNull Module module, @NotNull Set<PsiPackage> packages) {
        return Collections.emptySet();
      }
    };

    @NotNull
    public abstract Set<SpringStereotypeElement> includeStereotypes(@NotNull Module module, @NotNull Set<PsiPackage> packages);
  }

  public static abstract class IncludeExpression extends Include {
    @Nullable private final String myExpression;

    protected IncludeExpression(@Nullable String expression) {
      myExpression = expression;
    }

    @Nullable
    public String getExpression() {
      return myExpression;
    }
  }

  public static abstract class IncludeClasses extends Include {
    private final Collection<PsiClass> myClasses;

    protected IncludeClasses(@NotNull Collection<PsiClass> classes) {
      myClasses = classes;
    }

    public Collection<PsiClass> getClasses() {
      return myClasses;
    }
  }
}
