// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.jam.testContexts.profiles;

import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.reflect.JamAnnotationMeta;
import com.intellij.jam.reflect.JamAttributeMeta;
import com.intellij.jam.reflect.JamMemberMeta;
import com.intellij.openapi.util.NullableLazyValue;
import com.intellij.psi.*;
import com.intellij.psi.ref.AnnotationChildLink;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.semantic.SemKey;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.spring.model.aliasFor.SpringAliasFor;
import com.intellij.spring.model.aliasFor.SpringAliasForUtils;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.HashSet;
import java.util.Set;

public class SpringCustomActiveProfiles implements SpringActiveProfile {

  public static final SemKey<JamAnnotationMeta> JAM_ANNO_META_KEY = ACTIVE_PROFILE_JAM_ANNOTATION_KEY
    .subKey("CustomContextProfile");

  public static final SemKey<SpringCustomActiveProfiles> JAM_KEY = ACTIVE_PROFILE_JAM_KEY.subKey("SpringCustomActiveProfiles");
  public static final SemKey<JamMemberMeta<PsiMember, SpringCustomActiveProfiles>> META_KEY =
    ACTIVE_PROFILE_META_KEY.subKey("CustomContextProfile");

  private final PsiElementRef<PsiAnnotation> myPsiAnnotation;
  private final AnnotationChildLink myAnnotationChildLink;

  private final PsiAnchor myPsiMemberAnchor;

  private final NullableLazyValue<SpringJamActiveProfiles> myDefiningMetaAnnotation =
    new NullableLazyValue<>() {
      @Nullable
      @Override
      protected SpringJamActiveProfiles compute() {
        PsiMember element = getPsiElement();
        if (element == null) return null;

        final PsiAnnotation definingMetaAnnotation =
          SpringAliasForUtils.findDefiningMetaAnnotation(element, myAnnotationChildLink.getAnnotationQualifiedName(),
                                                         SpringAnnotationsConstants.ACTIVE_PROFILES);
        if (definingMetaAnnotation != null) {
          final PsiClass annotationType = PsiTreeUtil.getParentOfType(definingMetaAnnotation, PsiClass.class, true);
          if (annotationType != null) {
            return SpringJamActiveProfiles.META.getJamElement(annotationType);
          }
        }
        return null;
      }
    };

  public SpringCustomActiveProfiles(@NotNull String anno, @NotNull PsiMember psiMember) {
    myAnnotationChildLink = new AnnotationChildLink(anno);
    myPsiMemberAnchor = PsiAnchor.create(psiMember);
    myPsiAnnotation = myAnnotationChildLink.createChildRef(psiMember);
  }

  @Override
  public PsiMember getPsiElement() {
    return (PsiClass)myPsiMemberAnchor.retrieve();
  }

  @Override
  @Nullable
  public PsiAnnotation getAnnotation() {
    return myPsiAnnotation.getPsiElement();
  }

  @Nullable
  @Override
  public PsiElement getIdentifyingPsiElement() {
    return getPsiElement();
  }

  @NotNull
  @Override
  public Set<String> getActiveProfiles() {
    Set<String> profiles = new HashSet<>();

    for (String profilesAttr : PROFILES_ATTRS) {
      SpringAliasFor aliasFor = getAliasAttribute(profilesAttr);
      if (aliasFor != null) {
        for (JamStringAttributeElement<String> element : JamAttributeMeta.collectionString(aliasFor.getMethodName())
          .getJam(myPsiAnnotation)) {
          ContainerUtil.addIfNotNull(profiles, element.getValue());
        }
        return profiles;
      }
    }
    SpringJamActiveProfiles definingProfile = myDefiningMetaAnnotation.getValue();
    if (definingProfile != null) return definingProfile.getActiveProfiles();

    return profiles;
  }

  @Nullable
  private SpringAliasFor getAliasAttribute(@NotNull String attrName) {
    PsiMember element = getPsiElement();
    return element == null ? null : SpringAliasForUtils.findAliasFor(element,
                                                                     myAnnotationChildLink.getAnnotationQualifiedName(),
                                                                     SpringAnnotationsConstants.ACTIVE_PROFILES,
                                                                     attrName);
  }
}