// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.jam.testContexts.jdbc;

import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.reflect.*;
import com.intellij.psi.*;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.semantic.SemKey;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.spring.model.jam.testContexts.converters.PsiFileResourcePathReferenceConverter;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import static com.intellij.jam.reflect.JamAttributeMeta.collectionString;
import static com.intellij.jam.reflect.JamAttributeMeta.singleAnno;

public class SpringTestingSql implements TestingSql {
  private static final SemKey<JamAnnotationMeta> JAM_ANNO_META_KEY = JAM_ANNOTATION_KEY.subKey("SpringTestingSql");
  public static final SemKey<SpringTestingSql> REPEATABLE_ANNO_JAM_KEY = TestingSql.JAM_KEY.subKey("SpringTestingSql");

  public static final JamClassMeta<SpringTestingSql> CLASS_META = new JamClassMeta<>(null, SpringTestingSql.class, REPEATABLE_ANNO_JAM_KEY);
  public static final JamMethodMeta<SpringTestingSql> METHOD_META = new JamMethodMeta<>(null, SpringTestingSql.class, REPEATABLE_ANNO_JAM_KEY);

  private static final JamAnnotationAttributeMeta.Single<SpringTestingSqlConfig> SQL_CONFIG_ATTR =
    singleAnno(CONFIG_ATTR_NAME, SpringTestingSqlConfig.ANNO_META, SpringTestingSqlConfig.class);

  private static final JamStringAttributeMeta.Collection<List<PsiFile>> SCRIPTS_ATTR_META =
    collectionString(SCRIPTS_ATTR_NAME, new PsiFileResourcePathReferenceConverter());

  private static final JamStringAttributeMeta.Collection<List<PsiFile>> VALUE_ATTR_META =
    collectionString(VALUE_ATTR_NAME, new PsiFileResourcePathReferenceConverter());

  private static final JamAnnotationArchetype ARCHETYPE = new JamAnnotationArchetype()
    .addAttribute(SCRIPTS_ATTR_META)
    .addAttribute(VALUE_ATTR_META)
    .addAttribute(SQL_CONFIG_ATTR);

  public static final JamAnnotationMeta ANNO_META =
    new JamAnnotationMeta(SpringAnnotationsConstants.TEST_SQL, ARCHETYPE, JAM_ANNO_META_KEY);

  static {
    CLASS_META.addAnnotation(ANNO_META);
    METHOD_META.addAnnotation(ANNO_META);
  }

  private final PsiMember myPsiMember;
  private final PsiElementRef<PsiAnnotation> myPsiAnnotation;

  @SuppressWarnings("unused")
  public SpringTestingSql(@NotNull PsiMember psiMember) {
    myPsiMember = psiMember;
    myPsiAnnotation = ANNO_META.getAnnotationRef(psiMember);
  }

  @SuppressWarnings("unused")
  public SpringTestingSql(PsiAnnotation annotation) {
    myPsiMember = PsiTreeUtil.getParentOfType(annotation, PsiMember.class, true);
    myPsiAnnotation = PsiElementRef.real(annotation);
  }

  @Nullable
  public SpringTestingSqlConfig getSqlConfig() {
    PsiAnnotation element = myPsiAnnotation.getPsiElement();
    if (element != null) {
      PsiAnnotationMemberValue config = element.findDeclaredAttributeValue(CONFIG_ATTR_NAME);
      if (config instanceof PsiAnnotation) {
        return SQL_CONFIG_ATTR.getNestedJam(PsiElementRef.real((PsiAnnotation)config));
      }
    }
    return null;
  }

  @Nullable
  public PsiMember getPsiElement() {
    return myPsiMember;
  }

  @Nullable
  public PsiAnnotation getAnnotation() {
    return myPsiAnnotation.getPsiElement();
  }

  @NotNull
  @Override
  public Set<PsiFile> getScripts() {
    Set<PsiFile> locations = new LinkedHashSet<>();
    addFiles(locations, VALUE_ATTR_META.getJam(myPsiAnnotation));
    addFiles(locations, SCRIPTS_ATTR_META.getJam(myPsiAnnotation));
    return locations;
  }

  @NotNull
  @Override
  public List<JamStringAttributeElement<List<PsiFile>>> getScriptElements() {
    return ContainerUtil.concat(
      VALUE_ATTR_META.getJam(myPsiAnnotation),
      SCRIPTS_ATTR_META.getJam(myPsiAnnotation)
    );
  }

  private static void addFiles(@NotNull Set<PsiFile> locations,
                               @NotNull List<JamStringAttributeElement<List<PsiFile>>> jam) {
    for (JamStringAttributeElement<List<PsiFile>> element : jam) {
      List<PsiFile> value = element.getValue();
      if (value != null) {
        ContainerUtil.addAllNotNull(locations, value);
      }
    }
  }
}
