// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.jam.testContexts;

import com.intellij.jam.JamClassAttributeElement;
import com.intellij.jam.JamService;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.annotations.JamPsiValidity;
import com.intellij.jam.model.util.JamCommonUtil;
import com.intellij.jam.reflect.*;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElementRef;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.psi.xml.XmlFile;
import com.intellij.semantic.SemKey;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.spring.model.jam.testContexts.converters.ApplicationContextReferenceConverter;
import com.intellij.spring.model.jam.testContexts.profiles.SpringActiveProfile;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

public abstract class SpringContextConfiguration implements ContextConfiguration {
  public static final SemKey<JamAnnotationMeta> JAM_ANNO_META_KEY = CONTEXT_CONFIGURATION_JAM_ANNOTATION_KEY
    .subKey("SpringContextConfiguration");

  private static final SemKey<SpringContextConfiguration> JAM_KEY =
    ContextConfiguration.CONTEXT_CONFIGURATION_JAM_KEY.subKey("SpringContextConfiguration");
  public static final JamClassMeta<SpringContextConfiguration> META = new JamClassMeta<>(null, SpringContextConfiguration.class, JAM_KEY);

  public static final JamClassAttributeMeta.Single LOADER_ATTR_META = JamAttributeMeta.singleClass(LOADER_ATTR_NAME);

  public static final JamStringAttributeMeta.Collection<List<XmlFile>> LOCATION_ATTR_META =
    JamAttributeMeta.collectionString(LOCATIONS_ATTR_NAME, new ApplicationContextReferenceConverter());

  public static final JamStringAttributeMeta.Collection<List<XmlFile>> VALUE_ATTR_META =
    JamAttributeMeta.collectionString(VALUE_ATTR_NAME, new ApplicationContextReferenceConverter());

  protected static final JamClassAttributeMeta.Collection CLASSES_ATTR_META =
    new JamClassAttributeMeta.Collection(CLASSES_ATTR_NAME);

  protected static final JamClassAttributeMeta.Collection INITIALIZERS_ATTR_META =
    new JamClassAttributeMeta.Collection("initializers");

  public static final JamAnnotationArchetype ARCHETYPE =
    new JamAnnotationArchetype()
      .addAttribute(LOCATION_ATTR_META)
      .addAttribute(VALUE_ATTR_META)
      .addAttribute(CLASSES_ATTR_META)
      .addAttribute(INITIALIZERS_ATTR_META);

  public static final JamAnnotationMeta ANNO_META =
    new JamAnnotationMeta(SpringAnnotationsConstants.CONTEXT_CONFIGURATION, ARCHETYPE, JAM_ANNO_META_KEY);

  static {
    META.addAnnotation(ANNO_META);
  }

  private final PsiClass myPsiClass;
  private final PsiElementRef<PsiAnnotation> myPsiAnnotation;


  @SuppressWarnings("unused")
  public SpringContextConfiguration(@NotNull PsiClass psiClass) {
    myPsiClass = psiClass;
    myPsiAnnotation = ANNO_META.getAnnotationRef(psiClass);
  }

  @SuppressWarnings("unused")
  public SpringContextConfiguration(PsiAnnotation annotation) {
    myPsiClass = PsiTreeUtil.getParentOfType(annotation, PsiClass.class, true);
    myPsiAnnotation = PsiElementRef.real(annotation);
  }

  @NotNull
  public List<JamStringAttributeElement<List<XmlFile>>> getLocationsAttributeElement() {
    return LOCATION_ATTR_META.getJam(myPsiAnnotation);
  }

  @NotNull
  public List<JamStringAttributeElement<List<XmlFile>>> getValueAttributeElement() {
    return VALUE_ATTR_META.getJam(myPsiAnnotation);
  }

  public boolean isInheritLocations() {
    PsiAnnotation annotation = getAnnotation();
    Boolean value = annotation == null ? null :
                    JamCommonUtil.getObjectValue(annotation.findAttributeValue("inheritLocations"), Boolean.class);

    return value == null || value.booleanValue();
  }

  @Override
  @Nullable
  public PsiClass getPsiElement() {
    return myPsiClass;
  }

  @JamPsiValidity
  public abstract boolean isPsiValid();

  @Override
  @Nullable
  public PsiAnnotation getAnnotation() {
    return myPsiAnnotation.getPsiElement();
  }

  @Override
  @NotNull
  public Set<XmlFile> getLocations(PsiClass @NotNull ... contexts) {
    Set<XmlFile> locations = new LinkedHashSet<>();

    addFiles(contexts, locations, getValueAttributeElement());
    addFiles(contexts, locations, getLocationsAttributeElement());

    return locations;
  }

  private void addFiles(PsiClass @NotNull [] contexts,
                        @NotNull Set<XmlFile> locations,
                        @NotNull List<JamStringAttributeElement<List<XmlFile>>> elements) {
    for (JamStringAttributeElement<List<XmlFile>> stringAttributeElement : elements) {
      for (PsiClass context : contexts) {
        if (context.equals(getPsiElement())) {
          List<XmlFile> xmlFiles = stringAttributeElement.getValue();
          if (xmlFiles != null) {
            locations.addAll(xmlFiles);
          }
        }
        else {
          final String value = stringAttributeElement.getStringValue();
          if (value != null) {
            locations.addAll(ApplicationContextReferenceConverter.getApplicationContexts(value, context));
          }
        }
      }
    }
  }

  @Override
  @NotNull
  public List<PsiClass> getConfigurationClasses() {
    List<PsiClass> imported = new SmartList<>();
    for (JamClassAttributeElement jamClassAttributeElement : CLASSES_ATTR_META.getJam(myPsiAnnotation)) {
      ContainerUtil.addIfNotNull(imported, jamClassAttributeElement.getValue());
    }
    return imported;
  }

  public Set<String> getActiveProfiles() {
    final PsiClass psiClass = getPsiElement();
    if (psiClass == null) return Collections.emptySet();

    final SpringActiveProfile profiles = getSpringActiveProfiles(psiClass);
    return profiles != null ? profiles.getActiveProfiles() : Collections.emptySet();
  }

  @Nullable
  public static SpringActiveProfile getSpringActiveProfiles(@NotNull PsiClass psiClass) {
    return JamService.getJamService(psiClass.getProject()).getJamElement(SpringActiveProfile.ACTIVE_PROFILE_JAM_KEY, psiClass);
  }

  @Override
  public boolean hasLocationsAttribute() {
    return !getLocationsAttributeElement().isEmpty();
  }

  @Override
  public boolean hasValueAttribute() {
    return !getValueAttributeElement().isEmpty();
  }

  @Nullable
  @Override
  public PsiClass getLoaderClass() {
    return LOADER_ATTR_META.getJam(myPsiAnnotation).getValue();
  }

  @NotNull
  @Override
  public List<JamStringAttributeElement<List<XmlFile>>> getLocationElements() {
    return ContainerUtil.concat(
      getValueAttributeElement(),
      getLocationsAttributeElement()
    );
  }
}
