// Copyright 2000-2021 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.jam.testContexts;

import com.intellij.jam.JamClassAttributeElement;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.reflect.*;
import com.intellij.openapi.util.NullableLazyValue;
import com.intellij.psi.PsiAnchor;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElementRef;
import com.intellij.psi.ref.AnnotationChildLink;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.psi.xml.XmlFile;
import com.intellij.semantic.SemKey;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.spring.model.aliasFor.SpringAliasFor;
import com.intellij.spring.model.aliasFor.SpringAliasForUtils;
import com.intellij.spring.model.jam.testContexts.converters.ApplicationContextReferenceConverter;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

public class CustomContextConfiguration implements ContextConfiguration {

  public static final SemKey<JamAnnotationMeta> JAM_ANNO_META_KEY = CONTEXT_CONFIGURATION_JAM_ANNOTATION_KEY
    .subKey("CustomContextConfiguration");

  public static final SemKey<CustomContextConfiguration> JAM_KEY =
    ContextConfiguration.CONTEXT_CONFIGURATION_JAM_KEY.subKey("CustomContextConfiguration");
  public static final SemKey<JamMemberMeta<PsiClass, CustomContextConfiguration>> META_KEY =
    ContextConfiguration.CONTEXT_CONFIGURATION_META_KEY.subKey("CustomContextConfiguration");

  protected final PsiElementRef<PsiAnnotation> myPsiAnnotation;

  private final AnnotationChildLink myAnnotationChildLink;

  private final PsiAnchor myPsiClassAnchor;

  protected static final JamClassAttributeMeta.Collection CLASSES_ATTR_META = new JamClassAttributeMeta.Collection(CLASSES_ATTR_NAME);
  protected static final JamClassAttributeMeta.Collection VALUE_ATTR_META = new JamClassAttributeMeta.Collection(VALUE_ATTR_NAME);
  private final NullableLazyValue<SpringContextConfiguration> myDefiningMetaAnnotation =
    new NullableLazyValue<>() {
      @Nullable
      @Override
      protected SpringContextConfiguration compute() {
        final PsiAnnotation definingMetaAnnotation = SpringAliasForUtils
          .findDefiningMetaAnnotation(getPsiElement(), myAnnotationChildLink.getAnnotationQualifiedName(),
                                      SpringAnnotationsConstants.CONTEXT_CONFIGURATION, true);
        if (definingMetaAnnotation != null) {
          final PsiClass annotationType = PsiTreeUtil.getParentOfType(definingMetaAnnotation, PsiClass.class, true);
          if (annotationType != null) {
            return SpringContextConfiguration.META.getJamElement(annotationType);
          }
        }
        return null;
      }
    };

  public CustomContextConfiguration(@NotNull String anno, @NotNull PsiClass psiClassAnchor) {
    myAnnotationChildLink = new AnnotationChildLink(anno);
    myPsiClassAnchor = PsiAnchor.create(psiClassAnchor);
    myPsiAnnotation = myAnnotationChildLink.createChildRef(getPsiElement());
  }

  @Override
  public PsiClass getPsiElement() {
    return (PsiClass)myPsiClassAnchor.retrieve();
  }

  @Override
  @Nullable
  public PsiAnnotation getAnnotation() {
    return myPsiAnnotation.getPsiElement();
  }

  @Override
  @NotNull
  public Set<XmlFile> getLocations(PsiClass @NotNull ... contexts) {
    Set<XmlFile> files = new LinkedHashSet<>();
    for (String attrName : XML_FILES_ATTRS) {
      SpringAliasFor aliasFor = getAliasAttribute(attrName);
      if (aliasFor != null) {
        JamStringAttributeMeta.Collection<List<XmlFile>> collection =
          JamAttributeMeta.collectionString(aliasFor.getMethodName(), new ApplicationContextReferenceConverter());
        for (JamStringAttributeElement<List<XmlFile>> element : collection.getJam(myPsiAnnotation)) {
          List<XmlFile> values = element.getValue();
          if (values != null) files.addAll(values);
        }
        return files;
      }
    }

    final ContextConfiguration definingContextConfiguration = getDefiningContextConfiguration();
    if (definingContextConfiguration != null) {
      return definingContextConfiguration.getLocations(definingContextConfiguration.getPsiElement());
    }

    return Collections.emptySet();
  }

  @Override
  @NotNull
  public List<PsiClass> getConfigurationClasses() {
    List<PsiClass> psiClasses = new SmartList<>();
    SpringAliasFor aliasFor = getAliasAttribute(CLASSES_ATTR_NAME);
    if (aliasFor != null) {
      JamClassAttributeMeta.Collection collection = new JamClassAttributeMeta.Collection(aliasFor.getMethodName());
      for (JamClassAttributeElement classAttributeElement : collection.getJam(myPsiAnnotation)) {
        ContainerUtil.addIfNotNull(psiClasses, classAttributeElement.getValue());
      }
      return psiClasses;
    }

    final ContextConfiguration definingContextConfiguration = getDefiningContextConfiguration();
    if (definingContextConfiguration != null) {
      return definingContextConfiguration.getConfigurationClasses();
    }

    return Collections.emptyList();
  }

  private ContextConfiguration getDefiningContextConfiguration() {
    return myDefiningMetaAnnotation.getValue();
  }

  @Override
  public boolean hasLocationsAttribute() {
    final SpringAliasFor aliasAttribute = getAliasAttribute(LOCATIONS_ATTR_NAME);
    if (aliasAttribute != null) return true;

    final ContextConfiguration definingContextConfiguration = getDefiningContextConfiguration();
    return definingContextConfiguration != null && definingContextConfiguration.hasLocationsAttribute();
  }

  @Override
  public boolean hasValueAttribute() {
    final SpringAliasFor aliasAttribute = getAliasAttribute(VALUE_ATTR_NAME);
    if (aliasAttribute != null) return true;

    final ContextConfiguration definingContextConfiguration = getDefiningContextConfiguration();
    return definingContextConfiguration != null && definingContextConfiguration.hasValueAttribute();
  }

  @Nullable
  @Override
  public PsiClass getLoaderClass() {
    SpringAliasFor aliasFor = getAliasAttribute(LOADER_ATTR_NAME);
    if (aliasFor != null) {
      return JamAttributeMeta.singleClass(aliasFor.getMethodName()).getJam(myPsiAnnotation).getValue();
    }
    return null;
  }

  private SpringAliasFor getAliasAttribute(@NotNull String attrName) {
    return SpringAliasForUtils.findAliasFor(getPsiElement(),
                                            myAnnotationChildLink.getAnnotationQualifiedName(),
                                            SpringAnnotationsConstants.CONTEXT_CONFIGURATION,
                                            attrName);
  }

  @NotNull
  @Override
  public List<JamStringAttributeElement<List<XmlFile>>> getLocationElements() {
    return ContainerUtil.concat(
      getLocations(getAliasAttribute(VALUE_ATTR_NAME)),
      getLocations(getAliasAttribute(LOCATIONS_ATTR_NAME))
    );
  }

  @NotNull
  private List<JamStringAttributeElement<List<XmlFile>>> getLocations(@Nullable SpringAliasFor aliasFor) {
    if (aliasFor == null) return Collections.emptyList();

    JamStringAttributeMeta.Collection<List<XmlFile>> collection =
      new JamStringAttributeMeta.Collection<>(aliasFor.getMethodName(), new ApplicationContextReferenceConverter());

    return collection.getJam(myPsiAnnotation);
  }
}
