// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.jam.stereotype;

import com.intellij.jam.JamClassAttributeElement;
import com.intellij.jam.JamElement;
import com.intellij.jam.model.common.CommonModelElement;
import com.intellij.jam.reflect.*;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElementRef;
import com.intellij.psi.PsiMember;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import org.jetbrains.annotations.NotNull;

import java.util.LinkedHashSet;
import java.util.Set;

public class SpringComponentScanFilter extends CommonModelElement.PsiBase implements JamElement {

  private static final JamClassAttributeMeta.Collection VALUE_ATTR_META = JamAttributeMeta.classCollection("value");
  private static final JamClassAttributeMeta.Collection CLASSES_ATTR_META = JamAttributeMeta.classCollection("classes");
  private static final JamEnumAttributeMeta.Single<FilterType> FILTER_TYPE_ATTR_META =
    JamAttributeMeta.singleEnum("type", FilterType.class);

  public static final JamAnnotationMeta ANNOTATION_META =
    new JamAnnotationMeta(SpringAnnotationsConstants.COMPONENT_SCAN_FILTER)
      .addAttribute(VALUE_ATTR_META)
      .addAttribute(CLASSES_ATTR_META)
      .addAttribute(FILTER_TYPE_ATTR_META);

  private final PsiMember myMember;
  private final PsiElementRef<PsiAnnotation> myAnnotation;

  @SuppressWarnings("unused")
  public SpringComponentScanFilter(PsiMember member) {
    myMember = member;
    myAnnotation = ANNOTATION_META.getAnnotationRef(member);
  }

  @SuppressWarnings("unused")
  public SpringComponentScanFilter(PsiAnnotation annotation) {
    myAnnotation = PsiElementRef.real(annotation);
    myMember = PsiTreeUtil.getParentOfType(annotation, PsiMember.class, true);
  }

  public static final JamClassMeta<SpringComponentScanFilter> META =
    new JamClassMeta<>(SpringComponentScanFilter.class).addAnnotation(ANNOTATION_META);

  public Set<PsiClass> getFilteredClasses() {
    Set<PsiClass> result = new LinkedHashSet<>();

    addClasses(result, VALUE_ATTR_META);
    addClasses(result, CLASSES_ATTR_META);

    return result;
  }

  private void addClasses(Set<PsiClass> result, JamClassAttributeMeta.Collection meta) {
    for (JamClassAttributeElement element : meta.getJam(myAnnotation)) {
      final PsiClass psiClass = element.getValue();
      if (psiClass != null) {
        result.add(psiClass);
      }
    }
  }

  public FilterType getFilterType() {
    final FilterType filterType = FILTER_TYPE_ATTR_META.getJam(myAnnotation).getValue();
    if (filterType != null) {
      return filterType;
    }
    return FilterType.ANNOTATION;
  }

  @NotNull
  @Override
  public PsiMember getPsiElement() {
    return myMember;
  }
}
