// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.jam.profile;

import com.intellij.ide.presentation.Presentation;
import com.intellij.jam.JamConverter;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.model.common.CommonModelElement;
import com.intellij.jam.reflect.*;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.*;
import com.intellij.psi.util.PartiallyKnownString;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.semantic.SemKey;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.spring.constants.SpringCorePresentationConstants;
import com.intellij.spring.profiles.SpringProfilesFactory;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.uast.UExpression;
import org.jetbrains.uast.UastContextKt;
import org.jetbrains.uast.expressions.UStringConcatenationsFacade;

import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

@Presentation(typeName = SpringCorePresentationConstants.PROFILE)
public class SpringJamProfile extends CommonModelElement.PsiBase implements SpringContextProfile {
  public static final String PROFILE_DELIMITERS = "()&|"; // delimiters are supported since Spring 5.2

  private final PsiElementRef<PsiAnnotation> myPsiAnnotation;
  private final PsiMember myPsiMember;

  public static final SemKey<JamAnnotationMeta> JAM_ANNO_META_KEY = CONTEXT_PROFILE_JAM_ANNOTATION_KEY.subKey("SpringJamProfile");
  private static final SemKey<SpringJamProfile> JAM_KEY = SpringContextProfile.CONTEXT_PROFILE_JAM_KEY.subKey("SpringJamProfile");

  public static final JamMemberMeta<PsiMember, SpringJamProfile> META = new JamMemberMeta<>(null, SpringJamProfile.class, JAM_KEY);
  private static final JamStringAttributeMeta.Collection<String> VALUE_ATTR_META =
    JamAttributeMeta.collectionString("value", new SpringProfileConverter(PROFILE_DELIMITERS, true));
  private static final JamAnnotationArchetype ARCHETYPE = new JamAnnotationArchetype().addAttribute(VALUE_ATTR_META);

  public static final JamAnnotationMeta ANNO_META = new JamAnnotationMeta(SpringAnnotationsConstants.PROFILE, ARCHETYPE, JAM_ANNO_META_KEY);

  static {
    META.addAnnotation(ANNO_META);
  }

  @SuppressWarnings("unused")
  public SpringJamProfile(@NotNull PsiMember psiMember) {
    myPsiMember = psiMember;
    myPsiAnnotation = ANNO_META.getAnnotationRef(psiMember);
  }

  @SuppressWarnings("unused")
  public SpringJamProfile(@NotNull PsiAnnotation annotation) {
    myPsiMember = PsiTreeUtil.getParentOfType(annotation, PsiMember.class, true);
    myPsiAnnotation = PsiElementRef.real(annotation);
  }


  @Override
  @NotNull
  public Set<String> getExpressions() {
    Set<String> profiles = new LinkedHashSet<>();
    for (JamStringAttributeElement<String> element : getValueElements()) {
      String value = element.getStringValue();
      if (!StringUtil.isEmptyOrSpaces(value)) {
        profiles.add(value.trim());
      }
    }
    return profiles;
  }

  @NotNull
  @Override
  public List<JamStringAttributeElement<String>> getValueElements() {
    return ANNO_META.getAttribute(myPsiMember, VALUE_ATTR_META);
  }

  @Override
  @NotNull
  public PsiMember getPsiElement() {
    return myPsiMember;
  }

  @Override
  @Nullable
  public PsiAnnotation getAnnotation() {
    return myPsiAnnotation.getPsiElement();
  }

  public static class SpringProfileConverter extends JamConverter<String> {
    private final String myDelimiters;
    private final boolean myIsDefinition;

    public SpringProfileConverter(String delimiters, boolean isDefinition) {
      myDelimiters = delimiters;
      myIsDefinition = isDefinition;
    }

    @Override
    public String fromString(@Nullable String s, JamStringAttributeElement<String> context) {
      return s;
    }

    @Override
    public PsiReference @NotNull [] createReferences(@NotNull JamStringAttributeElement<String> context,
                                                     @NotNull PsiLanguageInjectionHost injectionHost) {
      return createReferences(injectionHost);
    }

    public PsiReference @NotNull [] createReferences(@NotNull PsiLanguageInjectionHost injectionHost) {
      UExpression uExpression = UastContextKt.toUElement(injectionHost, UExpression.class);
      if (uExpression == null) return PsiReference.EMPTY_ARRAY;

      UStringConcatenationsFacade facade = UStringConcatenationsFacade.createFromTopConcatenation(uExpression);
      if (facade == null) return PsiReference.EMPTY_ARRAY;

      PartiallyKnownString pks = facade.asPartiallyKnownString();
      if (pks.getSegments().size() > 1) return PsiReference.EMPTY_ARRAY;

      Module module = ModuleUtilCore.findModuleForPsiElement(injectionHost);
      if (module == null) return PsiReference.EMPTY_ARRAY;

      return SpringProfilesFactory.getInstance()
        .getProfilesReferences(module, injectionHost, pks.getValueIfKnown(), 0, myDelimiters, myIsDefinition);
    }
  }
}