/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.profile;

import com.intellij.jam.JamElement;
import com.intellij.jam.JamService;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.reflect.JamAnnotationMeta;
import com.intellij.jam.reflect.JamMemberMeta;
import com.intellij.openapi.util.text.DelimitedListProcessor;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiMember;
import com.intellij.semantic.SemKey;
import com.intellij.spring.model.SpringProfile;
import com.intellij.spring.profiles.SpringProfilesFactory;
import com.intellij.spring.profiles.SpringProfilesFactory.MalformedProfileExpressionException;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

public interface SpringContextProfile extends JamElement, SpringProfile {
  String VALUE_ATTR_NAME = "value";

  SemKey<JamAnnotationMeta> CONTEXT_PROFILE_JAM_ANNOTATION_KEY = JamService.ANNO_META_KEY.subKey("SpringContextProfile");
  SemKey<SpringContextProfile> CONTEXT_PROFILE_JAM_KEY = JamService.JAM_ELEMENT_KEY.subKey("SpringContextProfile");
  SemKey<JamMemberMeta> CONTEXT_PROFILE_META_KEY = JamService.getMetaKey(CONTEXT_PROFILE_JAM_KEY);

  @Nullable
  PsiMember getPsiElement();

  @Nullable
  PsiAnnotation getAnnotation();

  @NotNull
  List<JamStringAttributeElement<String>> getValueElements();

  @NotNull
  @Override
  default Set<String> getNames() {
    return getExpressions().stream()
      .flatMap(expression -> {
        Set<String> names = new HashSet<>();
        new DelimitedListProcessor(SpringJamProfile.PROFILE_DELIMITERS) {
          @Override
          protected void processToken(int start, int end, boolean delimitersOnly) {
            String name = expression.substring(start, end);
            name = StringUtil.trimStart(name, "!").trim();
            if (!name.isEmpty()) {
              names.add(name);
            }
          }
        }.processText(expression);
        return names.stream();
      })
      .collect(Collectors.toSet());
  }

  @Override
  default boolean matches(@NotNull Set<String> activeProfiles) {
    try {
      Predicate<Set<String>> profiles =
        SpringProfilesFactory.getInstance().parseProfileExpressions(getExpressions());
      return profiles.test(activeProfiles);
    }
    catch (MalformedProfileExpressionException e) {
      return false;
    }
  }
}
