// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.jam.profile;

import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.reflect.JamAnnotationMeta;
import com.intellij.jam.reflect.JamAttributeMeta;
import com.intellij.jam.reflect.JamMemberMeta;
import com.intellij.openapi.util.NullableLazyValue;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.*;
import com.intellij.psi.ref.AnnotationChildLink;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.semantic.SemKey;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.spring.model.aliasFor.SpringAliasFor;
import com.intellij.spring.model.aliasFor.SpringAliasForUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

public class CustomContextProfile implements SpringContextProfile {

  public static final SemKey<JamAnnotationMeta> JAM_ANNO_META_KEY = CONTEXT_PROFILE_JAM_ANNOTATION_KEY
    .subKey("CustomContextProfile");

  public static final SemKey<CustomContextProfile> JAM_KEY = CONTEXT_PROFILE_JAM_KEY.subKey("CustomContextProfile");
  public static final SemKey<JamMemberMeta<PsiMember, CustomContextProfile>> META_KEY =
    CONTEXT_PROFILE_META_KEY.subKey("CustomContextProfile");

  private final PsiElementRef<PsiAnnotation> myPsiAnnotation;
  private final AnnotationChildLink myAnnotationChildLink;

  private final PsiAnchor myPsiMemberAnchor;

  private final NullableLazyValue<SpringJamProfile> myDefiningMetaAnnotation =
    new NullableLazyValue<>() {
      @Nullable
      @Override
      protected SpringJamProfile compute() {
        PsiMember element = getPsiElement();
        if (element == null) return null;

        final PsiAnnotation definingMetaAnnotation =
          SpringAliasForUtils.findDefiningMetaAnnotation(element, myAnnotationChildLink.getAnnotationQualifiedName(),
                                                         SpringAnnotationsConstants.PROFILE);
        if (definingMetaAnnotation != null) {
          final PsiClass annotationType = PsiTreeUtil.getParentOfType(definingMetaAnnotation, PsiClass.class, true);
          if (annotationType != null) {
            return SpringJamProfile.META.getJamElement(annotationType);
          }
        }
        return null;
      }
    };

  public CustomContextProfile(@NotNull String anno, @NotNull PsiMember psiMember) {
    myAnnotationChildLink = new AnnotationChildLink(anno);
    myPsiMemberAnchor = PsiAnchor.create(psiMember);
    myPsiAnnotation = myAnnotationChildLink.createChildRef(psiMember);
  }

  @Override
  public PsiMember getPsiElement() {
    return (PsiMember)myPsiMemberAnchor.retrieve();
  }

  @Override
  @Nullable
  public PsiAnnotation getAnnotation() {
    return myPsiAnnotation.getPsiElement();
  }

  @Nullable
  @Override
  public PsiElement getIdentifyingPsiElement() {
    return getPsiElement();
  }

  @NotNull
  @Override
  public Set<String> getExpressions() {
    Set<String> profiles = new HashSet<>();
    SpringAliasFor aliasFor = getAliasAttribute(VALUE_ATTR_NAME);
    if (aliasFor != null) {
      for (JamStringAttributeElement<String> element : JamAttributeMeta.collectionString(aliasFor.getMethodName())
        .getJam(myPsiAnnotation)) {
        String value = element.getStringValue();
        if (!StringUtil.isEmptyOrSpaces(value)) {
          profiles.add(value.trim());
        }
      }
    }
    else {
      SpringJamProfile definingProfile = myDefiningMetaAnnotation.getValue();
      if (definingProfile != null) return definingProfile.getExpressions();
    }
    return profiles;
  }

  @Nullable
  private SpringAliasFor getAliasAttribute(@NotNull String attrName) {
    PsiMember element = getPsiElement();
    return element == null ? null : SpringAliasForUtils.findAliasFor(element,
                                                                     myAnnotationChildLink.getAnnotationQualifiedName(),
                                                                     SpringAnnotationsConstants.PROFILE,
                                                                     attrName);
  }

  @NotNull
  @Override
  public List<JamStringAttributeElement<String>> getValueElements() {
    SpringAliasFor aliasFor = getAliasAttribute(VALUE_ATTR_NAME);
    return aliasFor == null ? Collections.emptyList() :
           JamAttributeMeta.collectionString(aliasFor.getMethodName()).getJam(myPsiAnnotation);
  }
}