/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.converters;

import com.intellij.jam.JamConverter;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.*;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.PackageReferenceSet;
import com.intellij.spring.model.utils.SpringReferenceUtils;
import com.intellij.util.ObjectUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.uast.UastContextKt;
import org.jetbrains.uast.expressions.UInjectionHost;

import java.util.Collection;

public class PackageJamConverter extends JamConverter<Collection<PsiPackage>> {

  @Override
  public Collection<PsiPackage> fromString(@Nullable String s, JamStringAttributeElement<Collection<PsiPackage>> context) {
    if (StringUtil.isEmptyOrSpaces(s)) return null;

    PsiLanguageInjectionHost psiLiteral = context.getLanguageInjectionHost();
    if (psiLiteral == null) {
      return calculatePackagesOnFakeElement(s, context.getPsiElement());
    }

    return createReferenceSet(psiLiteral, s).resolvePackage();
  }

  @Nullable
  private Collection<PsiPackage> calculatePackagesOnFakeElement(@NotNull String s,
                                                                @Nullable PsiElement context) {
    if (context == null) return null;
    final PsiExpression psiExpression =
      JavaPsiFacade.getElementFactory(context.getProject()).createExpressionFromText("\"" + s + "\"", context);
    if (psiExpression instanceof PsiLanguageInjectionHost) {
      return createReferenceSet((PsiLanguageInjectionHost)psiExpression, s).resolvePackage();
    }
    return null;
  }

  @Override
  public PsiReference @NotNull [] createReferences(@NotNull JamStringAttributeElement<Collection<PsiPackage>> context,
                                                   @NotNull PsiLanguageInjectionHost injectionHost) {
    final String value = context.getStringValue();
    if (value == null) {
      return PsiReference.EMPTY_ARRAY;
    }

    return createReferenceSet(injectionHost, value).getPsiReferences();
  }

  @NotNull
  private PackageReferenceSet createReferenceSet(@NotNull PsiLanguageInjectionHost psiLiteral, String value) {
    return new SpringAntPatternPackageReferenceSet(value, psiLiteral, ElementManipulators.getOffsetInElement(psiLiteral),
                                                   SpringReferenceUtils.getResolveScope(psiLiteral));
  }
}
