// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.

package com.intellij.spring.model.jam;

import com.intellij.jam.annotations.JamPsiConnector;
import com.intellij.jam.annotations.JamPsiValidity;
import com.intellij.openapi.module.Module;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiMethod;
import com.intellij.psi.PsiType;
import com.intellij.psi.util.PsiTypesUtil;
import com.intellij.spring.model.SpringQualifier;
import com.intellij.spring.model.jam.qualifiers.SpringJamQualifier;
import com.intellij.spring.model.utils.SpringCommonUtils;
import com.intellij.util.ObjectUtils;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.uast.UExpression;

import java.util.Collection;
import java.util.Collections;

public abstract class JamPsiMethodSpringBean extends JamPsiMemberSpringBean<PsiMethod> {

  @Override
  @NotNull
  @JamPsiConnector
  public abstract PsiMethod getPsiElement();

  @Override
  @JamPsiValidity
  public abstract boolean isValid();

  @Override
  @Nullable
  public String getBeanName() {
    return getPsiElement().getName();
  }

  @Override
  @Nullable
  public PsiType getBeanType() {
    PsiMethod element = getPsiElement();
    return element.isValid()? element.getReturnType() : null;
  }

  /**
   * Returns actual {@code new MyImpl} type taken from method implementation if {@code considerFactories = true}.
   */
  @Nullable
  @Override
  public PsiType getBeanType(boolean considerFactories) {
    if (!considerFactories) {
      return getBeanType();
    }

    PsiType extractedType = ObjectUtils.doIfNotNull(ContainerUtil.getOnlyItem(SpringCommonUtils.getReturnedUExpression(getPsiElement())),
                                                    UExpression::getExpressionType);
    if (extractedType != null) return extractedType;
    return getBeanType();
  }

  @Override
  public @NotNull Collection<SpringQualifier> getQualifiers() {
    Module module = getModule();
    if (module == null) return Collections.emptySet();

    Collection<SpringQualifier> qualifiers = SpringJamQualifier.findSpringJamQualifiers(module, getPsiElement());
    if (!qualifiers.isEmpty()) return qualifiers;

    PsiClass psiClass = PsiTypesUtil.getPsiClass(getBeanType());
    if (psiClass != null) {
      return SpringJamQualifier.findSpringJamQualifiers(module, psiClass);
    }
    return Collections.emptySet();
  }
}