// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.highlighting.jam;

import com.intellij.codeInspection.ProblemHighlightType;
import com.intellij.codeInspection.ProblemsHolder;
import com.intellij.jam.JamConverter;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.model.util.JamCommonUtil;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiAnnotationMemberValue;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElement;
import com.intellij.psi.util.InheritanceUtil;
import com.intellij.spring.SpringApiBundle;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.jam.converters.SpringBeanReferenceJamConverter;
import com.intellij.spring.model.utils.SpringCommonUtils;
import com.intellij.spring.model.utils.SpringModelUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.uast.UElement;
import org.jetbrains.uast.UastContextKt;
import org.jetbrains.uast.expressions.UInjectionHost;

public abstract class SpringBeanPointerResolveInspection extends SpringJavaInspectionBase {
  protected static boolean isPlainJavaFileInSpringModule(@NotNull PsiElement psiElement) {
    if (!JamCommonUtil.isPlainJavaFile(psiElement.getContainingFile())) {
      return false;
    }
    final Module module = ModuleUtilCore.findModuleForPsiElement(psiElement);
    return SpringCommonUtils.hasSpringFacet(module) || SpringModelUtils.getInstance().hasAutoConfiguredModels(module);
  }

  /**
   * Takes required bean type from associated {@link SpringBeanReferenceJamConverter}.
   *
   * @param holder  Holder.
   * @param element JAM.
   */
  public static void checkBeanPointerResolve(ProblemsHolder holder, JamStringAttributeElement<SpringBeanPointer<?>> element) {
    String beanType = null;
    final JamConverter<SpringBeanPointer<?>> converter = element.getConverter();
    if (converter instanceof SpringBeanReferenceJamConverter) {
      beanType = ((SpringBeanReferenceJamConverter)converter).getBaseClass();
    }
    checkBeanPointerResolve(holder, element, beanType);
  }

  public static void checkBeanPointerResolve(ProblemsHolder holder,
                                             JamStringAttributeElement<SpringBeanPointer<?>> element,
                                             @Nullable String beanType) {
    final String beanName = element.getStringValue();
    if (beanName != null) {
      final PsiAnnotationMemberValue memberValue = element.getPsiElement();
      if (memberValue != null) {
        final SpringBeanPointer<?> value = element.getValue();
        if (value == null) {
          PsiElement problemPsiElement = getProblemPsiElement(memberValue);
          if (problemPsiElement != null) {
            holder.registerProblem(problemPsiElement, SpringApiBundle.message("model.bean.error.message", beanName),
                                   ProblemHighlightType.GENERIC_ERROR_OR_WARNING);
          }
        }
        else {
          if (StringUtil.isNotEmpty(beanType)) {
            final PsiClass psiClass = value.getBeanClass();
            if (psiClass != null && !InheritanceUtil.isInheritor(psiClass, beanType)) {
              PsiElement problemPsiElement = getProblemPsiElement(memberValue);
              if (problemPsiElement != null) {
                holder.registerProblem(problemPsiElement,
                                       SpringApiBundle.message("bean.must.be.of.type", beanType),
                                       ProblemHighlightType.GENERIC_ERROR_OR_WARNING);
              }
            }
          }
        }
      }
    }
  }

  @Nullable
  private static PsiElement getProblemPsiElement(PsiElement element) {
    UElement uElement = UastContextKt.toUElement(element, UInjectionHost.class);
    if (uElement == null) return null;
    return uElement.getSourcePsi();
  }
}
