// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.highlighting.dom;

import com.intellij.codeInspection.util.InspectionMessage;
import com.intellij.ide.nls.NlsMessages;
import com.intellij.psi.PsiClass;
import com.intellij.psi.util.InheritanceUtil;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.SpringApiBundle;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringModelSearchParameters;
import com.intellij.spring.model.utils.SpringModelSearchers;
import com.intellij.spring.model.utils.SpringModelUtils;
import com.intellij.spring.model.xml.RequiredBeanType;
import com.intellij.util.Function;
import com.intellij.util.xml.DomElement;
import com.intellij.util.xml.DomJavaUtil;
import com.intellij.util.xml.DomUtil;
import com.intellij.util.xml.GenericAttributeValue;
import com.intellij.util.xml.highlighting.DefineAttributeQuickFix;
import com.intellij.util.xml.highlighting.DomElementAnnotationHolder;
import com.intellij.util.xml.highlighting.RemoveDomElementQuickFix;

import java.util.ArrayList;
import java.util.List;

/**
 * @author Yann C&eacute;bron
 */
public class SpringDomInspectionUtils {

  private static final Function<DomElement, String> DOM_NAME_FUNCTION = DomElement::getXmlElementName;

  private final DomElementAnnotationHolder holder;

  public SpringDomInspectionUtils(DomElementAnnotationHolder holder) {
    this.holder = holder;
  }

  public static boolean hasAny(DomElement... values) {
    for (DomElement value : values) {
      if (DomUtil.hasXml(value)) {
        return true;
      }
    }
    return false;
  }

  public static boolean hasMoreThanOne(DomElement... values) {
    short count = 0;
    for (DomElement value : values) {
      if (DomUtil.hasXml(value)) {
        count++;
        if (count > 1) {
          return true;
        }
      }
    }
    return false;
  }

  public void oneOfRequired(DomElement element,
                            DomElement... elements) {
    if (hasMoreThanOne(elements) || !hasAny(elements)) {
      onlyOneOfProblem(element, elements);
    }
  }

  public void oneOfOrAllRequired(DomElement element,
                                 GenericAttributeValue... values) {
    if (!hasAny(values)) {
      holder.createProblem(element, getDomElementNamesMessage("spring.dom.one.or.all.of.attributes", values));
    }
  }

  public boolean onlyOneOf(DomElement element,
                           GenericAttributeValue... values) {
    if (hasMoreThanOne(values)) {
      onlyOneOfProblem(element, values);
      return true;
    }
    return false;
  }

  public void ifExistsOtherRequired(DomElement element,
                                    GenericAttributeValue exists,
                                    GenericAttributeValue required) {
    if (hasAny(exists) &&
        !hasAny(required)) {

      final String requiredName = DOM_NAME_FUNCTION.fun(required);
      holder.createProblem(element,
                           SpringApiBundle.message("spring.dom.if.exists.other.required",
                                                   DOM_NAME_FUNCTION.fun(exists),
                                                   requiredName),
                           new DefineAttributeQuickFix(requiredName));
    }
  }

  public void attributeSuperfluous(GenericAttributeValue value) {
    if (hasAny(value)) {
      holder.createProblem(value,
                           SpringApiBundle.message("spring.dom.superfluous.attribute",
                                                   DOM_NAME_FUNCTION.fun(value)),
                           new RemoveDomElementQuickFix(value)).highlightWholeElement();
    }
  }

  public <T> void attributeWithDefaultSuperfluous(GenericAttributeValue<T> value,
                                                  T defaultValue) {
    if (hasAny(value) &&
        defaultValue.equals(value.getValue())) {
      holder.createProblem(value,
                           SpringApiBundle.message("spring.dom.superfluous.attribute.with.default",
                                                   DOM_NAME_FUNCTION.fun(value),
                                                   value.getStringValue()),
                           new RemoveDomElementQuickFix(value)).highlightWholeElement();
    }
  }

  private void onlyOneOfProblem(DomElement element, DomElement... elements) {
    holder.createProblem(element, getDomElementNamesMessage("spring.dom.only.one.of", elements));
  }

  private static @InspectionMessage String getDomElementNamesMessage(String key,
                                                                     DomElement... elements) {
    List<String> elementNames = new ArrayList<>();
    for (DomElement element : elements) {
      String elementName = DOM_NAME_FUNCTION.fun(element);
      String name = element instanceof GenericAttributeValue ? "'" + elementName + "'" : "<" + elementName + ">";
      elementNames.add(name);
    }

    return SpringApiBundle.message(key, NlsMessages.formatOrList(elementNames));
  }

  public void beanOfType(GenericAttributeValue<SpringBeanPointer<?>> springBeanAttribute, String beanClass) {
    SpringBeanPointer<?>  pointer = springBeanAttribute.getValue();
    if (pointer == null) {
      return;
    }

    if (!InheritanceUtil.isInheritor(pointer.getBeanClass(), false, beanClass)) {
      holder.createProblem(springBeanAttribute,
                           SpringApiBundle.message("bean.must.be.of.type", beanClass));
    }
  }

  public void explicitBeanRequired(DomElement domElement,
                                   GenericAttributeValue<SpringBeanPointer<?>> springBeanAttribute,
                                   String defaultBeanName) {
    if (hasAny(springBeanAttribute)) {
      return;
    }

    final RequiredBeanType annotation = springBeanAttribute.getAnnotation(RequiredBeanType.class);
    assert annotation != null : springBeanAttribute;
    final String[] classNames = annotation.value();
    for (String beanClass : classNames) {
      if (existsBean(domElement, beanClass, defaultBeanName)) {
        return;
      }
    }

    holder.createProblem(domElement,
                         SpringApiBundle.message("spring.dom.explicit.bean.reference.required",
                                                 springBeanAttribute.getPresentation().getTypeName(),
                                                 defaultBeanName),
                         new DefineAttributeQuickFix(DOM_NAME_FUNCTION.fun(springBeanAttribute)));
  }

  public static boolean existsBean(DomElement domElement, String beanClass, final String defaultBeanName) {
    final PsiClass psiClass = DomJavaUtil.findClass(beanClass, domElement);
    if (psiClass == null) {
      return false;
    }

    final SpringModelSearchParameters.BeanClass searchParameters =
      SpringModelSearchParameters.byClass(psiClass).withInheritors().effectiveBeanTypes();
    if (!searchParameters.canSearch()) return false;

    final CommonSpringModel model = SpringModelUtils.getInstance().getSpringModel(domElement.getXmlTag());

    final SpringBeanPointer<?>  beanWithDefaultName = SpringModelSearchers.findBean(model, defaultBeanName);
    if (beanWithDefaultName == null) {
      return false;
    }

    return !model.processByClass(searchParameters, pointer -> !defaultBeanName.equals(pointer.getName()));
  }
}
