// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.converters;

import com.intellij.codeInspection.LocalQuickFix;
import com.intellij.lang.jvm.JvmModifier;
import com.intellij.psi.CommonClassNames;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiMethod;
import com.intellij.psi.PsiModifier;
import com.intellij.psi.scope.processor.MethodResolveProcessor;
import com.intellij.util.xml.ConvertContext;
import com.intellij.util.xml.GenericDomValue;

import static com.intellij.codeInsight.daemon.impl.quickfix.CreateMethodFixKt.createVoidMethodFixes;

public abstract class SpringBeanMethodConverter extends PsiMethodConverter {

  @Override
  protected MethodAccepter getMethodAccepter(final ConvertContext context, final boolean forCompletion) {
    return new MethodAccepter() {

      @Override
      public boolean accept(PsiMethod method) {
        if (method.isConstructor()) return false;

        final PsiClass containingClass = method.getContainingClass();
        if (containingClass == null) return false;
        final String containing = containingClass.getQualifiedName();
        if (CommonClassNames.JAVA_LANG_OBJECT.equals(containing)) return false;

        return checkParameterList(method) && checkModifiers(method) && checkReturnType(context, method, forCompletion);
      }
    };
  }

  protected boolean checkModifiers(final PsiMethod method) {
    return method.hasModifierProperty(PsiModifier.PUBLIC) && !method.hasModifierProperty(PsiModifier.ABSTRACT);
  }

  protected boolean checkParameterList(final PsiMethod method) {
    return method.getParameterList().getParametersCount() == 0;
  }

  protected boolean checkReturnType(final ConvertContext context, final PsiMethod method, final boolean forCompletion) {
    return true;
  }

  @Override
  public LocalQuickFix[] getQuickFixes(final ConvertContext context) {
    final GenericDomValue element = (GenericDomValue)context.getInvocationElement();

    final String elementName = element.getStringValue();
    final PsiClass beanClass = getPsiClass(context);
    if (elementName != null && elementName.length() > 0 && beanClass != null) {
      return createVoidMethodFixes(beanClass, elementName, JvmModifier.PRIVATE);
    }

    return LocalQuickFix.EMPTY_ARRAY;
  }

  @Override
  protected String getMethodIdentificator(PsiMethod method) {
    return method.getName();
  }

  @Override
  protected PsiMethod[] getMethodCandidates(String methodIdentificator, PsiClass psiClass) {
    return MethodResolveProcessor.findMethod(psiClass, methodIdentificator);
  }
}
