// Copyright 2000-2021 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.aliasFor;

import com.intellij.codeInsight.AnnotationUtil;
import com.intellij.jam.JamConverter;
import com.intellij.jam.JamService;
import com.intellij.jam.reflect.*;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.Pair;
import com.intellij.psi.JavaPsiFacade;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElement;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.util.CachedValueProvider;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.semantic.SemKey;
import com.intellij.spring.model.jam.SpringSemContributorUtil;
import com.intellij.spring.model.jam.utils.JamAnnotationTypeUtil;
import com.intellij.util.Consumer;
import com.intellij.util.NotNullFunction;
import com.intellij.util.SmartList;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.List;

import static com.intellij.codeInsight.AnnotationUtil.CHECK_HIERARCHY;
import static com.intellij.jam.JamService.CHECK_METHOD;

public final class SpringAliasForUtils {

  @Nullable
  public static SpringAliasFor findAliasFor(@Nullable PsiElement context,
                                            @Nullable String toSearchInAnnotation,
                                            @NotNull String aliasedClassName,
                                            @NotNull String attrName) {
    if (context == null) return null;
    return getAliasFor(context.getProject(), context.getResolveScope(), toSearchInAnnotation, aliasedClassName, attrName);
  }

  @Nullable
  public static SpringAliasFor findAliasFor(@NotNull Project project,
                                            @Nullable String toSearchInAnnotation,
                                            @NotNull String aliasedClassName,
                                            @NotNull String attrName) {
    return getAliasFor(project, GlobalSearchScope.allScope(project), toSearchInAnnotation, aliasedClassName, attrName);
  }

  private static SpringAliasFor getAliasFor(@NotNull Project project,
                                            @NotNull GlobalSearchScope scope,
                                            @Nullable String toSearchInAnnotation,
                                            @NotNull String aliasedClassName, @NotNull String attrName) {
    if (toSearchInAnnotation == null) return null;

    final JavaPsiFacade javaPsiFacade = JavaPsiFacade.getInstance(project);
    PsiClass toSearchInClass = javaPsiFacade.findClass(toSearchInAnnotation, scope);
    if (toSearchInClass == null || !toSearchInClass.isAnnotationType()) {
      return null;
    }

    PsiClass annotationClass = javaPsiFacade.findClass(aliasedClassName, scope);
    if (annotationClass == null || !annotationClass.isAnnotationType()) {
      return null;
    }

    for (SpringAliasFor aliasFor : getAliasForAttributes(toSearchInClass)) {
      if (attrName.equals(aliasFor.getAttributeName()) && annotationClass.equals(aliasFor.getAnnotationClass())) {
        return aliasFor;
      }
    }
    return null;
  }

  @NotNull
  private static List<SpringAliasFor> getAliasForAttributes(@NotNull PsiClass psiClass) {
    return CachedValuesManager.getCachedValue(psiClass, () -> {
      final List<SpringAliasFor> aliasForList = JamService.getJamService(psiClass.getProject())
        .getAnnotatedMembersList(psiClass, SpringAliasFor.SEM_KEY, CHECK_METHOD);
      return CachedValueProvider.Result.create(aliasForList, psiClass);
    });
  }

  @Nullable
  public static PsiAnnotation findDefiningMetaAnnotation(@Nullable PsiElement context,
                                                         @NotNull String customAnnotationFqn,
                                                         @NotNull String baseMetaAnnotationFqn) {
    return findDefiningMetaAnnotation(context, customAnnotationFqn, baseMetaAnnotationFqn, false);
  }

  @Nullable
  public static PsiAnnotation findDefiningMetaAnnotation(@Nullable PsiElement context,
                                                         @NotNull String customAnnotationFqn,
                                                         @NotNull String baseMetaAnnotationFqn,
                                                         boolean includingTests) {
    if (context == null) return null;
    final Module module = ModuleUtilCore.findModuleForPsiElement(context);
    if (module == null) return null;

    PsiClass customAnnoClass = JavaPsiFacade.getInstance(module.getProject())
      .findClass(customAnnotationFqn, context.getResolveScope());

    final Collection<PsiClass> allMetaAnnotations = includingTests ?
                                                    JamAnnotationTypeUtil.getInstance(module)
                                                      .getAnnotationTypesWithChildrenIncludingTests(baseMetaAnnotationFqn) :
                                                    JamAnnotationTypeUtil.getInstance(module)
                                                      .getAnnotationTypesWithChildren(baseMetaAnnotationFqn);

    return findDefiningMetaAnnotation(customAnnoClass, baseMetaAnnotationFqn, allMetaAnnotations);
  }

  @Nullable
  public static PsiAnnotation findDefiningMetaAnnotation(@Nullable PsiClass customAnnoClass,
                                                         @NotNull String baseMetaAnnotationFqn,
                                                         @NotNull final Collection<? extends PsiClass> allMetaAnnotations) {
    if (customAnnoClass == null || !customAnnoClass.isAnnotationType()) {
      return null;
    }

    final PsiAnnotation psiAnnotation = AnnotationUtil.findAnnotation(customAnnoClass, true, baseMetaAnnotationFqn);
    if (psiAnnotation != null) {
      return psiAnnotation;
    }

    for (PsiClass customMetaAnnoClass : allMetaAnnotations) {
      final String qualifiedName = customMetaAnnoClass.getQualifiedName();
      if (qualifiedName != null && AnnotationUtil.isAnnotated(customAnnoClass, qualifiedName, CHECK_HIERARCHY)) {
        return findDefiningMetaAnnotation(customMetaAnnoClass, baseMetaAnnotationFqn, allMetaAnnotations);
      }
    }
    return null;
  }

  /**
   * Builds JamAttributeMeta supporting converters for parent's meta String attributes.
   *
   * @param annoMetaKey SemKey.
   * @param parentMetas Parent metas to search for attributes.
   * @return Function.
   * @see SpringSemContributorUtil#createFunction(SemKey, Class, com.intellij.util.Function, com.intellij.util.Function, Consumer, NotNullFunction)
   */
  public static NotNullFunction<Pair<String, Project>, JamAnnotationMeta> getAnnotationMetaProducer(@NotNull SemKey<JamAnnotationMeta> annoMetaKey,
                                                                                                    JamMemberMeta<?, ?> @NotNull ... parentMetas) {
    return anno -> new JamAnnotationMeta(anno.first, null, annoMetaKey) {

      @Nullable
      @Override
      public JamAttributeMeta<?> findAttribute(@Nullable @NonNls String name) {
        JamAttributeMeta<?> attribute = super.findAttribute(name);
        if (attribute != null) return attribute;

        if (name == null) name = "value";
        return getAliasedAttributeMeta(name);
      }

      @Nullable
      private JamAttributeMeta<?> getAliasedAttributeMeta(@NotNull String name) {
        for (JamMemberMeta<?, ?> parentMeta : parentMetas) {
          for (JamAnnotationMeta annotationMeta : parentMeta.getAnnotations()) {
            List<JamAttributeMeta<?>> parentAnnotationAttributes = getRegisteredAttributes(annotationMeta);
            for (JamAttributeMeta<?> attributeMeta : parentAnnotationAttributes) {
              if (attributeMeta instanceof JamStringAttributeMeta) {
                final JamStringAttributeMeta<?, ?> meta = (JamStringAttributeMeta)attributeMeta;
                final JamConverter<?> converter = meta.getConverter();

                SpringAliasFor aliasFor = findAliasFor(anno.second,
                                                       anno.first,
                                                       annotationMeta.getAnnoName(),
                                                       meta.getAttributeLink().getAttributeName());
                if (aliasFor != null) {
                  String aliasForMethodName = aliasFor.getMethodName();
                  if (name.equals(aliasForMethodName)) {
                    if (attributeMeta instanceof JamStringAttributeMeta.Single) {
                      return JamAttributeMeta.singleString(aliasForMethodName, converter);
                    }
                    if (attributeMeta instanceof JamStringAttributeMeta.Collection) {
                      return JamAttributeMeta.collectionString(aliasForMethodName, converter);
                    }
                  }
                }
              }
            }
          }
        }
        return null;
      }

      private List<JamAttributeMeta<?>> getRegisteredAttributes(@NotNull JamAnnotationMeta annotationMeta) {
        List<JamAttributeMeta<?>> attributeMetas = new SmartList<>();
        attributeMetas.addAll(annotationMeta.getAttributes());
        JamAnnotationArchetype archetype = annotationMeta.getArchetype();
        if (archetype != null) {
          attributeMetas.addAll(archetype.getAttributes());
        }
        return attributeMetas;
      }
    };
  }
}
