// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model;

import com.intellij.jam.model.common.CommonModelElement;
import com.intellij.pom.references.PomService;
import com.intellij.psi.*;
import com.intellij.util.ArrayUtilRt;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;

/**
 * Used for beans created implicitly by other bean
 */
public class SpringImplicitBeanWithDefinition extends CommonModelElement.PsiBase
  implements CommonSpringBean {

  @NotNull
  private final String myName;
  @NotNull
  private final PsiClass myClass;
  @NotNull
  private final CommonSpringBean myDefiningBean;
  @NotNull
  private final PsiTarget myDefinitionTarget;

  /**
   * @param beanName
   * @param beanClass
   * @param definingBean     bean that implicitly creates this bean
   * @param definitionTarget target to psi element that is definition of this bean
   */
  public SpringImplicitBeanWithDefinition(@NotNull String beanName,
                                          @NotNull PsiClass beanClass,
                                          @NotNull CommonSpringBean definingBean,
                                          @NotNull PsiTarget definitionTarget) {
    myName = beanName;
    myClass = beanClass;
    myDefiningBean = definingBean;
    myDefinitionTarget = definitionTarget;
  }

  @Nullable
  @Override
  public String getBeanName() {
    return myName;
  }

  @Override
  public String @NotNull [] getAliases() {
    return ArrayUtilRt.EMPTY_STRING_ARRAY;
  }

  @Nullable
  @Override
  public PsiType getBeanType(boolean considerFactories) {
    return getBeanType();
  }

  @Nullable
  @Override
  public PsiType getBeanType() {
    return JavaPsiFacade.getElementFactory(myClass.getProject()).createType(myClass);
  }

  @NotNull
  @Override
  public Collection<SpringQualifier> getSpringQualifiers() {
    return Collections.singleton(DefaultSpringBeanQualifier.create(this));
  }

  @NotNull
  @Override
  public SpringProfile getProfile() {
    return myDefiningBean.getProfile();
  }

  @Override
  public boolean isPrimary() {
    return false;
  }

  @NotNull
  @Override
  public PsiElement getPsiElement() {
    return myDefinitionTarget.getNavigationElement();
  }

  @Override
  public PsiElement getIdentifyingPsiElement() {
    return PomService.convertToPsi(myDefiningBean.getPsiManager().getProject(), myDefinitionTarget);
  }
}
