// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model;

import com.intellij.ide.presentation.Presentation;
import com.intellij.openapi.util.Comparing;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiFile;
import com.intellij.psi.PsiType;
import com.intellij.spring.SpringPresentationProvider;
import com.intellij.util.Function;
import com.intellij.xml.util.PsiElementPointer;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Comparator;

/**
 * Lightweight reference to Spring bean.
 *
 * @author peter
 * @see BeanService#createSpringBeanPointer(CommonSpringBean)
 */
@Presentation(provider = SpringPresentationProvider.class)
public interface SpringBeanPointer<T extends CommonSpringBean> extends PsiElementPointer {

  Function<SpringBeanPointer<?>, CommonSpringBean> TO_BEAN = SpringBeanPointer::getSpringBean;

  /**
   * Sort beans for UI purposes.
   */
  Comparator<SpringBeanPointer<?>> DISPLAY_COMPARATOR = (o1, o2) -> Comparing.compare(SpringPresentationProvider.getSpringBeanName(o1),
                                                                                   SpringPresentationProvider.getSpringBeanName(o2));

  /**
   * Defined bean name.
   * <p/>
   * For presentation purposes use {@link SpringPresentationProvider#getSpringBeanName(SpringBeanPointer)}.
   *
   * @return {@code null} if no name could be determined.
   */
  @Nullable
  String getName();

  String[] getAliases();

  @NotNull
  T getSpringBean();

  boolean isValid();

  boolean isReferenceTo(@Nullable CommonSpringBean springBean);

  SpringBeanPointer<?>  derive(@NotNull String name);

  /**
   * @return {@code this} or underlying pointer (e.g. for derived pointers).
   */
  @NotNull
  SpringBeanPointer<?>  getBasePointer();

  @Nullable
  PsiClass getBeanClass();

  PsiType[] getEffectiveBeanTypes();

  @NotNull
  PsiFile getContainingFile();

  /**
   * Only for {@link com.intellij.spring.model.xml.beans.SpringBean}.
   *
   * @return {@code true} if bean is abstract.
   */
  boolean isAbstract();

  /**
   * Only for {@link com.intellij.spring.model.xml.beans.SpringBean}.
   *
   * @return Pointer to parent bean if defined.
   */
  @Nullable
  SpringBeanPointer<?>  getParentPointer();
}