// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model;

import com.intellij.openapi.util.Condition;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiFile;
import com.intellij.psi.PsiType;
import com.intellij.spring.model.jam.JamPsiMemberSpringBean;
import com.intellij.spring.model.jam.JamSpringBeanPointer;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public final class DelegateConditionalSpringBeanPointer implements JamSpringBeanPointer, ConditionalEvaluator {

  private final JamSpringBeanPointer myDelegate;
  private final Condition<ConditionalEvaluationContext> myCondition;

  public static @NotNull DelegateConditionalSpringBeanPointer createPointer(@NotNull JamSpringBeanPointer delegate,
                                                                            @NotNull Condition<ConditionalEvaluationContext> condition) {
    return new DelegateConditionalSpringBeanPointer(delegate, condition);
  }

  private DelegateConditionalSpringBeanPointer(@NotNull JamSpringBeanPointer delegate,
                                               @NotNull Condition<ConditionalEvaluationContext> condition) {
    myDelegate = delegate;
    myCondition = condition;
  }

  @Override
  public @Nullable PsiElement getPsiElement() {
    return myDelegate.getPsiElement();
  }

  @Override
  public @Nullable String getName() {
    return myDelegate.getName();
  }

  @Override
  public String[] getAliases() {
    return myDelegate.getAliases();
  }

  @Override
  public @NotNull JamPsiMemberSpringBean<?> getSpringBean() {
    return myDelegate.getSpringBean();
  }

  @Override
  public boolean isValid() {
    return myDelegate.isValid();
  }

  @Override
  public boolean isReferenceTo(@Nullable CommonSpringBean springBean) {
    return myDelegate.isReferenceTo(springBean);
  }

  @Override
  public SpringBeanPointer<?> derive(@NotNull String name) {
    return myDelegate.derive(name);
  }

  @Override
  public @NotNull SpringBeanPointer<?> getBasePointer() {
    return myDelegate.getBasePointer();
  }

  @Override
  public @Nullable PsiClass getBeanClass() {
    return myDelegate.getBeanClass();
  }

  @Override
  public PsiType[] getEffectiveBeanTypes() {
    return myDelegate.getEffectiveBeanTypes();
  }

  @Override
  public @NotNull PsiFile getContainingFile() {
    return myDelegate.getContainingFile();
  }

  @Override
  public boolean isAbstract() {
    return myDelegate.isAbstract();
  }

  @Override
  public @Nullable SpringBeanPointer<?> getParentPointer() {
    return myDelegate.getParentPointer();
  }

  @Override
  public boolean isActive(ConditionalEvaluationContext context) {
    return myDelegate.isValid() && myCondition.value(context);
  }

  @Override
  public int hashCode() {
    return myDelegate.hashCode();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (!(o instanceof DelegateConditionalSpringBeanPointer)) return false;

    return myDelegate.equals(((DelegateConditionalSpringBeanPointer)o).myDelegate);
  }
}
