// Copyright 2000-2021 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.gutter;

import com.intellij.ide.util.DefaultPsiElementCellRenderer;
import com.intellij.ide.util.PsiElementListCellRenderer;
import com.intellij.jam.JamService;
import com.intellij.openapi.fileEditor.UniqueVFilePathBuilder;
import com.intellij.openapi.project.DumbService;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.pom.PomTarget;
import com.intellij.pom.PomTargetPsiElement;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiMember;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.psi.xml.XmlTag;
import com.intellij.spring.SpringApiBundle;
import com.intellij.spring.SpringApiIcons;
import com.intellij.spring.SpringPresentationProvider;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPsiTarget;
import com.intellij.spring.model.jam.JamPsiMemberSpringBean;
import com.intellij.spring.model.pom.SpringBeanPomTargetUtils;
import com.intellij.spring.model.xml.DomSpringBean;
import com.intellij.util.xml.DomElement;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;

public class SpringBeansPsiElementCellRenderer extends DefaultPsiElementCellRenderer {

  public static final PsiElementListCellRenderer<PsiElement> INSTANCE = new SpringBeansPsiElementCellRenderer();

  private static final DomElementListCellRenderer DOM_RENDERER =
    new DomElementListCellRenderer(SpringApiBundle.message("spring.bean.with.unknown.name")) {

      @Override
      public String getElementText(PsiElement element) {
        final DomElement domElement = getDomElement(element);
        if (domElement instanceof DomSpringBean) {
          return SpringPresentationProvider.getSpringBeanName(((DomSpringBean)domElement));
        }
        return super.getElementText(element);
      }

      @Override
      protected Icon getIcon(PsiElement element) {
        return SpringPresentationProvider.getSpringIcon(getDomElement(element));
      }
    };

  @Override
  public String getElementText(final PsiElement element) {
    PsiElement psiElement = getElementToProcess(element);
    if (psiElement instanceof XmlTag) {
      return DOM_RENDERER.getElementText(psiElement);
    }

    @Nullable final CommonSpringBean springBean;
    if (psiElement instanceof PsiAnnotation) {
      final PsiMember member = PsiTreeUtil.getParentOfType(psiElement, PsiMember.class);
      springBean = member == null ? null : getJamBean(member);
    }
    else {
      springBean = SpringBeanPomTargetUtils.getSpringBean(element);
    }

    if (springBean != null) {
      return SpringPresentationProvider.getSpringBeanName(springBean);
    }

    return super.getElementText(psiElement);
  }

  private static PsiElement getElementToProcess(PsiElement element) {
    if (element instanceof PomTargetPsiElement) {
      PomTarget target = ((PomTargetPsiElement)element).getTarget();
      if (target instanceof SpringBeanPsiTarget) {
        return ((SpringBeanPsiTarget)target).getNavigationElement();
      }
    }
    return element;
  }

  @Override
  public String getContainerText(final PsiElement element, final String name) {
    PsiElement psiElement = getElementToProcess(element);
    if (psiElement instanceof XmlTag) {
      return getUniqueVirtualFilePath(psiElement);
    }

    if (psiElement instanceof PsiAnnotation) {
      final PsiClass psiClass = PsiTreeUtil.getParentOfType(psiElement, PsiClass.class);
      if (psiClass != null && psiClass.getName() != null) {
        return getUniqueVirtualFilePath(psiClass);
      }
    }

    return getUniqueVirtualFilePath(element);
  }

  private static String getUniqueVirtualFilePath(PsiElement psiElement) {
    final VirtualFile virtualFile = psiElement.getContainingFile().getVirtualFile();
    return "(" + UniqueVFilePathBuilder.getInstance().getUniqueVirtualFilePath(psiElement.getProject(), virtualFile) + ")";
  }

  @Nullable
  @Override
  protected Icon getIcon(final PsiElement element) {
    PsiElement psiElement = getElementToProcess(element);
    if (psiElement instanceof XmlTag) {
      return DOM_RENDERER.getIcon(psiElement);
    }
    else if (psiElement instanceof PsiAnnotation) {
      final PsiMember member = PsiTreeUtil.getParentOfType(psiElement, PsiMember.class);
      final CommonSpringBean springBean = member == null ? null : getJamBean(member);
      if (springBean != null) {
        return SpringApiIcons.SpringJavaBean;
      }
    }
    else {
      if (!DumbService.isDumb(element.getProject())) {
        final CommonSpringBean bean = SpringBeanPomTargetUtils.getSpringBean(element);
        if (bean != null) {
          return SpringPresentationProvider.getSpringIcon(bean);
        }
      }
    }
    return super.getIcon(psiElement);
  }

  private static JamPsiMemberSpringBean getJamBean(PsiMember member) {
    return JamService.getJamService(member.getProject())
      .getJamElement(JamPsiMemberSpringBean.PSI_MEMBER_SPRING_BEAN_JAM_KEY, member);
  }
}
