// Copyright 2000-2021 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.gutter;

import com.intellij.ide.util.PsiElementListCellRenderer;
import com.intellij.pom.PomTargetPsiElement;
import com.intellij.psi.PsiElement;
import com.intellij.psi.xml.XmlTag;
import com.intellij.util.xml.DomElement;
import com.intellij.util.xml.DomManager;
import org.jetbrains.annotations.Nls;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;

/**
 * @author Dmitry Avdeev
 */
public class DomElementListCellRenderer extends PsiElementListCellRenderer<PsiElement> {

  protected final String myUnknown;

  public DomElementListCellRenderer(@Nls String unknownElementText) {
    myUnknown = unknownElementText;
  }

  @Override
  public String getElementText(final PsiElement element) {
    String elementName = null;

    final DomElement domElement = getDomElement(element);
    if (domElement != null) {
      elementName = domElement.getPresentation().getElementName();
    }
    else if (element instanceof XmlTag) {
      return ((XmlTag)element).getName();
    }

    return elementName == null ? myUnknown : elementName;
  }

  @Nullable
  public static DomElement getDomElement(@Nullable PsiElement element) {
    if (element instanceof PomTargetPsiElement) {
      return getDomElement(element.getNavigationElement());
    }
    else if (element instanceof XmlTag) {
      return DomManager.getDomManager(element.getProject()).getDomElement((XmlTag)element);
    }
    return null;
  }

  @Override
  protected String getContainerText(final PsiElement element, final String name) {
    return getContainerText(element);
  }

  public static String getContainerText(final PsiElement element) {
    return " (" + element.getContainingFile().getName() + ")";
  }

  @Override
  protected Icon getIcon(final PsiElement element) {
    final DomElement domElement = getDomElement(element);

    if (domElement != null) {
      final Icon presentationIcon = domElement.getPresentation().getIcon();
      if (presentationIcon != null) {
        return presentationIcon;
      }
    }

    return super.getIcon(element);
  }
}