// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.facet;

import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.module.Module;
import com.intellij.psi.PsiFile;
import com.intellij.util.messages.Topic;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.Set;

/**
 * @author Yann C&eacute;bron
 */
public abstract class SpringFileSetService {
  public static final Topic<SpringFileSetListener> TOPIC = new Topic<>("Spring file set topic", SpringFileSetListener.class);

  public static SpringFileSetService getInstance() {
    return ApplicationManager.getApplication().getService(SpringFileSetService.class);
  }

  public abstract String getUniqueId(final Set<SpringFileSet> existing);

  public abstract String getUniqueName(String prefix, final Set<SpringFileSet> existing);

  /**
   * Returns all configured and provided file sets.
   *
   * @param facet Facet instance.
   * @return all working file sets for the facet.
   * @see SpringFacet#getFileSets()
   * @see #getModelProviderSets(SpringFacet)
   */
  @NotNull
  public abstract Set<SpringFileSet> getAllSets(final @NotNull SpringFacet facet);

  /**
   * Returns filesets provided by all {@link com.intellij.spring.SpringModelProvider}s.
   *
   * @param facet Facet instance.
   * @return filesets provided by {@link com.intellij.spring.SpringModelProvider}.
   * @see com.intellij.spring.SpringModelProvider
   */
  @NotNull
  public abstract List<SpringFileSet> getModelProviderSets(final @NotNull SpringFacet facet);

  /**
   * @param module  Module to search usage in.
   * @param psiFile Configuration file.
   * @return {@code null} if given file not configured.
   */
  @Nullable
  public abstract SpringFileSet findFileSet(Module module, PsiFile psiFile);

  // cross-fileset/module dependency management (internal)

  abstract String getQualifiedName(@NotNull SpringFileSet fileset);

  abstract String getDependencyIdFor(SpringFileSet current, SpringFileSet otherFileSet);

  @Nullable
  abstract SpringFileSet findDependencyFileSet(SpringFileSet current, @NotNull final String dependencyId);

  public interface SpringFileSetListener {
    void activeProfilesChanged();
  }
}
