// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.dom;

import com.intellij.openapi.util.Condition;
import com.intellij.psi.PsiFile;
import com.intellij.psi.PsiFileSystemItem;
import com.intellij.psi.xml.XmlFile;
import com.intellij.spring.model.xml.beans.Beans;
import com.intellij.util.xml.DomElement;
import com.intellij.util.xml.DomElementsNavigationManager;
import com.intellij.util.xml.DomFileElement;
import com.intellij.util.xml.DomManager;
import com.intellij.util.xml.stubs.index.DomElementClassIndex;
import com.intellij.util.xml.stubs.index.DomNamespaceKeyIndex;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * @author Yann C&eacute;bron
 */
public final class SpringDomUtils {

  public static final Condition<PsiFileSystemItem> SPRING_XML_CONDITION = item -> {
    if (item.isDirectory()) return false;

    final PsiFile file = item.getContainingFile();
    return file instanceof XmlFile &&
           isSpringXml((XmlFile)file);
  };

  public static void navigate(@Nullable DomElement domElement) {
    if (domElement == null) return;

    final DomElementsNavigationManager navigationManager = DomElementsNavigationManager.getManager(domElement.getManager().getProject());
    navigationManager.getDomElementsNavigateProvider(DomElementsNavigationManager.DEFAULT_PROVIDER_NAME).navigate(domElement, true);
  }

  public static boolean isSpringXml(@NotNull XmlFile configFile) {
    return getSpringDomFileElement(configFile) != null;
  }

  @Nullable
  public static DomFileElement<Beans> getSpringDomFileElement(@NotNull XmlFile configFile) {
    final DomFileElement<Beans> fileElement = DomManager.getDomManager(configFile.getProject()).getFileElement(configFile, Beans.class);
    return fileElement != null && fileElement.isValid() ? fileElement : null;
  }

  public static boolean hasNamespace(DomFileElement domFileElement,
                                     String namespaceKey) {
    return DomNamespaceKeyIndex.getInstance().hasStubElementsWithNamespaceKey(domFileElement, namespaceKey);
  }

  /**
   * @see com.intellij.util.xml.StubbedOccurrence
   */
  public static boolean hasElement(DomFileElement domFileElement,
                                   Class<? extends DomElement> domElementClazz) {
    return DomElementClassIndex.getInstance().hasStubElementsOfType(domFileElement, domElementClazz);
  }
}
