// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.contexts.model.visitors;

import com.intellij.openapi.progress.ProgressManager;
import com.intellij.openapi.util.RecursionManager;
import com.intellij.spring.CommonSpringModel;
import org.jetbrains.annotations.NotNull;

import java.util.Set;


public final class SpringModelVisitors {

  /**
   * Spring models are:
   * - connected (imports/component-scans/implicit fileset configurations/auto-configurations/etc.)
   * - contains other models {@link com.intellij.spring.contexts.model.SpringModel}
   * It visits all related spring models(xml,java,imported,scanned, children models, etc.).
   *
   * @return false to stop processing
   */
  public static <T> boolean visitRelatedModels(@NotNull CommonSpringModel parentModel,
                                               @NotNull CommonSpringModelVisitorContext<T> visitorContext) {
    return visitRelated(parentModel, visitorContext, true);
  }

  public static <T> boolean visitRelatedModels(@NotNull CommonSpringModel parentModel,
                                               @NotNull CommonSpringModelVisitorContext<T> visitorContext,
                                               boolean visitParentModel) {
    return visitRelated(parentModel, visitorContext, visitParentModel);
  }

  public static <T> boolean visitRecursionAwareRelatedModels(@NotNull CommonSpringModel parentModel,
                                                             @NotNull CommonSpringModelVisitorContext<T> visitorContext) {
    return visitRecursionAwareRelatedModels(parentModel, visitorContext, true);
  }

  public static <T> boolean visitRecursionAwareRelatedModels(@NotNull CommonSpringModel parentModel,
                                                             @NotNull CommonSpringModelVisitorContext<T> visitorContext,
                                                             boolean visitParentModel) {
    final Boolean aBoolean = RecursionManager.doPreventingRecursion(parentModel, false, () -> visitRelated(parentModel,
                                                                                                           visitorContext,
                                                                                                           visitParentModel));
    return aBoolean == null ? true : aBoolean;
  }

  private static <T> boolean visitRelated(@NotNull CommonSpringModel parentModel,
                                          @NotNull CommonSpringModelVisitorContext<T> visitorContext, boolean visitParent) {
    if (visitorContext.hasBeenVisited(parentModel)) return true;
    if (visitParent && !visitorContext.visit(parentModel)) return false;
    if (!visitRelated(visitorContext, parentModel.getRelatedModels())) return false;
    return true;
  }

  private static <T> boolean visitRelated(@NotNull CommonSpringModelVisitorContext<T> visitorContext,
                                          @NotNull Set<CommonSpringModel> relatedModels) {
    for (CommonSpringModel model : relatedModels) {
      ProgressManager.checkCanceled();
      if (!visitRelated(model, visitorContext, true)) return false;
    }
    return true;
  }
}
