// Copyright 2000-2021 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.contexts.model.visitors;

import com.intellij.spring.CommonSpringModel;
import com.intellij.util.Processor;
import org.jetbrains.annotations.NotNull;

import java.util.HashSet;
import java.util.Set;

public final class CommonSpringModelVisitorContext<P> {
  private final Set<CommonSpringModel> visited = new HashSet<>();

  private final Exec<P> exec;
  private final VisitorAwareProcessor<P> vp;

  private CommonSpringModelVisitorContext(Processor<? super P> p, Exec<P> e) {
    exec = e;
    vp = new VisitorAwareProcessor<>(this, p);
  }

  public interface Exec<P> {
    boolean run(@NotNull CommonSpringModel model, @NotNull Processor<? super P> params);
  }

  public boolean visit(@NotNull CommonSpringModel model) {
    if (hasBeenVisited(model)) return true;

    visited.add(model);
    return exec.run(model, vp);
  }

  public boolean hasBeenVisited(CommonSpringModel model) {
    return visited.contains(model);
  }

  private static class VisitorAwareProcessor<P> implements Processor<P> {
    private final CommonSpringModelVisitorContext<P> visitor;
    private final Processor<? super P> delegate;

    VisitorAwareProcessor(CommonSpringModelVisitorContext<P> visitor, Processor<? super P> delegate) {
      this.visitor = visitor;
      this.delegate = delegate;
    }

    @Override
    public boolean process(P p) {
      return delegate.process(p);
    }
  }

  public static <P> CommonSpringModelVisitorContext<P> context(@NotNull Processor<? super P> p, Exec<P> exec) {
      if (p instanceof VisitorAwareProcessor) {
        return ((VisitorAwareProcessor)p).visitor;
      }
      return new CommonSpringModelVisitorContext<>(p, exec);
  }
}
