// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.contexts.model.graph;

import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElement;
import com.intellij.psi.impl.FakePsiElement;
import com.intellij.psi.xml.XmlElement;
import com.intellij.util.IncorrectOperationException;
import com.intellij.util.xml.DomElement;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public final class LocalModelDependency {

  private final LocalModelDependencyType myType;
  private final PsiElement myIdentifyingElement;
  @NotNull
  private final String myLabel;

  public static LocalModelDependency create() {
    return new LocalModelDependency("", LocalModelDependencyType.UNKNOWN, null);
  }

  public static LocalModelDependency create(@NotNull LocalModelDependencyType type,
                                            @NotNull PsiElement identifyingElement) {
    String text = null;
    if (identifyingElement instanceof PsiClass) {
      text = ((PsiClass)identifyingElement).getQualifiedName();
    }
    else if (identifyingElement instanceof PsiAnnotation) {
      String qualifiedName = ((PsiAnnotation)identifyingElement).getQualifiedName();
      if (qualifiedName != null) {
        text = "@" + qualifiedName;
      }
    }
    else {
      text = identifyingElement.getText();
    }
    return new LocalModelDependency(text != null ? text : "", type, identifyingElement);
  }

  /**
   * Create a dependency defined by {@link DomElement} avoiding PSI access.
   */
  public static LocalModelDependency create(@NotNull String label,
                                            @NotNull LocalModelDependencyType type,
                                            DomElement identifyingElement) {
    return new LocalModelDependency(label, type, new FakePsiElement() {

      @NotNull
      private XmlElement getXmlElement() {
        final XmlElement xmlElement = identifyingElement.getXmlElement();
        assert xmlElement != null : identifyingElement;
        return xmlElement;
      }

      @Override
      public PsiElement getParent() {
        return getXmlElement();
      }

      // for diagram
      @Override
      public void delete() throws IncorrectOperationException {
        getXmlElement().delete();
      }

      @NotNull
      @Override
      public PsiElement getNavigationElement() {
        return getXmlElement();
      }
    });
  }

  public static LocalModelDependency create(@NotNull String label,
                                            @NotNull LocalModelDependencyType type,
                                            @Nullable PsiElement identifyingElement) {
    return new LocalModelDependency(label, type, identifyingElement);
  }

  private LocalModelDependency(@NotNull String label,
                               @NotNull LocalModelDependencyType type,
                               @Nullable PsiElement identifyingElement) {
    myLabel = label;
    myType = type;
    myIdentifyingElement = identifyingElement;
  }

  @NotNull
  public LocalModelDependencyType getType() {
    return myType;
  }

  @Nullable
  public PsiElement getIdentifyingElement() {
    return myIdentifyingElement;
  }

  @Override
  public String toString() {
    return "LocalModelDependency{" + myType + ", " + myIdentifyingElement + '}';
  }

  @NotNull
  public String getLabel() {
    return myLabel;
  }
}
