// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.contexts.model.graph;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.roots.ProjectRootManager;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.Pair;
import com.intellij.psi.util.CachedValue;
import com.intellij.psi.util.CachedValueProvider;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.psi.util.PsiModificationTracker;
import com.intellij.spring.SpringModificationTrackersManager;
import com.intellij.spring.contexts.model.LocalModel;
import com.intellij.spring.model.utils.SpringProfileUtils;
import com.intellij.util.containers.ConcurrentFactoryMap;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

public class LazyModelDependenciesGraph extends AbstractModelDependenciesGraph {
  private static final Key<CachedValue<Map<String, LazyModelDependenciesGraph>>> MODELS_GRAPH_KEY = Key.create("MODELS_GRAPH_KEY");

  public LazyModelDependenciesGraph(@NotNull Module module, @NotNull Set<String> profiles) {
    super(module, profiles);
  }

  @NotNull
  @Override
  public Collection<LocalModel<?>> getNodes() {
    throw new UnsupportedOperationException();
  }

  @NotNull
  @Override
  @SuppressWarnings("unchecked")
  protected Collection<Pair<LocalModel<?>, LocalModelDependency>> getDependencies(@NotNull LocalModel model) {
    return model.getDependentLocalModels();
  }

  @NotNull
  public static LazyModelDependenciesGraph getOrCreateLocalModelDependenciesGraph(@NotNull final Module module,
                                                                                  @NotNull final Set<String> activeProfiles) {
    String key = SpringProfileUtils.profilesAsString(activeProfiles);
    final Map<String, LazyModelDependenciesGraph> graphsMap = CachedValuesManager.getManager(module.getProject())
                                                                                 .getCachedValue(module, MODELS_GRAPH_KEY,
                                                                                                 createGraphProvider(module,
                                                                                                                     activeProfiles),
                                                                                                 false);

    return graphsMap.get(key);
  }

  @NotNull
  private static CachedValueProvider<Map<String, LazyModelDependenciesGraph>> createGraphProvider(final Module module,
                                                                                                  final Set<String> activeProfiles) {
    return () -> {
      Map<String, LazyModelDependenciesGraph> map =
        ConcurrentFactoryMap.createMap(key -> new LazyModelDependenciesGraph(module, activeProfiles));

      return CachedValueProvider.Result.create(map, getDependencies(module.getProject()));
    };
  }

  private static Object[] getDependencies(@NotNull Project project) {
    Set<Object> set = new LinkedHashSet<>();

    final SpringModificationTrackersManager springModificationTrackersManager = SpringModificationTrackersManager.getInstance(project);
    set.add(springModificationTrackersManager.getProfilesModificationTracker());
    set.add(PsiModificationTracker.MODIFICATION_COUNT);
    set.add(springModificationTrackersManager.getCustomBeanParserModificationTracker());
    set.add(ProjectRootManager.getInstance(project));

    return set.toArray();
  }
}
