// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.contexts.model.graph;

import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.Pair;
import com.intellij.openapi.util.UserDataHolderBase;
import com.intellij.psi.util.CachedValue;
import com.intellij.util.Function;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.graph.Graph;
import org.jetbrains.annotations.NotNull;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

public abstract class LazyDependenciesGraph<N, E> extends UserDataHolderBase implements Graph<N> {
  private static final Key<CachedValue<Map<String, LazyModelDependenciesGraph>>> OUTS_KEY = Key.create("OUTS_KEY");

  private final Map<N, Collection<Pair<N, E>>> myIns = new ConcurrentHashMap<>();
  private final Map<N, Collection<Pair<N, E>>> myOuts = new ConcurrentHashMap<>();

  private volatile boolean myIsBuilt = false;

  private final Function<Pair<N, E>, N> mySourceNodeFunction = pair -> pair.getFirst();

  protected abstract @NotNull Collection<Pair<N, E>> getDependencies(@NotNull N n);

  @Override
  public @NotNull Iterator<N> getIn(N n) {
    guaranteeGraphIsBuilt();
    Collection<Pair<N, E>> dependencies = myIns.get(n);
    return dependencies != null ? getIterator(dependencies): Collections.emptyIterator();
  }

  @Override
  public @NotNull Iterator<N> getOut(N n) {
    return getIterator(getOrCreateOutDependencies(n));
  }

  private @NotNull Iterator<N> getIterator(Collection<Pair<N, E>> dependencies) {
    return ContainerUtil.map(dependencies, mySourceNodeFunction).iterator();
  }

  protected final void guaranteeGraphIsBuilt() {
    if (!myIsBuilt) {
      final Collection<N> nodes = getNodes();
      for (N node : nodes) {
        getOrCreateOutDependencies(node);
      }
      myIsBuilt = true;
    }
  }

  public final @NotNull Collection<Pair<N, E>> getOrCreateOutDependencies(@NotNull N n) {
    Collection<Pair<N, E>> outSet = myOuts.get(n);
    if (outSet == null) {
      outSet = getDependencies(n);
      for (Pair<N, E> dependency : outSet) {
        addInDependency(n, dependency.first, dependency.second);
      }
      myOuts.put(n, outSet);
    }
    return myOuts.get(n);
  }

  public void addInDependency(@NotNull N from, @NotNull N to, @NotNull E edgeDescriptor) {
    Collection<Pair<N, E>> inNodes = myIns.get(to);
    if (inNodes == null) {
      inNodes = new LinkedHashSet<>();
    }
    inNodes.add(Pair.create(from, edgeDescriptor));
    myIns.put(to, inNodes);
  }
}
