// Copyright 2000-2021 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.contexts.model;

import com.intellij.openapi.util.Condition;
import com.intellij.util.containers.hash.EqualityPolicy;
import com.intellij.util.containers.hash.LinkedHashMap;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Map;

public abstract class SpringSizeLimitedCache<K, V> {
  private final Map<K, V> myQueue;
  private @NotNull final Condition<? super K> myKeyValidityCheck;
  private final int myMaxQueueSize;
  private final Object myLock = new Object();

  protected SpringSizeLimitedCache(final int maxQueueSize, @NotNull Condition<? super K> keyValidityCheck) {
    myQueue = new LinkedHashMap<>(10, 0.6f, (EqualityPolicy<? super K>)EqualityPolicy.CANONICAL, true) {
      @Override
      protected boolean removeEldestEntry(Map.Entry<K, V> eldest, K key, V value) {
        return (size() > myMaxQueueSize);
      }
    };
    myKeyValidityCheck = keyValidityCheck;
    myMaxQueueSize = maxQueueSize;
  }

  @Nullable
  protected V getCachedValue(K key) {
    V value;
    synchronized (myLock) {
      value = myQueue.get(key);
    }
    return value;
  }

  @NotNull
  protected abstract V createValue(K key);

  @NotNull
  public V get(K key) {
    V value = getCachedValue(key);
    if (value != null) return value;

    final V newValue = createValue(key);
    synchronized (myLock) {
      if (myQueue.size() > myMaxQueueSize && !allKeysAreValid()) {
        myQueue.clear();
      }
      myQueue.put(key, newValue);
    }

    return newValue;
  }

  private boolean allKeysAreValid() {
    for (K k : myQueue.keySet()) {
      if (!myKeyValidityCheck.value(k)) return false;
    }
    return true;
  }
}
